/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ----------------------------------------------------------------- */

#include <string.h>
#include <ald_i2c.h>
#include "main.h"
#include "wm8978.h"
#include "sample_wav.h"
#include "stdio.h"
#include "bsp_uart.h"
#include "wm_i2s.h"
#include "rd_buf.h"
#include "ff_gen_drv.h"
#include "micro_sd_diskio.h"
/* Private Macros ----------------------------------------------------------- */

#define U2C4(X) (X)&0xFF,(X>>8)&0xFF,(X>>16)&0xFF,(X>>24)&0xFF
#define WAVF_RIFF 0x46464952
#define WAVF_WAVE 0x45564157
#define WAVF_fmt  0x20746D66
#define WAVF_data 0x61746164
/* Private Variables--------------------------------------------------------- */

typedef struct
{
    uint32_t   ChunkID;       /* 0 */
    uint32_t   FileSize;      /* 4 */
    uint32_t   FileFormat;    /* 8 */

    uint32_t   SubChunk1ID;   /* 12 */
    uint32_t   SubChunk1Size; /* 16*/
    uint16_t   AudioFormat;   /* 20 */
    uint16_t   NbrChannels;   /* 22 */
    uint32_t   SampleRate;    /* 24 */

    uint32_t   ByteRate;      /* 28 */
    uint16_t   BlockAlign;    /* 32 */
    uint16_t   BitPerSample;  /* 34 */
    uint32_t   SubChunk2ID;   /* 36 */
    uint32_t   SubChunk2Size; /* 40 */

} wave_formattypedef_t;

/* Public Variables -------------------------------------------------------- */
static i2c_handle_t g_hi2c;
extern uart_handle_t g_h_uart;

/* Private Constants -------------------------------------------------------- */

/* Private function prototypes ---------------------------------------------- */
void print_wav_info(wave_formattypedef_t *wav);
void i2s0_pin_init(void);
void i2c0_init(void);
int sd_test(void);
void sd_init(void);
/* Private Function---------------------------------------------------------- */

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/* DMA使用的双缓冲区 4K bytes*/
int16_t audio_dma_buffer[2 * 1024] __attribute((aligned(4)));
/* 录音数据暂存区 64K bytes*/
int16_t audio_record_buffer[32 * 1024];
/* SD卡 */
char micro_sd_path[] = "0:/";
int sd_status = 0;
FATFS micro_sd_fatfs;
/* SD卡及缓存 */
uint8_t sd_data[1024 * 2 * 4 + sizeof(struct RD_buffer)];
struct RD_buffer *sd_buf = (struct RD_buffer *)sd_data;
FIL sd_music_file;
char sd_file_name[32];
int sd_music = 0;
int sd_empty_frames = 0;
/* 发送信息 */
const uint8_t *g_audio_data;
uint32_t g_audio_length;
uint32_t g_current_pos = 0;
uint32_t audio_play_channels;

/* 接收缓存 */
uint8_t *record_buffer;
uint32_t record_size;
uint32_t record_length = 0;
/* NOTE: 缓冲区的size 单位是int16而不是byte */

/* Demo控制用的数据 */
uint8_t audio_volume = 30;
volatile int record_full = 0;
uint32_t wm8978_status = 0;

/**
  * @brief  将缓冲区的数据发送到WM8978
  * @retval None
  */
uint16_t audio_send_buf(void *buf, uint16_t length)
{
    int16_t *dst = (int16_t *)buf;
    int16_t *src = (int16_t *)(rd_get(sd_buf));

    if (!src)
    {
        memset(dst, 0, length * 2);
        sd_empty_frames++;
        return length;
    }

    rd_pop(sd_buf);

    if (audio_play_channels == 1)
    {
        for (int i = 0; i < length / 2; i++)
        {
            dst[i * 2] = src[i];
            dst[i * 2 + 1] = src[i];
        }
    }
    else
    {
        memcpy(dst, src, length * 2);
    }

    g_current_pos += length;
    return length;
}

/**
  * @brief  接收数据并放到缓冲区，等待eventloop写入SD卡
  * @retval None
  */
uint16_t audio_recv_buf(void *buf, uint16_t length)
{
    int16_t *dst = (int16_t *)(rd_new(sd_buf));
    int16_t *src = (int16_t *)(buf);

    if (!dst)
    {
        sd_empty_frames++;
        return length;
    }

    if (audio_play_channels == 1)
    {
        for (int i = 0; i < length / 2; i++)
        {
            dst[i] = src[i * 2];
        }

        g_current_pos += length / 2;
    }
    else
    {
        memcpy(dst, src, length * 2);
        g_current_pos += length;
    }

    return length;
}

/**
  * @brief  从ROM中读取数据发送
  * @retval None
  */
uint16_t audio_send_data(void *buf, uint16_t length)
{
    uint32_t new_pos;

    if (g_current_pos == g_audio_length)
        g_current_pos = 0;

    new_pos = g_current_pos + length;

    if (new_pos > g_audio_length)
    {
        length = (g_audio_length - g_current_pos);
    }

    int16_t *dst = (int16_t *)buf;
    int16_t *src = (int16_t *)(g_audio_data + g_current_pos);

    if (audio_play_channels == 1)
    {
        for (int i = 0; i < length / 2; i++)
        {
            dst[i * 2] = src[i];
            dst[i * 2 + 1] = src[i];
        }
    }
    else
    {
        memcpy(dst, src, length * 2);
    }

    g_current_pos += length;
    return length;
}

/**
  * @brief  接收录音数据并放到内存
  * @retval None
  */
uint16_t audio_recv_data(void *buf, uint16_t length)
{
    uint32_t new_pos;
    uint16_t res = length;

    length = length / 2;
    new_pos = record_length + length;

    if (new_pos >= record_size)
    {
        wm_stop();
        length = (record_size - record_length);
        res = 0;
        record_full = 1;
    }

    int16_t *dst = (int16_t *)(record_buffer + record_length);
    int16_t *src = (int16_t *)buf;

    for (int i = 0; i < length; i++)
    {
        dst[i] = src[i * 2];
    }

    record_length += length;

    return res;
}

/**
  * @brief  从SD卡读取数据
  * @retval None
  */
void read_sd_data(uint32_t timeout)
{
    uint8_t *p;
    uint32_t readbytes;
    uint32_t tick = ald_get_tick();

    while ((p = rd_new(sd_buf)) != NULL)
    {
        if (f_read(&sd_music_file, p, sd_buf->block_size, &readbytes) != FR_OK)
        {
            printf("Read SD buffer failed\r\n");
        }

        if (readbytes < sd_buf->block_size)
        {
            printf("Audio play end\r\n");
            memset(p + readbytes, 0, sd_buf->block_size - readbytes);
            f_close(&sd_music_file);
            sd_music = 0;
            wm_stop();
            break;
        }

        if (ald_get_tick() - tick > timeout)
            break;
    }
}

/**
  * @brief  将缓存的数据写入SD卡
  * @retval None
  */
void write_sd_data(uint32_t timeout)
{
    uint8_t *p;
    uint32_t writebytes;
    uint32_t tick = ald_get_tick();

    while ((p = rd_get(sd_buf)) != NULL)
    {
        if (f_write(&sd_music_file, p, sd_buf->block_size, &writebytes) != FR_OK)
        {
            printf("Write SD buffer failed\r\n");
        }

        if (writebytes < sd_buf->block_size)
        {
            printf("Write failed %d writes (expect %d) \r\n", writebytes, sd_buf->block_size);
            break;
        }

        rd_pop(sd_buf);

        if (ald_get_tick() - tick > timeout)
            break;
    }
}

/**
  * @brief  检查WAV文件格式
  * @retval None
  */
int check_wav(wave_formattypedef_t *wav)
{
    uint32_t wav_sample_rate = 0;
    uint32_t wav_channels = 0;
    uint32_t wav_audio_fmt = 0;
    uint32_t wav_sample_bits = 0;

    wav_audio_fmt = wav->AudioFormat;
    wav_channels = wav->NbrChannels;
    wav_sample_bits = wav->BitPerSample;
    wav_sample_rate = wav->SampleRate;

    audio_play_channels = wav_channels;

    if (wav->ChunkID != WAVF_RIFF || wav->FileFormat != WAVF_WAVE
            || wav->SubChunk1ID != WAVF_fmt || wav->SubChunk2ID != WAVF_data)
        return -1;

    if (wav_sample_bits != 16)
        return -2;

    if (wav_channels != 1 && wav_channels != 2)
        return -3;

    if (wav_audio_fmt != 1)
        return -4;

    if (wm_set_sample_rate(wav_sample_rate))
        return -5;

    return 0;
}

/**
  * @brief  消息处理
  * @retval None
  */
void test_wm()
{
    g_audio_data = (g_sample_wav_data + 44);
    g_audio_length = g_sample_wav_data_size - 44;
    /* 初始化WM8978 */
    i2c0_init();

    if (WM8978_Init(&g_hi2c))
    {
        printf("Error WM8978 init\r\n");
        return ;
    }

    wm8978_status = 1;
    /* 初始化I2S */
    i2s0_pin_init();
    /* 音频初始化信息 */
    struct wm_i2s_info wm_info;
    uint8_t *g_audio_buffer;

    g_audio_buffer = (uint8_t *)audio_dma_buffer;
    wm_info.buffers[0] = g_audio_buffer;
    wm_info.buffers[1] = g_audio_buffer + 1024 * 2;
    wm_info.buffer_length = 1024;
    wm_info.spi_x = SPI0;
    wm_info.feed_data = audio_send_data;
    wm_info.sample_rate = 16000;
    wm_info.dma_tx_ch = 2;
    wm_info.dma_rx_ch = 3;

    if (wm_i2s_init(&wm_info))
    {
        printf("I2S init failed\r\n");
    }

    printf("Audio init ok\r\n");
    audio_play_channels = 1;
    record_buffer = (uint8_t *)audio_record_buffer;
    record_size = sizeof(audio_record_buffer) / 2;
    record_length = 0;
    printf("Record buffer (size:%d)\r\n", record_size);
}

/**
  * @brief  消息处理
  * @retval None
  */
void do_msg(char *msg)
{
    char *argv[8];
    int argc;
    char *p = NULL;

    /* 分割成argc argv的形式 */
    argv[0] = strtok_r(msg, " ", &p);
    argc = 1;

    for (int i = 1; i < 8; i++)
    {
        char *s = strtok_r(NULL, " ", &p);

        if (!s)
            break;

        argv[argc++] = s;
    }

    printf("[msg:%s]\r\n", msg);

    if (!strcmp(msg, "play"))
    {
        /* 播放 */
        if (argc <= 1)
        {
            /* 没参数，播放内置音频 */
            g_audio_data = g_sample_wav_data;
            g_audio_length = g_sample_wav_data_size;
            g_current_pos = 0;
            wm_set_callback(audio_send_data);
            wm_set_mode(WM_SEND);
            wm_play();
        }
        else
        {
            /* 有参数，尝试打开SD卡上的音频播放 */
            if (sd_status == 0)
            {
                printf("No SD card found!\r\n");
                return;
            }

            if (sd_music)
            {
                f_close(&sd_music_file);
            }

            if (f_open(&sd_music_file, argv[1], FA_READ) != FR_OK)
            {
                printf("Open '%s' failed\r\n", argv[1]);
                return;
            }

            strncpy(sd_file_name, argv[1], 32);

            /* 读取WAV信息 验证 */
            wave_formattypedef_t wav_head;
            uint32_t bytesread;
            f_read(&sd_music_file, &wav_head, sizeof(wave_formattypedef_t), (UINT *)&bytesread);

            if (bytesread != sizeof(wave_formattypedef_t))
            {
                f_close(&sd_music_file);
                printf("Read wav file failed\r\n");
                return;
            }

            print_wav_info(&wav_head);

            if (check_wav(&wav_head) != 0)
            {
                f_close(&sd_music_file);
                printf("Wav file error\r\n");
                return;
            }

            /* 准备播放音频 */
            if (audio_play_channels == 1)
            {
                rd_static_init(sd_data, sizeof(sd_data), 8);
            }
            else
            {
                rd_static_init(sd_data, sizeof(sd_data), 4);
            }

            read_sd_data(10000);
            printf("Read data ok, start playing\r\n");
            wm_set_callback(audio_send_buf);
            wm_set_mode(WM_SEND);
            sd_music = 1;
            g_current_pos = 0;
            wm_play();
        }
    }
    else if (!strcmp(msg, "pause"))
    {
        /* 暂停 */
        wm_pause();
    }
    else if (!strcmp(msg, "resume"))
    {
        /* 继续 */
        wm_resume();
    }
    else if (!strcmp(msg, "stop"))
    {
        /* 停止 */
        wm_stop();

        if (sd_music)
        {
            if (sd_music == 2)
            {
                /* 录音模式还需要保存WAV信息 */
                uint32_t writebytes;
                wave_formattypedef_t wav_head;
                wav_head.ChunkID = WAVF_RIFF;
                wav_head.FileSize = g_current_pos + 44;;
                wav_head.FileFormat = WAVF_WAVE;
                wav_head.SubChunk1ID = WAVF_fmt;
                wav_head.SubChunk1Size = 16;
                wav_head.AudioFormat = 1;
                wav_head.NbrChannels = 1;
                wav_head.SampleRate = 16000;
                wav_head.ByteRate = 16000;
                wav_head.BlockAlign = 2;
                wav_head.BitPerSample = 16;
                wav_head.SubChunk2ID = WAVF_data;
                wav_head.SubChunk2Size = g_current_pos;

                f_lseek(&sd_music_file, 0);
                f_write(&sd_music_file, &wav_head, sizeof(wave_formattypedef_t), &writebytes);

                if (writebytes != sizeof(wave_formattypedef_t))
                {
                    printf("Write WAV head failed\r\n");
                }
            }

            sd_music = 0;
            f_close(&sd_music_file);
        }
    }
    else if (!strcmp(msg, "record"))
    {
        /* 录音 */
        if (argc == 1)
        {
            /* 没参数，录音内容放到RAM缓冲区 */
            record_length = 0;
            wm_set_mode(WM_RECV);
            wm_set_callback(audio_recv_data);
            wm_record();
        }
        else if (sd_status == 1)
        {
            /* 录音内容存到SD卡 */
            if (sd_music)
            {
                f_close(&sd_music_file);
            }

            if (f_open(&sd_music_file, argv[1], FA_CREATE_NEW | FA_WRITE) != FR_OK)
            {
                printf("Open '%s' failed\r\n", argv[1]);
                return;
            }

            strncpy(sd_file_name, argv[1], 32);
            uint32_t byteswrite;
            wave_formattypedef_t wav_head;
            memset(&wav_head, 0, sizeof(wave_formattypedef_t));
            f_write(&sd_music_file, &wav_head, sizeof(wav_head), &byteswrite);

            if (byteswrite != sizeof(wav_head))
            {
                printf("Error write record file\r\n");
                f_close(&sd_music_file);
                return;
            }

            audio_play_channels = 1;
            g_current_pos = 0;
            sd_music = 2;
            rd_static_init(sd_data, sizeof(sd_data), 8);
            wm_set_mode(WM_RECV);
            wm_set_callback(audio_recv_buf);
            wm_record();
        }
    }
    else if (!strcmp(msg, "replay"))
    {
        /* 重放内存里的录音 */
        printf("Replay length= %d\r\n", record_length);

        g_audio_data = record_buffer;
        g_audio_length = record_length;
        g_current_pos = 0;
        wm_set_mode(WM_SEND);
        wm_set_callback(audio_send_data);
        wm_play();
    }
    else if (!strcmp(msg, "volh"))
    {
        /* 调高音量 */
        audio_volume = audio_volume >= 90 ? 100 : audio_volume + 10;
        wm_set_vol(0x7, audio_volume);
        printf("Volume = %d \r\n", audio_volume);
    }
    else if (!strcmp(msg, "voll"))
    {
        /* 调低音量 */
        audio_volume = audio_volume <= 10 ? 0 : audio_volume - 10;
        wm_set_vol(0x7, audio_volume);
        printf("Volume = %d \r\n", audio_volume);
    }
    else if (!strcmp(msg, "music"))
    {
        /* 列出 SD卡 MUSIC目录下的所有文件 */
        DIR dp;
        FILINFO f;

        if (f_findfirst(&dp, &f, "0:/MUSIC", "*.wav") != FR_OK)
        {
            printf("FATFS findfirst failed\r\n");
        }

        while (f.fname[0])
        {
            printf("%16s %u\r\n", f.fname, (uint32_t)f.fsize);
            f_findnext(&dp, &f);
        }

    }
    else if (!strcmp(msg, "info") || !strcmp(msg, "status"))
    {
        /* 输出当前信息 */
        printf("SD card    = %d\r\n", sd_status);
        printf("WM8978     = %d\r\n", wm8978_status);
        printf("I2S        = %d\r\n", wm_status());

        if (sd_music)
        {
            printf("SD file    = %s\r\n", sd_file_name);
        }

        printf("Current pos= %d\r\n", g_current_pos);
        printf("Record len = %d\r\n", record_length);
    }
    else
    {
        printf("Unknown command\r\n");
    }
}

/**
  * @brief 初始化SD卡
  * @retval None
**/
void sd_init(void)
{
    /* Regeister the NAND Flash disk I/O driver */
    if (ff_drv_register(0, &micro_sd_drv, (uint8_t *)micro_sd_path, sizeof(micro_sd_path), 0) != FR_OK)
    {
        printf("ff_drv_register failed\r\n");
        return;
    }

    /* Register the file system object to the FatFs module */
    if (f_mount(&micro_sd_fatfs, (const TCHAR *)micro_sd_path, 0) != FR_OK)
    {
        printf("f_mount failed\r\n");
        /* FatFs Initialization Error */
        return ;
    }

    int retry = 1;

    while (retry--)
    {
        ald_delay_ms(100);
        uint32_t sd_frees;
        FATFS *fs;

        /* check if we can read sd card */
        if (f_getfree("0:/", (DWORD *)&sd_frees, &fs))
        {
            printf("f_getfree error\r\n");
            printf("Retry time = %d\r\n", retry);
            continue;
        }

        printf("SD csize = %u \r\n", fs->csize);
        printf("SD total = %u clusters\r\n", (uint32_t)(fs->n_fatent - 2));
        printf("SD free  = %u clusters\r\n", sd_frees);
        break;
    }

    printf("SD init ok\r\n");
    sd_status = 1;
}

/**
  * @brief 事件循环，处理上位机消息，读取/写入SD卡
  * @retval None
*/
void event_loop()
{
    char recv_buffer[65];
    uint32_t recv_ptr = 0;

    while (1)
    {
        recv_ptr = bsp_uart_recved(recv_buffer, 64);

        if (recv_ptr)
        {
            recv_buffer[recv_ptr] = 0;

            for (int i = recv_ptr - 1; i >= 0; i--)
            {
                switch (recv_buffer[i])
                {
                    case ' ':
                    case '\n':
                    case '\r':
                    case '\t':
                        recv_buffer[i] = 0;
                        break;

                    default:
                        i = -1;
                }
            }

            do_msg(recv_buffer);
        }

        if (sd_music == 1)
        {
            read_sd_data(20);
        }

        if (sd_music == 2)
        {
            write_sd_data(20);
        }

        if (sd_empty_frames)
        {
            printf("sd_empty_frames:%d\r\n", sd_empty_frames);
            sd_empty_frames = 0;
        }

        if (record_full)
        {
            printf("Record buffer full, stopped\r\n");
            record_full = 0;
        }
    }
}

/**
  * @brief 输出WAV信息
  * @retval None
*/
void print_wav_info(wave_formattypedef_t *wav)
{
    printf("Chunk ID     = %d,%c%c%c%c\r\n", wav->ChunkID, U2C4(wav->ChunkID));
    printf("FileSize     = %d\r\n", wav->FileSize);
    printf("FileFormat   = %d,%c%c%c%c\r\n", wav->FileFormat, U2C4(wav->FileFormat));
    printf("SubChunk1ID  = %d,%c%c%c%c\r\n", wav->SubChunk1ID, U2C4(wav->SubChunk1ID));
    printf("SubChunk1Size= %d\r\n", wav->SubChunk1Size);
    printf("AudioFormat  = %d\r\n", wav->AudioFormat);
    printf("NbrChannels  = %d\r\n", wav->NbrChannels);
    printf("SampleRate   = %d\r\n", wav->SampleRate);
    printf("ByteRate     = %d\r\n", wav->SampleRate);
    printf("BlockAlign   = %d\r\n", wav->BlockAlign);
    printf("BitPerSample = %d\r\n", wav->BitPerSample);
    printf("SubChunk2ID  = %d,%c%c%c%c\r\n", wav->SubChunk2ID, U2C4(wav->SubChunk2ID));
    printf("SubChunk2Size= %d\r\n", wav->SubChunk2Size);
}

/**
  * @brief  初始化I2S0 管脚
  * @retval None
*/
void i2s0_pin_init(void)
{
    gpio_init_t x;

    /* Initialize NSS/WS pin */
    x.mode  = GPIO_MODE_OUTPUT;
    x.odos  = GPIO_PUSH_PULL;
    x.pupd  = GPIO_PUSH_DOWN;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_TTL;
    x.func  = GPIO_FUNC_6;
    ald_gpio_init(GPIOA, GPIO_PIN_4, &x);

    /* Initialize SCK/CK pin */
    x.mode  = GPIO_MODE_OUTPUT;
    x.odos  = GPIO_PUSH_PULL;
    x.pupd  = GPIO_PUSH_DOWN;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_TTL;
    x.func  = GPIO_FUNC_4;
    ald_gpio_init(GPIOD, GPIO_PIN_3, &x);

    /* Initialize MISO/MCLK pin */
    x.mode  = GPIO_MODE_OUTPUT;
    x.odos  = GPIO_PUSH_PULL;
    x.pupd  = GPIO_PUSH_DOWN;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_TTL;
    x.func  = GPIO_FUNC_4;
    ald_gpio_init(GPIOB, GPIO_PIN_4, &x);

    /* Initialize MOSI/SD pin */
    x.mode  = GPIO_MODE_OUTPUT;
    x.odos  = GPIO_PUSH_PULL;
    x.pupd  = GPIO_PUSH_DOWN;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_TTL;
    x.func  = GPIO_FUNC_4;
    ald_gpio_init(GPIOB, GPIO_PIN_5, &x);

    return;
}

/**
  * @brief  初始化I2C0
  * @retval None
*/
void i2c0_init()
{
    gpio_init_t x;

    /* Initialize SCK pin */
    x.mode  = GPIO_MODE_OUTPUT;
    x.odos  = GPIO_OPEN_DRAIN;
    x.pupd  = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_CMOS;
    x.func  = GPIO_FUNC_5;
    ald_gpio_init(GPIOB, GPIO_PIN_6, &x);

    /* Initialize SDA pin */
    x.mode  = GPIO_MODE_OUTPUT;
    x.odos  = GPIO_OPEN_DRAIN;
    x.pupd  = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_CMOS;
    x.func  = GPIO_FUNC_5;
    ald_gpio_init(GPIOB, GPIO_PIN_7, &x);

    memset(&g_hi2c, 0, sizeof(g_hi2c));
    g_hi2c.perh = I2C0;
    g_hi2c.init.module  = I2C_MODULE_MASTER;
    g_hi2c.init.addr_mode    = I2C_ADDR_7BIT;
    g_hi2c.init.clk_speed    = 100000;
    g_hi2c.init.dual_addr    = I2C_DUALADDR_ENABLE;
    g_hi2c.init.general_call = I2C_GENERALCALL_ENABLE;
    g_hi2c.init.no_stretch   = I2C_NOSTRETCH_DISABLE;
    g_hi2c.init.own_addr1    = 0xA0;
    ald_i2c_init(&g_hi2c);
}

/**
  * @brief  测试SD卡
  * @retval Status
*/
int sd_test(void)
{
    FRESULT res;
    uint32_t byteswritten, bytesread;                     /* File write/read counts */
    FIL     test_file[1];
    char work[FF_MAX_SS];
    const char wtext[] = "Hello world!\r\n";

    /* Register the file system object to the FatFs module */
    if (f_mount(&micro_sd_fatfs, (const TCHAR *)micro_sd_path, 0) != FR_OK)
    {
        printf("f_mount failed\r\n");
        /* FatFs Initialization Error */
        return -1;
    }

    printf("f_mount ok\r\n");

    /* Create FAT volume on the logical drive 0
    if (f_mkfs((const TCHAR *)micro_sd_path, FM_ANY, 0, work, sizeof(work)))
    {
        printf("f_mkfs failed\r\n");
        return -2;
    }
    printf("f_mkfs ok\r\n");*/

    /* Create and Open a new text file object with write access */
    if (f_open(test_file, "0:/test.txt", FA_CREATE_ALWAYS | FA_WRITE) != FR_OK)
    {
        printf("f_open failed\r\n");
        /* 'ESSEMI.TXT' file Open for write Error */
        return -3;
    }

    printf("f_open ok\r\n");

    /* Write data to the text file */
    res = f_write(test_file, (const void *)wtext, sizeof(wtext), &byteswritten);

    if ((byteswritten == 0) || (res != FR_OK))
    {
        printf("f_write failed\r\n");
        /* 'ESSEMI.TXT' file Write or EOF Error */
        return -4;
    }

    printf("f_write ok\r\n");

    /* Close the open text file */
    if (f_close(test_file) != FR_OK)
    {
        printf("f_close failed\r\n");
        return -5;
    }

    printf("f_close ok\r\n");

    if (f_open(test_file, "0:/test.txt", FA_READ) != FR_OK)
    {
        printf("f_open failed\r\n");
        /* 'ESSEMI.TXT' file Open for read Error */
    }

    printf("f_open ok\r\n");

    /* Read data from the text file */
    res = f_read(test_file, work, sizeof(wtext), (UINT *)&bytesread);

    if ((bytesread == 0) || (res != FR_OK)) /* EOF or Error */
    {
        printf("f_read failed\r\n");
        /* 'ESSEMI.TXT' file Read or EOF Error */
        return -6;
    }

    printf("f_read ok\r\n");

    /* Close the open text file */
    f_close(test_file);

    /* Compare read data with the expected data */
    if ((bytesread != byteswritten) || (memcmp(wtext, work, sizeof(wtext))))
    {
        printf("compare failed\r\n");
        /* Read data is different from the expected data */
        return -7;
    }

    printf("compare ok\r\n");

    printf("Write speed test\r\n");
    uint32_t tick = 0;

    if (f_open(test_file, "0:/test.txt", FA_CREATE_ALWAYS | FA_WRITE) != FR_OK)
    {
        printf("Speed test file open error\r\n");
    }

    for (int i = 0; i < FF_MAX_SS; i++)
        work[i] = 'U';

    tick = ald_get_tick();

    for (int i = 0; i < 128; i++)
    {
        if (f_write(test_file, work, FF_MAX_SS, &byteswritten) != FR_OK)
            printf("Write error: write\r\n");

        if (byteswritten != FF_MAX_SS)
            printf("Write error: byteswritten\r\n");
    }

    tick = ald_get_tick() - tick;
    f_close(test_file);
    printf("128 cycles write in %d ms,%d bytes/s\r\n", tick, 1000 * 128 * FF_MAX_SS / tick);
    return 0;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Initialize ALD */
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_72M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
    init_uart();
    printf("Startup...\r\n");
    ald_delay_ms(100);
    sd_init();
    sd_test();
    print_wav_info((wave_formattypedef_t *)g_sample_wav_data);
    test_wm();
    ald_delay_ms(100);

    event_loop();
}
/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
