/**********************************************************************************
 *
 * @file    gen_dev.c
 * @brief   FatFs generic low level device driver layer.
 * @date    12 Nov 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          12 Nov 2021     biyq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "ff_gen_drv.h"

#ifdef __cplusplus
extern "C" {
#endif

/* Private Macros ------------------------------------------------------------ */

/* Private Types ------------------------------------------------------------ */

/* Public Variables ---------------------------------------------------------- */
gen_ldrv_t  h_ldrv[FF_VOLUMES] = {0};
gen_pdrv_t  pdrv_info[FF_VOLUMES] = {0};

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Variables--------------------------------------------------------- */
/* record the total logical drive numver */
uint8_t     ld_num = 0;


#if FF_MULTI_PARTITION
/* Private functions ---------------------------------------------------------*/
/**
  * @brief  Get the number of partitions in sepecified physical drive in the volume management table.
  * @param  pdrv_num:   the physical drive.
  * @retval Returns the number of logical volume in specified physical drive in the volume management table.
  */
uint8_t ff_get_volume_num(uint8_t pdrv_num)
{
    uint8_t i, drv_num;

    for (i = 0; i < FF_VOLUMES; i++)
    {
        if (VolToPart[i].pd == pdrv_num)
            drv_num++;
    }

    return drv_num;
}
#endif

/**
  * @brief  Create logic drive path and link the created logic drive to the general device
  *         driver, and then increments the number of active volumes.
  * @note   The number of linked drivers (logic volumes) is less than or equal to FF_VOLUMES.
  * @param  pdrv_num:   the physical drive.
  * @param  pdrv:       pointer to the driver structure of physical drive.
  * @param  ldrv_path:  pointer to the logical drive path.
  * @param  ldpath_len    length of logical drive path, the value should be larger than or equal to 3.
  * @param  lun:        only used for USB Device to add multi-lun management else the parameter must
  *         be equal to 0
  * @retval Returns 0 if success, return 1 if excceed the limitation of logical drive number,
  *         return 2 if physical disk is full, return 3 if logical drive path is too short.
  */
DSTATUS ff_drv_register(const uint8_t pdrv_num, const gen_pdrv_io_t *pdrv, uint8_t *ldrv_path, uint8_t ldpath_len, uint8_t lun)
{
    /* criticize if logical drive number has been reach the limitation */
    if (ld_num >= FF_VOLUMES)
    {
        return 1;
    }

#if FF_MULTI_PARTITION

    /* criticize if current physical disk is full */
    if (pdrv_info[pdrv_num].vol_num >= ff_get_volume_num(pdrv_num))
    {
        return 2;
    }
    else
    {
        pdrv_info[pdrv_num].vol_num++;
    }

#endif
    ld_num++;

    /* record the volum information */
    h_ldrv[ld_num].pdrv_num = pdrv_num;
    h_ldrv[ld_num].lun = lun;

    /* link driver and add the number of active volumes */
    if (pdrv_info[pdrv_num].drv_io == (void *)0 || pdrv_info[pdrv_num].is_init == 0)
        pdrv_info[pdrv_num].drv_io = (gen_pdrv_io_t *)pdrv;

    /* create the logical drive path */
#if FF_STR_VOLUME_ID

#else

    if (ldpath_len < 3)
        return 3;

    ldrv_path[0] = ld_num - 1 + '0';
    ldrv_path[1] = ':';
    ldrv_path[2] = '/';
#endif

    return 0;
}

/**
  * @brief  Remove link for the logic drive and decrements the number of active volumes.
  * @note   The number of linked drivers (logic volumes) is more than or equal to 0.
  * @param  ldrv_path: pointer to the logical drive path.
  * @param  ldpath_len    length of logical drive path, should larger than or equal to 3.
  * @param  lun : only used for USB Device to add multi-lun management else the parameter
  *         must be equal to 0
  * @retval Returns 0 if success, return 1 if excceed the limitation of logical drive number,
  *         return 2 if no logic drive in current physical disk.
  */
DSTATUS ff_drv_unregister(const uint8_t pdrv_num, uint8_t *ldrv_path, uint8_t ldpath_len, uint8_t lun)
{
    uint8_t i;

    /* criticize if logical drive number has been reach the limitation */
    if (ld_num < 1)
    {
        return 1;
    }

#if FF_MULTI_PARTITION

    /* criticize if there's no logic drive in current physical disk */
    if (pdrv_info[pdrv_num].vol_num < 1)
    {
        return 2;
    }
    else
    {
        pdrv_info[pdrv_num].vol_num--;
    }

#endif
    ld_num--;

    /* restore the volum information to default */
    h_ldrv[ld_num].pdrv_num = 0;
    h_ldrv[ld_num].lun = 0;

    /* unlink driver and decrease the number of active volumes */
    pdrv_info[pdrv_num].drv_io = (void *)0;

#if FF_STR_VOLUME_ID

#else

    for (i = 0; i < ldpath_len; i++)
        ldrv_path[i] = 0;

#endif

#if FF_MULTI_PARTITION

    if (pdrv_info[pdrv_num].vol_num < 1)
    {
        /* free disk */
        pdrv_info[pdrv_num].drv_io = (void *)0;
        /* mark the current physical drive as uninitialized */
        pdrv_info[pdrv_num].is_init = 0;
    }

#else
    /* free disk */
    pdrv_info[pdrv_num].drv_io = (void *)0;
    /* mark the current physical drive as uninitialized */
    pdrv_info[pdrv_num].is_init = 0;
#endif
    return 0;
}

/**
  * @brief  get number of physical drive and logical drive.
  * @note   None.
  * @param  pdrv_num: pointer to the physical drive number.
  * @param  ldrv_num: pointer to the logical drive number.0
  * @retval None.
  */
void ff_drv_info(uint8_t *pdnum, uint8_t *ldnum)
{
#if FF_MULTI_PARTITION
    uint8_t i, cnt, flag, sum;

    cnt = sizeof(VolToPart) / sizeof(PARTITION);
    flag = VolToPart[0].pd;
    sum = 1;

    for (i = 0; i < cnt; i++)
    {
        if (VolToPart[i].pd != flag)
            sum++;
    }

    *pdnum = sum;
#else
    *pdnum = ld_num;
#endif

    *ldnum = ld_num;

    return;
}

#ifdef __cplusplus
}
#endif

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
