/*-----------------------------------------------------------------------*/
/* Low level disk I/O module skeleton for FatFs     (C)ChaN, 2016        */
/*-----------------------------------------------------------------------*/
/* If a working storage control module is available, it should be        */
/* attached to the FatFs via a glue function rather than modifying it.   */
/* This is an example of glue functions to attach various exsisting      */
/* storage control modules to the FatFs module with a defined API.       */
/*-----------------------------------------------------------------------*/

#include "diskio.h"     /* FatFs lower layer API */
#include "ff_gen_drv.h"

/* Definitions of physical drive number for each drive */
#define DEV_RAM         0   /* Example: Map Ramdisk to physical drive 0 */
#define DEV_NAND        1   /* Example: Map NAND Flash to physical drive 1 */
#define DEV_SRAM        2   /* Example: Map SRAM to physical drive 2 */
#define DEV_MMC         3   /* Example: Map MMC/SD card to physical drive 3 */
#define DEV_USB         4   /* Example: Map USB MSD to physical drive 4 */
#define DEV_USB_HOST    5

extern gen_pdrv_t pdrv_info[FF_VOLUMES];

/*-----------------------------------------------------------------------*/
/* Get Drive Status                                                      */
/*-----------------------------------------------------------------------*/
DSTATUS disk_status(
    BYTE pdrv       /* Physical drive nmuber to identify the drive */
)
{
    if (pdrv_info[pdrv].vol_num == 0)
        return STA_NODISK;

    if (pdrv_info[pdrv].is_init == 1)
        return STA_READY;
    else
        return STA_NOINIT;
}

/*-----------------------------------------------------------------------*/
/* Inidialize a Drive                                                    */
/*-----------------------------------------------------------------------*/
DSTATUS disk_initialize(
    BYTE pdrv               /* Physical drive nmuber to identify the drive */
)
{
    if (disk_status(pdrv) == STA_READY)
        return RES_OK;

    if (pdrv_info[pdrv].drv_io->pdrv_init())
        return RES_ERROR;

    /* mark the current physical drive as initialized */
    pdrv_info[pdrv].is_init = 1;

    return RES_OK;
}



/*-----------------------------------------------------------------------*/
/* Read Sector(s)                                                        */
/*-----------------------------------------------------------------------*/

DRESULT disk_read(
    BYTE pdrv,      /* Physical drive nmuber to identify the drive */
    BYTE *buff,     /* Data buffer to store read data */
    DWORD sector,   /* Start sector in LBA */
    UINT count      /* Number of sectors to read */
)
{
    if (pdrv_info[pdrv].drv_io->pdrv_read(buff, sector, count))
        return RES_PARERR;

    return RES_OK;
}



/*-----------------------------------------------------------------------*/
/* Write Sector(s)                                                       */
/*-----------------------------------------------------------------------*/
DRESULT disk_write(
    BYTE pdrv,          /* Physical drive nmuber to identify the drive */
    const BYTE *buff,   /* Data to be written */
    DWORD sector,       /* Start sector in LBA */
    UINT count          /* Number of sectors to write */
)
{
    if (pdrv_info[pdrv].drv_io->pdrv_write(buff, sector, count))
        return RES_PARERR;

    return RES_OK;
}

/*-----------------------------------------------------------------------*/
/* Miscellaneous Functions                                               */
/*-----------------------------------------------------------------------*/

DRESULT disk_ioctl(
    BYTE pdrv,      /* Physical drive nmuber (0..) */
    BYTE cmd,       /* Control code */
    void *buff      /* Buffer to send/receive control data */
)
{
    if (pdrv_info[pdrv].is_init == 0)
        return RES_PARERR;

    if (pdrv_info[pdrv].drv_io->pdrv_ioctl(cmd, buff))
        return RES_PARERR;

    return RES_OK;
}

/**
  * @brief  Gets Time from RTC
  * @param  None
  * @retval Time in DWORD
  */
__weak DWORD get_fattime(void)
{
    return 0;
}
