/**********************************************************************************
 *
 * @file    wm_i2s.c
 * @brief   I2S driver for WM8978
 *
 * @date    31 Dec 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          31 Dec 2021     shiwa           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ----------------------------------------------------------------- */
#include <ald_conf.h>
#include <ald_spi.h>
#include <ald_i2s.h>
#include <ald_gpio.h>
#include "wm8978.h"
#include "wm_i2s.h"
#include <stdio.h>

/* Private Macros ----------------------------------------------------------- */
#define LOG_E printf
#define LOG_I printf
#define DMA_MSEL_SPIx (SPIx==SPI0?DMA_MSEL_SPI0:DMA_MSEL_SPI1)
/* Private Variables--------------------------------------------------------- */

/* Public Variables -------------------------------------------------------- */
static i2s_handle_t g_h_i2s;
static uint16_t spi_t_FFFF = 0xFFFF;
static void *i2s_buffers[2];
static uint16_t buffer_length ;
static func_I2S_feed_data feed_data;
static uint32_t i2s_send_dma_ch, i2s_recv_dma_ch;
static uint32_t status = 0;
extern dma_descriptor_t dma0_ctrl_base[28];
/* Private Constants -------------------------------------------------------- */

/* Private function prototypes ---------------------------------------------- */
void i2s_dma_send_config(void);
void i2s_dma_recv_config(void);
void dma_i2s_pingpong_config_struct(uint8_t channel);
/* Private Function---------------------------------------------------------- */
void wm_set_vol(uint32_t type, uint8_t vol)
{
    if (status)
    {
        I2S_DISABLE(&g_h_i2s);
    }

    if (type & 0x1)
    {
        WM8978_HPvol_Set(vol, vol);   //
    }

    if (type & 0x2)
    {
        WM8978_SPKvol_Set(vol);      //
    }

    if (type & 0x4)
    {
        WM8978_MIC_Gain(vol);
    }

    if (status)
    {
        I2S_ENABLE(&g_h_i2s);
    }
}
void wm_set_mode(uint32_t mode)
{
    if (mode == WM_SEND)
    {
        WM8978_ADDA_Cfg(1, 0);                              /*Open the DAC*/
        WM8978_Input_Cfg(0, 0, 0);                          /*Close the input channel*/
        WM8978_Output_Cfg(1, 0);                            /*Open the DAC output*/

    }
    else
    {
        WM8978_ADDA_Cfg(0, 1);
        WM8978_Input_Cfg(1, 1, 0);
        WM8978_Output_Cfg(0, 1);
    }
}
void wm_set_callback(func_I2S_feed_data f)
{
    feed_data = f;
}
void wm_play(void)
{
    i2s_dma_send_config();
    status = 1;
}
void wm_record(void)
{
    i2s_dma_recv_config();
    status = 2;
}
void wm_stop(void)
{
    ald_i2s_dma_stop(&g_h_i2s);
    I2S_DISABLE(&g_h_i2s);
    status = 0;
}
void wm_pause(void)
{
    if (status)
    {
        I2S_DISABLE(&g_h_i2s);
        ald_i2s_dma_pause(&g_h_i2s);
    }
}
void wm_resume(void)
{
    if (status)
    {
        I2S_ENABLE(&g_h_i2s);
        ald_i2s_dma_resume(&g_h_i2s);
    }
}
uint32_t wm_status()
{
    return status;
}

int wm_set_sample_rate(uint32_t rate)
{
    if (rate != g_h_i2s.init.sampling)
    {
        g_h_i2s.init.sampling = rate;

        if (ald_i2s_init(&g_h_i2s))
        {
            return 1;
        }
    }

    return 0;
}
int wm_i2s_init(struct wm_i2s_info *wm_info)
{
    g_h_i2s.perh = wm_info->spi_x;
    g_h_i2s.init.standard = I2S_STD_PHI;
    g_h_i2s.init.data_len = I2S_LEN_16;
    g_h_i2s.init.ch_len = I2S_WIDE_16;
    g_h_i2s.init.mck_en = ENABLE;
    g_h_i2s.init.sampling = wm_info->sample_rate;
    g_h_i2s.init.polarity = I2S_INACTIVE_LOW;

    if (ald_i2s_init(&g_h_i2s))
    {
        LOG_E("I2S init failed\r\n");
        return 1;
    }

    i2s_buffers[0] = wm_info->buffers[0];
    i2s_buffers[1] = wm_info->buffers[1];
    buffer_length = wm_info->buffer_length > 1024 ? 1024 : wm_info->buffer_length;
    feed_data = wm_info->feed_data;

    //TODO : Specify I2S DMA channels
    i2s_send_dma_ch = wm_info->dma_tx_ch;
    i2s_recv_dma_ch = wm_info->dma_rx_ch;
    return 0;
}

void i2s_dma_send_callback(void *arg)
{
    DMA0->CHENSET = 1 << i2s_send_dma_ch;
    dma_i2s_pingpong_config_struct(i2s_send_dma_ch);
}
void i2s_dma_send_config()
{
    dma_config_t dma_config;
    SPI_I2S_TypeDef *SPIx = g_h_i2s.perh;

    ald_dma_config_struct(&dma_config);
    ald_i2s_dma_stop(&g_h_i2s);

    /* Disable I2S */
    SPIx->I2SCFG &= ~(1 << SPI_I2SCFG_I2SE_POS);
    //SPIx->I2SCFG &= ~(1 << SPI_I2SCFG_I2SE_POS);
    /* Send mode */
    SPIx->I2SCFG &= ~(3 << SPI_I2SCFG_I2SCFG_POSS);
    SPIx->I2SCFG |= (2 << SPI_I2SCFG_I2SCFG_POSS);
    /* Output MCK */
    SPIx->I2SPR |= 1U << SPI_I2SPR_MCKOE_POS;

    dma_config.channel = i2s_send_dma_ch;
    dma_config.primary = DISABLE;
    dma_config.R_power = DMA_R_POWER_1;
    dma_config.data_width = DMA_DATA_SIZE_HALFWORD;
    dma_config.dst     = (void *)(&SPIx->DATA);
    dma_config.dst_inc = DMA_DATA_INC_NONE;
    dma_config.src     = i2s_buffers[1];
    dma_config.src_inc = DMA_DATA_INC_HALFWORD;
    dma_config.size    = buffer_length;
    dma_config.msel    = DMA_MSEL_SPIx;
    dma_config.msigsel = DMA_MSIGSEL_SPI_TXEMPTY;
    dma_config.burst   = ENABLE;
    dma_config.interrupt = ENABLE;
    ald_dma_config_ping_pong(DMA0, &dma_config, 0, i2s_dma_send_callback);

    dma_config.channel = i2s_send_dma_ch;
    dma_config.primary = ENABLE;
    dma_config.R_power = DMA_R_POWER_1;
    dma_config.data_width = DMA_DATA_SIZE_HALFWORD;
    dma_config.dst     = (void *)(&SPIx->DATA);
    dma_config.dst_inc = DMA_DATA_INC_NONE;
    dma_config.src     = i2s_buffers[0];
    dma_config.src_inc = DMA_DATA_INC_HALFWORD;
    dma_config.size    = buffer_length;
    dma_config.msel    = DMA_MSEL_SPIx;
    dma_config.msigsel = DMA_MSIGSEL_SPI_TXEMPTY;
    dma_config.burst   = ENABLE;
    dma_config.interrupt = ENABLE;
    ald_dma_config_ping_pong(DMA0, &dma_config, 1, i2s_dma_send_callback);

    feed_data(i2s_buffers[0], buffer_length);
    feed_data(i2s_buffers[1], buffer_length);
    ald_i2s_dma_req_config(&g_h_i2s, I2S_DMA_REQ_TX, ENABLE);
    /* Enable I2S */
    SPIx->I2SCFG |= (1 << SPI_I2SCFG_I2SE_POS);
    /* Enable DMA */
    DMA0->CHENSET |= (1 << i2s_send_dma_ch);
}

void dma_i2s_pingpong_config_struct(uint8_t channel)
{
    dma_descriptor_t *desc;
    void *sel_buf;
    uint16_t length;
    desc = (dma_descriptor_t *)(DMA0->CTRLBASE) + channel;
    sel_buf = i2s_buffers[0];

    if (desc->ctrl.cycle_ctrl != 0)
    {
        desc = (dma_descriptor_t *)(DMA0->ALTCTRLBASE) + channel;
        sel_buf = i2s_buffers[1];
    }

    length = feed_data(sel_buf, buffer_length);

    if (length > 0)
    {
        desc->ctrl.cycle_ctrl = DMA_CYCLE_CTRL_PINGPONG;
        desc->ctrl.src_inc = DMA_DATA_INC_HALFWORD;
        desc->src = (void *)((uint32_t)sel_buf + ((length - 1) << desc->ctrl.src_size));
        desc->ctrl.n_minus_1 = length - 1;
    }
}
void dma_i2s_clk_pingpong_config_struct(uint8_t channel)
{
    dma_descriptor_t *desc;

    desc = (dma_descriptor_t *)(DMA0->CTRLBASE) + channel;

    if (desc->ctrl.cycle_ctrl != 0)
    {
        desc = (dma_descriptor_t *)(DMA0->ALTCTRLBASE) + channel;
    }

    desc->ctrl.cycle_ctrl = DMA_CYCLE_CTRL_PINGPONG;
    desc->ctrl.n_minus_1 = 1024 - 1;
}
void dma_i2s_recv_pingpong_config_struct(uint8_t channel)
{
    dma_descriptor_t *desc;
    void *sel_buf;
    uint16_t length;
    desc = (dma_descriptor_t *)(DMA0->CTRLBASE) + channel;
    sel_buf = i2s_buffers[0];

    if (desc->ctrl.cycle_ctrl != 0)
    {
        desc = (dma_descriptor_t *)(DMA0->ALTCTRLBASE) + channel;
        sel_buf = i2s_buffers[1];
    }

    length = feed_data(sel_buf, buffer_length);

    if (length > 0)
    {
        desc->ctrl.cycle_ctrl = DMA_CYCLE_CTRL_PINGPONG;

        desc->dst = (void *)((uint32_t)sel_buf + ((length - 1) << desc->ctrl.src_size));
        desc->ctrl.n_minus_1 = length - 1;
    }
}
void i2s_dma_recv_clk_callback(void *arg)
{
    DMA0->CHENSET = 1 << i2s_send_dma_ch;
    dma_i2s_clk_pingpong_config_struct(i2s_send_dma_ch);
}
void i2s_dma_recv_callback(void *arg)
{
    DMA0->CHENSET = 1 << i2s_recv_dma_ch;
    dma_i2s_recv_pingpong_config_struct(i2s_recv_dma_ch);
}


void i2s_dma_recv_config()
{
    dma_config_t dma_config;
    SPI_I2S_TypeDef *SPIx = g_h_i2s.perh;
    ald_i2s_dma_stop(&g_h_i2s);
    ald_dma_config_struct(&dma_config);

    /* Disable I2S */
    SPIx->I2SCFG &= ~(1 << SPI_I2SCFG_I2SE_POS);
    /* Recv mode */
    SPIx->I2SCFG |= (3 << SPI_I2SCFG_I2SCFG_POSS);

    /* Output MCK */
    SPIx->I2SPR |= 1U << SPI_I2SPR_MCKOE_POS;

    dma_config.channel = i2s_send_dma_ch;
    dma_config.primary = DISABLE;
    dma_config.R_power = DMA_R_POWER_1;
    dma_config.data_width = DMA_DATA_SIZE_HALFWORD;
    dma_config.dst     = (void *)(&SPIx->DATA);
    dma_config.dst_inc = DMA_DATA_INC_NONE;
    dma_config.src     = (void *)(&spi_t_FFFF);
    dma_config.src_inc = DMA_DATA_INC_NONE;
    dma_config.size    = buffer_length;
    dma_config.msel    = DMA_MSEL_SPIx;
    dma_config.msigsel = DMA_MSIGSEL_SPI_TXEMPTY;
    dma_config.burst   = ENABLE;
    dma_config.interrupt = ENABLE;
    ald_dma_config_ping_pong(DMA0, &dma_config, 0, i2s_dma_recv_clk_callback);

    dma_config.channel = i2s_send_dma_ch;
    dma_config.primary = ENABLE;
    dma_config.R_power = DMA_R_POWER_1;
    dma_config.data_width = DMA_DATA_SIZE_HALFWORD;
    dma_config.dst     = (void *)(&SPIx->DATA);
    dma_config.dst_inc = DMA_DATA_INC_NONE;
    dma_config.src     = (void *)(&spi_t_FFFF);
    dma_config.src_inc = DMA_DATA_INC_NONE;
    dma_config.size    = buffer_length;
    dma_config.msel    = DMA_MSEL_SPIx;
    dma_config.msigsel = DMA_MSIGSEL_SPI_TXEMPTY;
    dma_config.burst   = ENABLE;
    dma_config.interrupt = ENABLE;
    ald_dma_config_ping_pong(DMA0, &dma_config, 1, i2s_dma_recv_clk_callback);

    dma_config.channel = i2s_recv_dma_ch;
    dma_config.primary = DISABLE;
    dma_config.R_power = DMA_R_POWER_1;
    dma_config.data_width = DMA_DATA_SIZE_HALFWORD;
    dma_config.dst     = i2s_buffers[1];
    dma_config.dst_inc = DMA_DATA_INC_HALFWORD;
    dma_config.src     = (void *)(&SPIx->DATA);
    dma_config.src_inc = DMA_DATA_INC_NONE;
    dma_config.size    = buffer_length;
    dma_config.msel    = DMA_MSEL_SPIx;
    dma_config.msigsel = DMA_MSIGSEL_SPI_RNR;
    dma_config.burst   = ENABLE;
    dma_config.interrupt = ENABLE;
    ald_dma_config_ping_pong(DMA0, &dma_config, 0, i2s_dma_recv_callback);

    dma_config.channel = i2s_recv_dma_ch;
    dma_config.primary = ENABLE;
    dma_config.R_power = DMA_R_POWER_1;
    dma_config.data_width = DMA_DATA_SIZE_HALFWORD;
    dma_config.dst     = i2s_buffers[0];
    dma_config.dst_inc = DMA_DATA_INC_HALFWORD;
    dma_config.src     = (void *)(&SPIx->DATA);
    dma_config.src_inc = DMA_DATA_INC_NONE;
    dma_config.size    = buffer_length;
    dma_config.msel    = DMA_MSEL_SPIx;
    dma_config.msigsel = DMA_MSIGSEL_SPI_RNR;
    dma_config.burst   = ENABLE;
    dma_config.interrupt = ENABLE;
    ald_dma_config_ping_pong(DMA0, &dma_config, 1, i2s_dma_recv_callback);

    ald_i2s_dma_req_config(&g_h_i2s, I2S_DMA_REQ_TX, ENABLE);
    ald_i2s_dma_req_config(&g_h_i2s, I2S_DMA_REQ_RX, ENABLE);

    /* Enable I2S */
    SPIx->I2SCFG |= (1 << SPI_I2SCFG_I2SE_POS);
    /* Enable DMA */
    DMA0->CHENSET |= (1 << i2s_send_dma_ch) | (1 << i2s_recv_dma_ch);
}
