/**********************************************************************************
 *
 * @file    rd_buf.c
 * @brief   Read/write ring buffer
 *
 * @date    31 Dec 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          31 Dec 2021     shiwa           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ----------------------------------------------------------------- */
#include "rd_buf.h"
/* Private Macros ----------------------------------------------------------- */

/* Private Variables--------------------------------------------------------- */

/* Public Variables -------------------------------------------------------- */

/* Private Constants -------------------------------------------------------- */

/* Private function prototypes ---------------------------------------------- */

/* Private Function---------------------------------------------------------- */

uint8_t *rd_get(struct RD_buffer *buf)
{
    if (buf->num == 0)
        return NULL;

    return buf->buffer + buf->ibegin * buf->block_size;
}
int rd_pop(struct RD_buffer *buf)
{
    if (buf->num == 0)
        return -1;

    buf->ibegin = (buf->ibegin + 1) % buf->blocks;
    buf->num--;
    return 0;
}
uint8_t *rd_new(struct RD_buffer *buf)
{
    uint32_t iend = buf->iend;

    if (buf->num >= buf->blocks)
        return NULL;

    buf->iend = (buf->iend + 1) % buf->blocks;
    buf->num++;
    return buf->buffer + iend * buf->block_size;
}
int rd_put(struct RD_buffer *buf, uint8_t *data)
{
    if (buf->num >= buf->blocks)
        return -1;

    memcpy(buf->buffer + buf->iend * buf->block_size, data, buf->block_size);
    buf->iend = (buf->iend + 1) % buf->blocks;
    buf->num++;
    return 0;
}
int rd_reset(struct RD_buffer *buf)
{
    buf->ibegin = buf->iend;
    buf->num = 0;
    return 0;
}
struct RD_buffer *rd_static_init(void *data, uint32_t total_size, uint32_t blocks)
{
    struct RD_buffer *buf = (struct RD_buffer *)data;
    buf->buffer = data;
    buf->buffer += sizeof(struct RD_buffer);
    buf->buffer_size = total_size - sizeof(struct RD_buffer);
    buf->blocks = blocks;
    buf->block_size = buf->buffer_size / blocks;
    buf->ibegin = buf->iend = 0;
    buf->num = 0;
    return buf;
}
