/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    20 Mar 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          20 Mar 2023     shiwa           the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include <stdio.h>
#include "main.h"
#include "stdio_uart.h"
#include "wm_i2s.h"
#include "wm8978.h"
#include "adpcm.h"
#include "hpf.h"

/* Private Macros ------------------------------------------------------------ */
#define VOL_ADPCM 25
#define VOL_NORMAL 45
/* Private Variables --------------------------------------------------------- */
static i2c_handle_t g_hi2c;
static HPF_1orderRC_F g_hpf;
/* Public Variables ---------------------------------------------------------- */
int16_t audio_dma_buffer[2 * 1024] __attribute((aligned(4)));
struct wm_i2s_info g_wm;
uint32_t g_current_pos=0;
uint8_t g_use_adpcm=1;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */
void i2s0_pin_init(void);
void i2c0_init(void);
void init_audio(void);
uint16_t audio_send_data(void *buf, uint16_t length);
/* Private Function ---------------------------------------------------------- */
/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Initialize ALD */
    ald_cmu_init();
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_72M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

    /* Initialize pin */
    init_uart_pin();
    init_uart();
    i2s0_pin_init();
    i2c0_init();

    printf("hello\r\n");
    
    HPF_1orderRC_F_init(&g_hpf);
    init_audio();
    g_current_pos = 0;
    wm_set_callback(audio_send_data);
    wm_set_mode(WM_SEND);
    wm_set_vol(3,VOL_ADPCM);
    wm_play();
    while (1);
}
uint16_t audio_send_data(void *buf, uint16_t length)
{
    if (g_use_adpcm)
    {
        int16_t *dst = (int16_t *)buf;
        uint8_t *src = (uint8_t *)(g_sample_wav_data + g_current_pos);
        
        /* Samples = length/2 */
        /* Per sample has 4 bit */
        /* Read len = length /4 */
        if (g_current_pos+length/4>=g_sample_wav_data_len)
        {
            length=(g_sample_wav_data_len-g_current_pos)*4;
            g_current_pos=0;
            //g_use_adpcm=0;
            wm_set_vol(3,VOL_NORMAL);
        }
        else
        {
            g_current_pos+=length/4;
        }
        for (int i=0;i<length/4;i++)
        {
            int16_t sample;
            sample=ADPCM_Decode(src[i]&0xF);
            //sample=HPF_1orderRC_F_FUNC_i(&g_hpf,sample);
            dst[i*4]=dst[i*4+1]=sample;
            sample=ADPCM_Decode(src[i]>>4);
            //sample=HPF_1orderRC_F_FUNC_i(&g_hpf,sample);
            dst[i*4+2]=dst[i*4+3]=sample;
        }
        return length;
    }
    else
    {
        int16_t *dst = (int16_t *)buf;
        int16_t *src = (int16_t *)(g_sample2_wav_data + g_current_pos);
        
        /* Samples = length/2 */
        /* Per sample has 16 bit I2bytes)*/
        /* Read len = length*/
        if (g_current_pos+length>=g_sample2_wav_data_len)
        {
            length=(g_sample2_wav_data_len-g_current_pos);
            g_current_pos=0;
            g_use_adpcm=1;
            wm_set_vol(3,VOL_ADPCM);
        }
        else
        {
            g_current_pos+=length;
        }
        for (int i = 0; i < length / 2; i++)
        {
            dst[i * 2] = dst[i * 2 + 1] = src[i];
        }
        return length;
    }

}
void init_audio()
{
    i2c0_init();

    if (WM8978_Init(&g_hi2c))
    {
        printf("Error WM8978 init\r\n");
        return ;
    }

    struct wm_i2s_info wm_info;
    uint8_t *g_audio_buffer;

    g_audio_buffer = (uint8_t *)audio_dma_buffer;
    wm_info.buffers[0] = g_audio_buffer;
    wm_info.buffers[1] = g_audio_buffer + 1024 * 2;
    wm_info.buffer_length = 1024;
    wm_info.spi_x = SPI0;
    wm_info.feed_data = audio_send_data;
    wm_info.sample_rate = 8000;
    wm_info.dma_tx_ch = 2;
    wm_info.dma_rx_ch = 3;

    if (wm_i2s_init(&wm_info))
    {
        printf("I2S init failed\r\n");
    }

    printf("Audio init ok\r\n");
}

/**
  * @brief  ???I2S0 ??
  * @retval None
*/
void i2s0_pin_init(void)
{
    gpio_init_t x;

    /* Initialize NSS/WS pin */
    x.mode  = GPIO_MODE_OUTPUT;
    x.odos  = GPIO_PUSH_PULL;
    x.pupd  = GPIO_PUSH_DOWN;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_TTL;
    x.func  = GPIO_FUNC_6;
    ald_gpio_init(GPIOA, GPIO_PIN_4, &x);

    /* Initialize SCK/CK pin */
    x.mode  = GPIO_MODE_OUTPUT;
    x.odos  = GPIO_PUSH_PULL;
    x.pupd  = GPIO_PUSH_DOWN;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_TTL;
    x.func  = GPIO_FUNC_4;
    ald_gpio_init(GPIOB, GPIO_PIN_3, &x);

    /* Initialize MISO/MCLK pin */
    x.mode  = GPIO_MODE_OUTPUT;
    x.odos  = GPIO_PUSH_PULL;
    x.pupd  = GPIO_PUSH_DOWN;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_TTL;
    x.func  = GPIO_FUNC_4;
    ald_gpio_init(GPIOB, GPIO_PIN_4, &x);

    /* Initialize MOSI/SD pin */
    x.mode  = GPIO_MODE_OUTPUT;
    x.odos  = GPIO_PUSH_PULL;
    x.pupd  = GPIO_PUSH_DOWN;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_TTL;
    x.func  = GPIO_FUNC_4;
    ald_gpio_init(GPIOB, GPIO_PIN_5, &x);

    return;
}

/**
  * @brief  ???I2C0
  * @retval None
*/
void i2c0_init()
{
    gpio_init_t x;

    /* Initialize SCK pin */
    x.mode  = GPIO_MODE_OUTPUT;
    x.odos  = GPIO_OPEN_DRAIN;
    x.pupd  = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_CMOS;
    x.func  = GPIO_FUNC_5;
    ald_gpio_init(GPIOB, GPIO_PIN_6, &x);

    /* Initialize SDA pin */
    x.mode  = GPIO_MODE_OUTPUT;
    x.odos  = GPIO_OPEN_DRAIN;
    x.pupd  = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_CMOS;
    x.func  = GPIO_FUNC_5;
    ald_gpio_init(GPIOB, GPIO_PIN_7, &x);

    memset(&g_hi2c, 0, sizeof(g_hi2c));
    g_hi2c.perh = I2C0;
    g_hi2c.init.module  = I2C_MODULE_MASTER;
    g_hi2c.init.addr_mode    = I2C_ADDR_7BIT;
    g_hi2c.init.clk_speed    = 100000;
    g_hi2c.init.dual_addr    = I2C_DUALADDR_ENABLE;
    g_hi2c.init.general_call = I2C_GENERALCALL_ENABLE;
    g_hi2c.init.no_stretch   = I2C_NOSTRETCH_DISABLE;
    g_hi2c.init.own_addr1    = 0xA0;
    ald_i2c_init(&g_hi2c);
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
