/**********************************************************************************
 *
 * @file    spi_flash_diskio.c
 * @brief   IO defination of spi flash
 *
 * @date    07 Dec 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          07 Dec 2021     biyq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "spi_flash_diskio.h"

/* Private Macros ------------------------------------------------------------ */

/* Private variables ---------------------------------------------------------*/

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes -----------------------------------------------*/
DSTATUS spi_flash_init(BYTE);
DSTATUS spi_flash_stat(BYTE);
DRESULT spi_flash_read(BYTE, BYTE *, DWORD, UINT);
#if  _USE_WRITE
    DRESULT spi_flash_write(BYTE, const BYTE *, DWORD, UINT);
#endif
#if  _USE_IOCTL
    DRESULT spi_flash_ioctl(BYTE, BYTE, void *);
#endif
/* Public Variables ---------------------------------------------------------- */
const Diskio_drvTypeDef spi_flash_drv =
{
    spi_flash_init,
    spi_flash_stat,
    spi_flash_read,

#if  _USE_WRITE
    spi_flash_write,
#endif

#if  _USE_IOCTL
    spi_flash_ioctl,
#endif
};

/* Private functions ---------------------------------------------------------*/
/**
  * @brief  Initializes a Drive
  * @param  NONE
  * @retval DSTATUS: Operation status
  */
DSTATUS spi_flash_init(BYTE lun)
{
    /* Configure the NAND flash device */
    if (ll_spi_flash_init())
    {
        return RES_ERROR;
    }

    return RES_OK;
}

/**
  * @brief  Gets Disk Status
  * @param  parm_num
  * @param  param
  * @retval DSTATUS: Operation status
  */
DSTATUS spi_flash_stat(BYTE lun)
{
    return RES_OK;
}

/**
  * @brief  Reads Data
  * @param  *buff: Data buffer to store read data
  * @param  sector: Sector address (LBA)
  * @param  count: Number of sectors to read (1..128)
  * @retval DRESULT: Operation result
  */
DRESULT spi_flash_read(BYTE lun, BYTE *buff, DWORD sector, UINT count)
{
    ald_status_t res;

    __disable_irq();

    res = ll_flash_sector_read(buff, (uint32_t)(sector), count);

    __enable_irq();

    if (res == ERROR)
        return RES_ERROR;
    else
        return RES_OK;
}

/**
  * @brief  Writes Sector(s)
  * @param  *buff: Data to be written
  * @param  sector: Sector address (LBA)
  * @param  count: Number of sectors to write (1..128)
  * @retval DRESULT: Operation result
  */
#if _USE_WRITE
DRESULT spi_flash_write(BYTE lun, const BYTE *buff, DWORD sector, UINT count)
{
    ald_status_t res;

    __disable_irq();

    res = ll_flash_sector_erase(sector, sector + count - 1);

    if (res == OK)
        ll_flash_sector_write(buff, sector, count);

    __enable_irq();

    if (res == ERROR)
        return RES_ERROR;
    else
        return RES_OK;
}
#endif

#if  _USE_IOCTL
/**
 * @brief  I/O control operation
 * @param  cmd: Control code
 * @param  *buff: Buffer to send/receive control data
 * @retval DRESULT: Operation result
 */
DRESULT spi_flash_ioctl(BYTE lun, BYTE cmd, void *buff)
{
    DRESULT res = RES_ERROR;

    switch (cmd)
    {
        /* Make sure that no pending write process */
        case CTRL_SYNC:
            res = RES_OK;
            break;

        /* Get number of sectors on the disk (DWORD) */
        case GET_SECTOR_COUNT :
            *(DWORD *)buff = DEV_SIZE * BLK_SIZE;
            res = RES_OK;
            break;

        /* Get R/W sector size (WORD) */
        case GET_SECTOR_SIZE:
            *(WORD *)buff = SEC_SIZE;
            res = RES_OK;
            break;

        /* Get erase block size in unit of sector (DWORD) */
        case GET_BLOCK_SIZE:
            *(DWORD *)buff = BLK_SIZE;
            res = RES_OK;
            break;

        /* Informs the data on the block of sectors can be erased. */
        case CTRL_TRIM:

            __disable_irq();

            if (! ll_flash_sector_erase(*(DWORD *)buff, *((DWORD *)buff + 1)))
                res = RES_OK;

            __enable_irq();

            break;

        default:
            res = RES_PARERR;
            break;
    }

    return res;
}

#endif

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
