/**********************************************************************************
 *
 * @file    spi_flash.c
 * @brief   SPI Flash file for DEMO
 *
 * @date    07 Dec 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          07 Dec 2021     biyq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include "spi_flash.h"

/* Private Macros ------------------------------------------------------------ */
/* CMD */
#define FLASH_WRITE_ENABLE  0x06
#define FLASH_SEC_ERASE     0x20
#define FLASH_PAGE_PROGRAM  0x02
#define FLASH_READ          0x03
#define FLASH_ID            0x9F
#define FLASH_STATUS        0x05

/* flash information */
#define PAGE_SIZE           256
#define PAGE_PER_SEC        16

/* function */
#define FLASH_CS_SET() (ald_gpio_write_pin(GPIOB, GPIO_PIN_0, 1))
#define FLASH_CS_RST() (ald_gpio_write_pin(GPIOB, GPIO_PIN_0, 0))

/* Private Variables --------------------------------------------------------- */
static spi_handle_t s_spi;

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initializate spi flash pin
  * @retval None.
  */
static void spi_pin_init(void)
{
    gpio_init_t l_gpio;

    l_gpio.mode = GPIO_MODE_OUTPUT;
    l_gpio.odos = GPIO_PUSH_PULL;
    l_gpio.pupd = GPIO_PUSH_UP;
    l_gpio.flt  = GPIO_FILTER_DISABLE;
    l_gpio.nodrv = GPIO_OUT_DRIVE_1;
    l_gpio.podrv = GPIO_OUT_DRIVE_1;
    l_gpio.type = GPIO_TYPE_TTL;
    l_gpio.func = GPIO_FUNC_1;

    ald_gpio_init(GPIOB, GPIO_PIN_0, &l_gpio);  /*Initialize the selected pin*/
    FLASH_CS_SET();   /*Choose the output high, release of Flash*/

    l_gpio.func = GPIO_FUNC_4;
    ald_gpio_init(GPIOD, GPIO_PIN_3, &l_gpio);  /*Initialize the clock output pin*/

    l_gpio.func = GPIO_FUNC_4;
    ald_gpio_init(GPIOB, GPIO_PIN_5, &l_gpio);  /*Initialize the MOSI pin*/

    l_gpio.mode = GPIO_MODE_INPUT;
    l_gpio.func = GPIO_FUNC_4;
    ald_gpio_init(GPIOB, GPIO_PIN_4, &l_gpio);  /*Initializes the MISO pins*/

    return;
}

/**
  * @brief  wait until flash unbusy.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t flash_wait_unbusy(void)
{
    uint8_t status;
    int   r_flag;

    /* enable the selected Flash */
    FLASH_CS_RST();

    /* send command */
    if (ald_spi_send_byte_fast(&s_spi, (uint8_t)FLASH_STATUS) != OK)
    {
        FLASH_CS_SET();
        return ERROR;
    }

    /* verify flash not in write progress */
    do
    {
        /* read status */
        status = ald_spi_recv_byte_fast(&s_spi, &r_flag);

        if (r_flag != OK)
        {
            FLASH_CS_SET();
            return ERROR;
        }
    }
    while (status & 0x01);

    /* release flash */
    FLASH_CS_SET();

    return OK;
}

/**
  * @brief  flash write enable function.
  * @retval None.
  */
static ald_status_t flash_write_enable(void)
{
    /* wait until flash unbusy */
    flash_wait_unbusy();

    /* enable the selected Flash */
    FLASH_CS_RST();

    /* send write enable command */
    if (ald_spi_send_byte_fast(&s_spi, FLASH_WRITE_ENABLE) != OK)
    {
        FLASH_CS_SET();
        return ERROR;
    }

    /* release Flash */
    FLASH_CS_SET();

    /* wait until flash unbusy */
    flash_wait_unbusy();

    return OK;
}

/* Public Function ---------------------------------------------------------- */
/**
  * @brief  low-level flash sector erase function.
  * @param  start_sec: Specified start sector to be erase.
  * @param  end_sec: Specified end sector to be erase.
  * @retval Status.
  */
ald_status_t ll_flash_sector_erase(uint32_t start_sec, uint32_t end_sec)
{
    uint8_t  cmd_buf[4];
    uint8_t  i = 0U;
    uint32_t sec_addr;

    /*Flash sector erase command*/
    cmd_buf[0] = FLASH_SEC_ERASE;

    /*Erase Flash sectors */
    for (; start_sec <= end_sec; start_sec++)
    {
        /* group command and address */
        sec_addr = start_sec * SEC_SIZE;
        cmd_buf[1] = (sec_addr >> 16) & 0xFF;   /*24 bit Flash address*/
        cmd_buf[2] = (sec_addr >> 8) & 0xFF;
        cmd_buf[3] = (sec_addr) & 0xFF;

        /* enable flash write */
        flash_write_enable();

        /* erase current sector */
        /* enable the selected Flash */
        FLASH_CS_RST();

        /*Send the sector erase instructions and Flash address three bytes*/
        for (i = 0; i < sizeof(cmd_buf); i++)
        {
            if (ald_spi_send_byte_fast(&s_spi, cmd_buf[i]) != OK)
            {
                FLASH_CS_SET();
                return ERROR;
            }
        }

        /* release Flash */
        FLASH_CS_SET();

        /* wait until flash not in write progress */
        flash_wait_unbusy();
    }

    return OK;
}

/**
  * @brief  low-level flash sector write operation function.
  * @param  src_buf: pointer to the expected written data.
  * @param  sec: start sector to write.
  * @param  sec_num: amount of sector.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ll_flash_sector_write(const uint8_t *src_buf, uint32_t sec, uint32_t sec_num)
{
    uint8_t  cmd_buf[4];
    uint32_t  i, j, k;
    uint32_t page_addr, sec_addr;

    if (src_buf == NULL)
        return ERROR;

    /*Flash page program command*/
    cmd_buf[0] = FLASH_PAGE_PROGRAM;

    /*Erase Flash sectors */
    for (i = 0; i < sec_num; i++)
    {
        /* calculate current sector start address */
        sec_addr = sec * SEC_SIZE;

        for (j = 0; j < PAGE_PER_SEC; j++)
        {
            /* group command and address */
            page_addr = sec_addr + PAGE_SIZE * j;
            cmd_buf[1] = (page_addr >> 16) & 0xff;
            cmd_buf[2] = (page_addr >> 8) & 0xff;
            cmd_buf[3] = (page_addr) & 0xff;

            /* enable flash write */
            flash_write_enable();

            /* write current page */
            /* enable the selected Flash */
            FLASH_CS_RST();

            /*Send the sector erase instructions and Flash address three bytes*/
            for (k = 0; k < sizeof(cmd_buf); k++)
            {
                if (ald_spi_send_byte_fast(&s_spi, cmd_buf[k]) != OK)
                {
                    FLASH_CS_SET();
                    return ERROR;
                }
            }

            /* write the data to the Flash*/
            for (k = 0; k < PAGE_SIZE; k++)
            {
                if (ald_spi_send_byte_fast(&s_spi, src_buf[i * SEC_SIZE + j * PAGE_SIZE + k]) != OK)
                {
                    FLASH_CS_SET();
                    return ERROR;
                }
            }

            /* release Flash */
            FLASH_CS_SET();

            /* wait until flash not in write progress */
            flash_wait_unbusy();
        }
    }

    return OK;
}

/**
  * @brief  low-level flash sector read operation function.
  * @param  str_buf: pointer to the first item of the byte array to store read data.
  * @param  sec: start sector.
  * @param  sec_num: amount of sector to be read.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t ll_flash_sector_read(uint8_t *str_buf, uint32_t sec, uint32_t sec_num)
{

    uint8_t  cmd_buf[4];
    uint32_t  i, j;
    uint32_t sec_addr;
    int      r_flag;

    if (str_buf == NULL)
        return ERROR;

    /*Flash read command*/
    cmd_buf[0] = FLASH_READ;
    /* group command and address */
    sec_addr = sec * SEC_SIZE;
    cmd_buf[1] = (sec_addr >> 16) & 0xFF;   /*24 bit Flash address*/
    cmd_buf[2] = (sec_addr >> 8) & 0xFF;
    cmd_buf[3] = (sec_addr) & 0xFF;

    /* enable the selected Flash */
    FLASH_CS_RST();

    /* Send the read instructions and Flash address */
    for (i = 0; i < sizeof(cmd_buf); i++)
    {
        if (ald_spi_send_byte_fast(&s_spi, cmd_buf[i]) != OK)
        {
            FLASH_CS_SET();
            return ERROR;
        }
    }

    /* read sectors data */
    for (i = 0; i < sec_num; i++)
    {
        /* read data cycle */
        for (j = 0; j < SEC_SIZE; j++)
        {
            str_buf[i * SEC_SIZE + j] = ald_spi_recv_byte_fast(&s_spi, &r_flag);

            if (r_flag != OK)
            {
                FLASH_CS_SET();
                return ERROR;
            }
        }
    }

    /* release Flash */
    FLASH_CS_SET();

    return OK;
}

/**
  * @brief spi flash init function
  * @retval None.
  */
ald_status_t ll_spi_flash_init(void)
{
    spi_pin_init();

    s_spi.perh           = SPI0;               /*Using SPI0*/
    s_spi.init.mode      = SPI_MODE_MASTER;    /*SPI host mode*/
    s_spi.init.baud      = SPI_BAUD_64;        /*clock / 64*/
    s_spi.init.data_size = SPI_DATA_SIZE_8;    /*8 bit pattern*/
    s_spi.init.polarity  = SPI_CPOL_HIGH;      /*Free high level*/
    s_spi.init.phase     = SPI_CPHA_SECOND;    /*The second edge receiving data*/
    s_spi.init.first_bit = SPI_FIRSTBIT_MSB;   /*Send the MSB first*/
    s_spi.init.dir       = SPI_DIRECTION_2LINES;
    s_spi.init.ss_en     = DISABLE;
    s_spi.init.crc_calc  = DISABLE;

    if (ald_spi_init(&s_spi) == OK)  /*According to the parameter initialization SPI peripherals*/
        return OK;
    else
        return ERROR;
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
