/**
  *********************************************************************************
  *
  * @file    main.c
  * @brief   Main file for DEMO
  *
  * @version V1.0
  * @date    26 Jun 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          26 Jun 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include <string.h>
#include "main.h"
#include "support_pdfgen.h"
#include "es_pdf_test.h"

#include "spi_flash_diskio.h"
#include "spi_flash.h"

#include <string.h>
#include "main.h"
#include "usbd_msc.h"

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

#define STATE_DEVICE_NO     0x01
#define STATE_DEVICE_CONN   0x02
#define STATE_DEVICE_READY  0x04
#define STATE_DEVICE_READ   0x08
#define STATE_DEVICE_WRITE  0x10
#define STATE_DEVICE_IDLE   0x20
#define STATE_POWER_FAULT   0x40

FATFS   es_fatfs;        /* File system object for flash volume */
uint8_t es_fatfs_path[32];
bool start_flag = false;

uint32_t idx_media = 1;
uint32_t _msc_flag = STATE_DEVICE_NO;
uint32_t _per_flag = STATE_DEVICE_NO;

uint32_t msc_event_handle(void *data, uint32_t event, uint32_t value, void *p_data);

/**
  * @brief  Initializate pin of uart module.
  * @retval None
  */
void usb_pin_init(void)
{
    gpio_init_t x;

    /* Initialize HSCO pin */
    x.mode = GPIO_MODE_OUTPUT;
    x.odos = GPIO_PUSH_PULL;
    x.pupd = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_20;
    x.nodrv = GPIO_OUT_DRIVE_20;
    x.flt  = GPIO_FILTER_DISABLE;
    x.type = GPIO_TYPE_TTL;
    x.func = GPIO_FUNC_6;
    ald_gpio_init(GPIOA, GPIO_PIN_15, &x);

    return;
}

const uint8_t lang_desc[] =
{
    4,
    USB_DTYPE_STRING,
    USBShort(USB_LANG_EN_US)
};

const uint8_t manufact_str[] =
{
    (17 + 1) * 2,
    USB_DTYPE_STRING,
    'E', 0, 'a', 0, 's', 0, 't', 0, 's', 0, 'o', 0, 'f', 0, 't', 0, ' ', 0,
    'S', 0, 'h', 0, 'a', 0, 'n', 0, 'g', 0, 'h', 0, 'a', 0, 'i', 0
};

const uint8_t product_str[] =
{
    (19 + 1) * 2,
    USB_DTYPE_STRING,
    'M', 0, 'a', 0, 's', 0, 's', 0, ' ', 0, 'S', 0, 't', 0, 'o', 0,
    'r', 0, 'a', 0, 'g', 0, 'e', 0, ' ', 0, 'D', 0, 'e', 0, 'v', 0,
    'i', 0, 'c', 0, 'e', 0
};

const uint8_t serial_num_str[] =
{
    (8 + 1) * 2,
    USB_DTYPE_STRING,
    '1', 0, '2', 0, '3', 0, '4', 0, '5', 0, '6', 0, '7', 0, '8', 0
};

const uint8_t data_interface_str[] =
{
    (19 + 1) * 2,
    USB_DTYPE_STRING,
    'B', 0, 'u', 0, 'l', 0, 'k', 0, ' ', 0, 'D', 0, 'a', 0, 't', 0,
    'a', 0, ' ', 0, 'I', 0, 'n', 0, 't', 0, 'e', 0, 'r', 0, 'f', 0,
    'a', 0, 'c', 0, 'e', 0
};

const uint8_t config_str[] =
{
    (23 + 1) * 2,
    USB_DTYPE_STRING,
    'B', 0, 'u', 0, 'l', 0, 'k', 0, ' ', 0, 'D', 0, 'a', 0, 't', 0,
    'a', 0, ' ', 0, 'C', 0, 'o', 0, 'n', 0, 'f', 0, 'i', 0, 'g', 0,
    'u', 0, 'r', 0, 'a', 0, 't', 0, 'i', 0, 'o', 0, 'n', 0
};

const uint8_t *const string_desc[] =
{
    lang_desc,
    manufact_str,
    product_str,
    serial_num_str,
    data_interface_str,
    config_str
};

#define NUM_STRING_DESCRIPTORS (sizeof(string_desc) / sizeof(uint8_t *))

void *msc_open(uint32_t device)
{
    return (void *)&idx_media;
}

void msc_close(void *device)
{
    return;
}

uint32_t msc_read(void *device, uint8_t *buf, uint32_t sector, uint32_t num_block)
{
    if (spi_flash_read(0, buf, sector, num_block))
        return 0;
    else
        return (num_block * SEC_SIZE);
}

uint32_t msc_write(void *device, uint8_t *buf, uint32_t sector, uint32_t num_block)
{
    if (spi_flash_write(0, buf, sector, num_block))
        return 0;
    else
        return (num_block * SEC_SIZE);
}

uint32_t msc_get_block_num(void *device)
{
    /* size = (DISK_BLOCK_NR * DISK_BLOCK_SIZE) */
    return (BLK_SIZE * DEV_SIZE);
}

uint32_t msc_get_block_size(void *device)
{
    return SEC_SIZE;
}

usbd_msc_dev_t msc_device =
{
    USB_VID_EASTSOFT_30CC,
    USB_PID_MSC,
    "EASTSOFT",
    "Mass Storage    ",
    "1.00",
    500,
    USB_CONF_ATTR_SELF_PWR,
    string_desc,
    NUM_STRING_DESCRIPTORS,
    {
        msc_open,
        msc_close,
        msc_read,
        msc_write,
        msc_get_block_num,
        msc_get_block_size,
    },
    msc_event_handle,
};

uint32_t msc_event_handle(void *data, uint32_t event, uint32_t value, void *p_data)
{
    switch (event)
    {
        case USB_EVENT_DISCONNECTED:
            _msc_flag = STATE_DEVICE_NO;

            if (_per_flag == STATE_DEVICE_CONN)
            {
                _per_flag = STATE_DEVICE_NO;
                start_flag = true;
            }

            break;

        case USB_EVENT_CONNECTED:
            _msc_flag = STATE_DEVICE_CONN;
            _per_flag = STATE_DEVICE_CONN;

            break;

        case USBD_MSC_EVENT_WRITING:
            _msc_flag = STATE_DEVICE_WRITE;
            break;

        case USBD_MSC_EVENT_READING:
            _msc_flag = STATE_DEVICE_READ;
            break;

        case USBD_MSC_EVENT_IDLE:
            _msc_flag = STATE_DEVICE_IDLE;
            break;

        default:
            break;
    }

    return 0;
}

void spiflash_fatfs_pdfgen(void)
{
    spi_flash_init(0);
    
    /* Regeister the disk I/O driver */
    if (FATFS_LinkDriverEx(&spi_flash_drv, (char *)es_fatfs_path, 0) == 0)
    {
        /* Register the file system object to the FatFs module */
        if (f_mount(&es_fatfs, (const TCHAR *)es_fatfs_path, 1) != FR_OK)
        {
            if(sizeof(usbd_msc_dev_t) > (FF_MAX_SS))
            {
                BYTE *work = (BYTE *)(&msc_device);
                
                if (f_mkfs((const TCHAR *)es_fatfs_path, FM_ANY, SEC_SIZE, work, FF_MAX_SS) != FR_OK)
                {
                    printf_e("f_mkfs err\r\n");
                }
                /*复位*/
                SYSCFG_UNLOCK();
                RMU->AHB2RSTR = RMU_AHB2RSTR_CHIPRST_MSK;
                SYSCFG_LOCK();
            }
            
            printf_e("\r\nfatfs mount fail\r\n", es_fatfs_path);
        }
        else
        {
            printf_e("\r\nfatfs mount succes , path :%s\r\n", es_fatfs_path);
        }
    }

    /*Display the contents in the file system and generate pdf*/
    es_fatfs_ls(0, 0);
    printf_e("\r\nstart generate pdf\r\n");
    es_pdf_test(NULL);
    es_fatfs_ls(0, 0);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    /* Initialize ALD */
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_48M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);

    /* Enable mode clock */
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

    uart_stdio_init();
    printf_e("System start...\n\r");

    spiflash_fatfs_pdfgen();

    /*USB init*/
    ald_pmu_perh_power_config(PMU_POWER_USB, ENABLE);
    ald_cmu_perh_clock_config(CMU_PERH_USB, ENABLE);
    ald_cmu_perh_clock_config(CMU_PERH_GPIO, ENABLE);
    ald_cmu_usb_clock_config(CMU_USB_CLOCK_SEL_HOSC, CMU_USB_DIV_1);
    ald_rmu_reset_periperal(RMU_PERH_USB);
    ald_mcu_irq_config(USB_INT_IRQn, 2, 2, ENABLE);
    ald_mcu_irq_config(USB_DMA_IRQn, 2, 2, ENABLE);
    usb_pin_init();
    ald_usb_high_speed_enable(false);
    ald_usb_otg_session_request(true);
    usbd_msc_init(0, &msc_device);

    while (1)
    {
        switch (_msc_flag)
        {
            case STATE_DEVICE_CONN:
                printf_e("\rMSC Ready!\n\r");
                _msc_flag = STATE_DEVICE_READY;
                break;

            case  STATE_DEVICE_READY:
                break;

            case STATE_DEVICE_IDLE:
            {
                /*Synchronize file system information and view files*/
                if (f_mount(0, (const TCHAR *)es_fatfs_path, 0) == FR_OK)
                {
                    if (f_mount(&es_fatfs, (const TCHAR *)es_fatfs_path, 1) == FR_OK)
                    {
                        es_fatfs_ls(0, 0);
                    }
                }

                ald_delay_ms(10000);
            }

            case STATE_DEVICE_NO:
            case STATE_POWER_FAULT:
                break;

            default:
                break;
        }
    }
}

/**
  * @}
  */
/**
  * @}
  */
