/**********************************************************************************
 *
 * @file    spi_flash_diskio.c
 * @brief   IO defination of spi flash
 *
 * @date    07 Dec 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          07 Dec 2021     biyq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------*/
#include "usbh_msc_diskio.h"
#include "string.h"

/* Private Macros ------------------------------------------------------------ */

#define ES_FATFS_BASE_PATH_MAX_LEN (32)
#define GET_BE32(field) \
    (((uint32_t)(field)[0] << 24) | ((uint32_t)(field)[1] << 16) | ((uint32_t)(field)[2] << 8) | ((uint32_t)(field)[3] << 0))

/* Private variables ---------------------------------------------------------*/

FATFS   es_fatfs;        /* File system object for NAND flash volume */
static uint8_t es_fatfs_path[ES_FATFS_BASE_PATH_MAX_LEN];

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes -----------------------------------------------*/
DSTATUS usbh_msc_fatfs_drv_init(BYTE);
DSTATUS usbh_msc_fatfs_drv_stat(BYTE);
DRESULT usbh_msc_fatfs_drv_read(BYTE, BYTE *, DWORD, UINT);
#if  _USE_WRITE
    DRESULT usbh_msc_fatfs_drv_write(BYTE, const BYTE *, DWORD, UINT);
#endif
#if  _USE_IOCTL
    DRESULT usbh_msc_fatfs_drv_ioctl(BYTE, BYTE, void *);
#endif
/* Public Variables ---------------------------------------------------------- */
const Diskio_drvTypeDef usbh_msc_fatfs_drv =
{
    usbh_msc_fatfs_drv_init,
    usbh_msc_fatfs_drv_stat,
    usbh_msc_fatfs_drv_read,

#if  _USE_WRITE
    usbh_msc_fatfs_drv_write,
#endif

#if  _USE_IOCTL
    usbh_msc_fatfs_drv_ioctl,
#endif
};

/* Private functions ---------------------------------------------------------*/
/**
  * @brief  Initializes a Drive
  * @param  NONE
  * @retval DSTATUS: Operation status
  */
DSTATUS usbh_msc_fatfs_drv_init(BYTE lun)
{
    /* Configure the NAND flash device */
    if (!(env.flag))
    {
        return RES_ERROR;
    }

    return RES_OK;
}

/**
  * @brief  Gets Disk Status
  * @param  parm_num
  * @param  param
  * @retval DSTATUS: Operation status
  */
DSTATUS usbh_msc_fatfs_drv_stat(BYTE lun)
{
    return RES_OK;
}

/**
  * @brief  Reads Data
  * @param  *buff: Data buffer to store read data
  * @param  sector: Sector address (LBA)
  * @param  count: Number of sectors to read (1..128)
  * @retval DRESULT: Operation result
  */
DRESULT usbh_msc_fatfs_drv_read(BYTE lun, BYTE *buff, DWORD sector, UINT count)
{
    if (usbh_msc_block_read(inst, sector, buff, count))
    {
        return RES_ERROR;
    }

    return RES_OK;
}

/**
  * @brief  Writes Sector(s)
  * @param  *buff: Data to be written
  * @param  sector: Sector address (LBA)
  * @param  count: Number of sectors to write (1..128)
  * @retval DRESULT: Operation result
  */
#if _USE_WRITE

DRESULT usbh_msc_fatfs_drv_write(BYTE lun, const BYTE *buff, DWORD sector, UINT count)
{
    if (usbh_msc_block_write(inst, sector, (uint8_t *)buff, count))
    {
        return RES_ERROR;
    }

    return RES_OK;
}
#endif

#if  _USE_IOCTL
/**
 * @brief  I/O control operation
 * @param  cmd: Control code
 * @param  *buff: Buffer to send/receive control data
 * @retval DRESULT: Operation result
 */
DRESULT usbh_msc_fatfs_drv_ioctl(BYTE lun, BYTE cmd, void *buff)
{
    DRESULT res = RES_ERROR;

    switch (cmd)
    {
        /* Make sure that no pending write process */
        case CTRL_SYNC:
            res = RES_OK;
            break;

        /* Get number of sectors on the disk (DWORD) */
        case GET_SECTOR_COUNT :
        {
            uint8_t usb_msc_capacites_info[8];
            uint32_t size = 8;

            usbh_scsi_read_capacity(inst->in_pipe, inst->out_pipe, usb_msc_capacites_info, &size);

            *(DWORD *)buff = GET_BE32(&(usb_msc_capacites_info[0]));
            res = RES_OK;
            break;
        }

        /* Get R/W sector size (WORD) */
        case GET_SECTOR_SIZE:
        {
            uint8_t usb_msc_capacites_info[8];
            uint32_t size = 8;

            usbh_scsi_read_capacity(inst->in_pipe, inst->out_pipe, usb_msc_capacites_info, &size);

            *(WORD *)buff = GET_BE32(&(usb_msc_capacites_info[4]));
            res = RES_OK;
            break;
        }

        /* Get erase block size in unit of sector (DWORD) */
        case GET_BLOCK_SIZE:
        {
            uint8_t usb_msc_capacites_info[8];
            uint32_t size = 8;

            usbh_scsi_read_capacity(inst->in_pipe, inst->out_pipe, usb_msc_capacites_info, &size);

            *(DWORD *)buff = GET_BE32(&(usb_msc_capacites_info[4]));
            res = RES_OK;
            break;
        }

        /* Informs the data on the block of sectors can be erased. */
        case CTRL_TRIM:
            //   if (! ll_flash_sector_erase(*(DWORD *)buff, *((DWORD *)buff + 1)))
            res = RES_ERROR;

            break;

        default:
            res = RES_PARERR;
            break;
    }

    return res;
}

void es_fatfs_error_handler(void)
{
    printf_e("[es_fatfs_error]");
}

void es_usb_msc_fatfs_register(void)
{
    /* Regeister the disk I/O driver */
    if (FATFS_LinkDriverEx(&usbh_msc_fatfs_drv, (char *)es_fatfs_path, 0) == 0)
    {
        /* Register the file system object to the FatFs module */
        if (f_mount(&es_fatfs, (const TCHAR *)es_fatfs_path, 0) != FR_OK)
        {
            /* FatFs Initialization Error */
            es_fatfs_error_handler();

            /*nUnlink the spi Flash disk I/O driver */
            FATFS_UnLinkDriverEx((char *)es_fatfs_path, 0);
        }
        else
        {
            printf_e("\r\nfatfs mount succes , path :%s\r\n", es_fatfs_path);
        }
    }
}

uint32_t es_fatfs_ls(uint32_t p_num, uint8_t **p_strs)
{
    char *default_fullpath = "0:/";
    char *fullpath;
    DIR dir;
    FILINFO fno;
    FRESULT res;

    if (p_num)
        fullpath = (char *)(p_strs[0]);
    else
        fullpath =  default_fullpath;

    res = f_opendir(&dir, fullpath);

    if (res == FR_OK)
    {
        printf_e("\r\n%s\r\n", fullpath);

        while (1)
        {
            res = f_readdir(&dir, &fno);

            if ((res != FR_OK) || (fno.fname[0] == '\0'))
                break;

            if ((fno.fattrib) & (AM_DIR))
            {
                printf_e("%-32s ", fno.fname);
                printf_e("<DIR>\r\n", fno.fsize);
            }
            else
            {
                printf_e("%-32s ", fno.fname);
                printf_e("%d\r\n", fno.fsize);
            }
        }
    }
    else
    {
        es_fatfs_error_handler();
    }

    return 0;
}

#endif

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
