/**
  *********************************************************************************
  *
  * @file    usbd_hid_game.c
  * @brief   USB HID game pad device driver.
  *
  * @version V1.0
  * @date    28 Feb 2022
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          28 Feb 2022     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "usbd_hid_custom.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup DEVICE
  * @{
  */
/** @defgroup Device_HID_User HID User
  * @brief Device HID driver
  * @{
  */
/** @defgroup Device_Hid_User_Private_Variables Private Variables
  * @{
  */
/**
  * @brief Configuration descriptor header
  */
static uint8_t __hid_user_desc[] =
{
    9,                       /**< Size of the descriptor */
    USB_DTYPE_CONFIGURATION, /**< Type of the descriptor */
    USBShort(41),            /**< Total size of the descriptor */
    1,                       /**< Number of the inerfaces */
    1,                       /**< Unique value for this configuration */
    5,                       /**< Index of the string */
    USB_CONF_ATTR_SELF_PWR,  /**< Type of the power */
    250,                     /**< Maximum power in 2mA increments */
};

/**
  * @brief Interface descriptor
  */
static uint8_t __hid_interface[HID_INTERFACE_SIZE] =
{
    9,                   /**< Size of the interface */
    USB_DTYPE_INTERFACE, /**< Type of the interface */
    0,                   /**< Index of the interface */
    0,                   /**< Alternate setting */
    2,                   /**< Number of endpoints in the interface */
    USB_CLASS_HID,       /**< Interface class */
    0,                   /**< Interface sub-class */
    0,                   /**< Interface protocol */
    4,                   /**< Index of the string */
};

/**
  * @brief Endpoint descriptor
  */
static const uint8_t __hid_in_ep[HID_IN_ENDPOINT_SIZE] =
{
    7,                                  /**< Size of the endpoint descriptor */
    USB_DTYPE_ENDPOINT,                 /**< Type is an endpoint */
    USB_EP_DESC_IN | USB_EP_1,          /**< Endpoint direction */
    USB_EP_ATTR_INT,                    /**< Endpoint type */
    USBShort(64),                       /**< Maximum packet size */
    10,                                 /**< Polling interval */
};

static const uint8_t __hid_out_ep[HID_OUT_ENDPOINT_SIZE] =
{
    7,                                  /**< Size of the endpoint descriptor */
    USB_DTYPE_ENDPOINT,                 /**< Type is an endpoint */
    USB_EP_DESC_OUT | USB_EP_2,          /**< Endpoint direction */
    USB_EP_ATTR_INT,                    /**< Endpoint type */
    USBShort(64),                       /**< Maximum packet size */
    10,                                 /**< Polling interval */
};

/**
  * @brief HID report structure
  */
static const uint8_t __hid_user_report_desc[] =
{
    0x06,
    ((0xFF00) & 0xFF),
    (((0xFF00) >> 8) & 0xFF),
    USAGE(USB_HID_POINTER),
    COLLECTION(USB_HID_APPLICATION),

    USAGE(2),
    LOGICAL_MIN(0),
    LOGICAL_MAX(256),
    REPORT_SIZE(8),
    REPORT_COUNT(64),
    OUTPUT(0x82),

    USAGE(3),
    LOGICAL_MIN(0),
    LOGICAL_MAX(256),
    REPORT_SIZE(8),
    REPORT_COUNT(64),
    INPUT(0x82),

    END_COLLECTION
};

/**
  * @brief HID descriptor
  */
static hid_desc_t __boot_hid_desc =
{
    9,                                             /**< Length of the descriptor */
    USB_HID_DTYPE_HID,                             /**< Type of the descriptor */
    0x111,                                         /**< Version 1.11 in BCD code */
    0,                                             /**< Country code */
    1,                                             /**< Number of descriptor */
    {
        {
            USB_HID_DTYPE_REPORT,                   /**< Report descriptor */
            sizeof(__hid_user_report_desc)          /**< Size of the report descriptor */
        }
    }
};

/**
  * @brief HID configuration section
  */
static const config_section_t __hid_config_sec =
{
    sizeof(__hid_user_desc),
    __hid_user_desc
};

/**
  * @brief HID interface section
  */
static const config_section_t hid_interface_sec =
{
    sizeof(__hid_interface),
    __hid_interface
};

/**
  * @brief HID IN endpoint section
  */
static const config_section_t __hid_in_ep_sec =
{
    sizeof(__hid_in_ep),
    __hid_in_ep
};

/**
  * @brief HID OUT endpoint section
  */
static const config_section_t __hid_out_ep_sec =
{
    sizeof(__hid_out_ep),
    __hid_out_ep
};

/**
  * @brief HID IN endpoint section
  */
static config_section_t __hid_desc_sec =
{
    sizeof(__boot_hid_desc),
    (const uint8_t *) &__boot_hid_desc
};

/**
  * @brief HID section
  */
static const config_section_t *__hid_sec[] =
{
    &__hid_config_sec,
    &hid_interface_sec,
    &__hid_desc_sec,
    &__hid_in_ep_sec,
    &__hid_out_ep_sec,
};

/**
  * @brief Number of the section
  */
#define NUM_HID_SECTIONS    (sizeof(__hid_sec) / sizeof(config_section_t *))

/**
  * @brief HID configuration header
  */
static config_head_t __hid_config_head =
{
    NUM_HID_SECTIONS,
    __hid_sec
};

/**
  * @brief HID configuration descriptor
  */
static const config_head_t *const __hid_config_desc[] =
{
    &__hid_config_head
};

/**
  * @brief HID Boot configuration descriptor
  */
static const uint8_t *__hid_user_class_desc[] =
{
    __hid_user_report_desc
};
/**
  * @}
  */

/** @defgroup Device_Hid_User_Private_Functions Private Functions
  * @{
  */
/**
  * @brief  Handle HID device class receive event.
  * @param  device: HID user device.
  * @param  event:  Event.
  * @param  param:  Event-specific value.
  * @param  data:   Event-specific data.
  * @retval Status.
  */
static uint32_t hid_user_rx_handler(void *device, uint32_t event, uint32_t param, void *data)
{
    hid_user_inst_t *inst;
    usbd_hid_user_dev_t *dev;

    assert_param(device);

    /* Create a pointer to the device instance */
    dev  = (usbd_hid_user_dev_t *)device;
    inst = &dev->inst;

    /* Which event? */
    switch (event)
    {
        case USB_EVENT_CONNECTED:
            inst->config = 1;
            /* Send a CONNECTED event to the client */
            dev->cbk(dev->arg, USB_EVENT_CONNECTED, 0, NULL);
            break;

        case USB_EVENT_DISCONNECTED:
            inst->config = 0;
            /* Send a DISCONNECTED event to the client */
            dev->cbk(dev->arg, USB_EVENT_DISCONNECTED, 0, NULL);
            break;

        case USB_EVENT_RX_AVAILABLE:
            dev->cbk(dev->arg, USB_EVENT_RX_AVAILABLE, 0, NULL);
            break;

        case USBD_HID_EVENT_IDLE_TIMEOUT:
        case USBD_HID_EVENT_GET_REPORT:
            /* Set the report pointer */
            *(uint8_t **)data = inst->txReport;
            /* Return the size of the report */
            return 8;

        case USBD_HID_EVENT_REPORT_SENT:
            break;

        case USBD_HID_EVENT_GET_REPORT_BUFFER:
            return 0;

        case USBD_HID_EVENT_SET_REPORT:
            break;

        case USBD_HID_EVENT_SET_PROTOCOL:
            /* Set the report protocol */
            inst->protocol = param;
            break;

        case USBD_HID_EVENT_GET_PROTOCOL:
            /* Get the report protocol */
            return inst->protocol;

        case USB_EVENT_ERROR:
        case USB_EVENT_SUSPEND:
        case USB_EVENT_RESUME:
        case USB_EVENT_LPM_RESUME:
        case USB_EVENT_LPM_SLEEP:
        case USB_EVENT_LPM_ERROR:
            return dev->cbk(dev->arg, event, param, data);

        default:
            break;
    }

    return 0;
}

/**
  * @brief  Handle HID device class transmit event.
  * @param  device: HID User device.
  * @param  event:  Event.
  * @param  param:  Event-specific value.
  * @param  data:   Event-specific data.
  * @retval Status.
  */
static uint32_t hid_user_tx_handler(void *device, uint32_t event, uint32_t param, void *data)
{
    hid_user_inst_t *inst;
    usbd_hid_user_dev_t *dev;

    assert_param(device);

    /* Create a pointer to the device instance */
    dev  = (usbd_hid_user_dev_t *)device;
    inst = &dev->inst;

    /* Which event? */
    switch (event)
    {
        case USB_EVENT_TX_COMPLETE:
            inst->state = HID_USER_STATE_IDLE;
            /* Send the TX_COMPLETE event to the client */
            dev->cbk(dev->arg, USB_EVENT_TX_COMPLETE, param, NULL);
            break;

        default:
            break;
    }

    return 0;
}

/** @defgroup Device_HID_User_Public_Functions Public Functions
  * @{
  */
/**
  * @brief  Initializes HID User device.
  * @param  idx: Index of the USB controller.
  * @param  dev: HID User device.
  * @retval Device structure.
  */
void *usbd_hid_user_init(uint32_t idx, usbd_hid_user_dev_t *dev)
{
    void *ret;
    usbd_hid_dev_t *_dev;
    config_desc_t *desc;

    assert_param(dev);
    assert_param(dev->desc_str);
    assert_param(dev->cbk);

    /* Create a pointer to the device */
    _dev = &dev->inst.dev;
    /* Composite initialization */
    ret  = usbd_hid_user_init_comp(idx, dev, 0);
    /* Fill the configuration descriptor */
    desc = (config_desc_t *)__hid_user_desc;
    desc->bmAttributes = dev->attr_pwr;
    desc->bMaxPower = (uint8_t)(dev->max_power >> 1);

    if (ret)
    {
        /* Initialze device controler */
        usbd_hid_init(idx, _dev);
        return (void *)dev;
    }

    return NULL;
}

/**
  * @brief  Initializes HID User device.
  * @param  idx: Index of the USB controller.
  * @param  dev: HID User device.
  * @param  entry: Composite entry.
  * @retval Device structure.
  */
void *usbd_hid_user_init_comp(uint32_t idx, usbd_hid_user_dev_t *dev, comp_entry_t *entry)
{
    hid_user_inst_t *inst;
    usbd_hid_dev_t *_dev;

    assert_param(dev);
    assert_param(dev->desc_str);
    assert_param(dev->cbk);

    /* Create a pointer to the device instance */
    inst = &dev->inst;
    /* Initialize the device instance information */
    inst->config      = 0;
    inst->protocol    = USB_HID_PROTOCOL_REPORT;
    inst->idle.dur4ms = 125;
    inst->idle.id     = 0;
    inst->idle.since  = 0;
    inst->idle.next   = 0;
    inst->state       = HID_USER_STATE_UNCONFIG;

    /* Initialize the device information */
    _dev = &inst->dev;
    _dev->vid             = dev->vid;
    _dev->pid             = dev->pid;
    _dev->max_power       = dev->max_power;
    _dev->attr_pwr        = dev->attr_pwr;
    _dev->sub_class       = USB_HID_SCLASS_NONE;
    _dev->protocol        = USB_HID_PROTOCOL_NONE;
    _dev->nr_input_report = 1;
    _dev->idle            = &inst->idle;
    _dev->rx_cbk          = hid_user_rx_handler;
    _dev->rx_arg          = (void *)dev;
    _dev->tx_cbk          = hid_user_tx_handler;
    _dev->tx_arg          = (void *)dev;
    _dev->use_oep         = 0,
          _dev->desc_hid        = &__boot_hid_desc;
    _dev->desc_class      = __hid_user_class_desc;
    _dev->desc_str        = dev->desc_str;
    _dev->num_str         = dev->num_str;
    _dev->desc_config     = __hid_config_desc;

    /* Call the lower layer composite initialization */
    return usbd_hid_init_comp(idx, _dev, entry);
}

/**
  * @brief  Terminal the HID keyboard device.
  * @param  device: HID User device.
  * @retval None
  */
void usbd_hid_user_term(void *device)
{
    usbd_hid_user_dev_t *dev;
    usbd_hid_dev_t *_dev;

    assert_param(device);

    /* Create a pointer to the device */
    dev  = (usbd_hid_user_dev_t *)device;
    _dev = &dev->inst.dev;
    /* Clear the flag */
    dev->inst.config = 0;

    /* Call the lower layer function */
    usbd_hid_term(_dev);
    return;
}

/**
  * @brief  Schedule a report to be sent.
  * @param  dev:    HID User device.
  * @param  report: Data to send to the host.
  * @param  size:   Size of the buffer.
  * @retval Status.
  */
uint32_t usbd_hid_user_report_send(usbd_hid_user_dev_t *dev, void *report, uint32_t size)
{
    uint32_t cnt;
    hid_user_inst_t *inst;
    usbd_hid_dev_t *_dev;

    /* Create a pointer to the device instance */
    inst = &dev->inst;
    /* Create a pointer to the device */
    _dev = &dev->inst.dev;

    /* Check the state */
    if (!inst->config)
        return  CUSTOM_ERR_NOT_CONFIGURED;

    /* Check the TX_FIFO */
    if ((usbd_hid_tx_packet_avail((void *)_dev)) == 0)
        return CUSTOM_ERR_TX_ERROR;

    /* Set the state */
    inst->state = HID_USER_STATE_SEND;
    /* Send a report to the host */
    cnt = usbd_hid_report_write((void *)_dev, report, size, 1);

    return cnt ? CUSTOM_SUCCESS : CUSTOM_ERR_TX_ERROR;
}

/**
  * @brief  Schedule a report to be recv.
  * @param  dev:    HID User device.
  * @param  report: Data to recv from the host.
  * @param  size:   Size of the buffer.
  * @retval Status.
  */
uint32_t usbd_hid_user_report_recv(usbd_hid_user_dev_t *dev, void *report, uint32_t size)
{
    uint32_t cnt;
    hid_user_inst_t *inst;
    usbd_hid_dev_t *_dev;

    /* Create a pointer to the device instance */
    inst = &dev->inst;
    /* Create a pointer to the device */
    _dev = &dev->inst.dev;

    /* Check the state */
    if (!inst->config)
        return  CUSTOM_ERR_NOT_CONFIGURED;

    /* Check the TX_FIFO */
    if ((usbd_hid_rx_packet_avail((void *)_dev)) == 0)
        return CUSTOM_ERR_RX_ERROR;

    /* Set the state */
    inst->state = HID_USER_STATE_WAIT_DATA;
    /* Read a report to the host */
    cnt = usbd_hid_packet_read((void *)_dev, report, size, 1);

    return cnt ? CUSTOM_SUCCESS : CUSTOM_ERR_RX_ERROR;
}

/**
  * @}
  */

/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
