/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */
uint8_t  g_tx_buf[512]; 
uint8_t  g_rx_buf[10];

uint16_t g_tx_i = 0U;
uint8_t  g_rx_i = 0U;

uint32_t g_data_addr;
uint8_t  g_data_len;
uint8_t  g_frame_flag = 0U;

/* Private Constants --------------------------------------------------------- */


/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */


/**
  * @brief  Initializate pin of usart module.
  * @retval None
  */
static void init_usart_pin(void)
{
    md_gpio_init_t gpio_config;
    md_gpio_init_struct(&gpio_config);

    /* Initialize tx pin */
    gpio_config.mode = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos = MD_GPIO_PUSH_PULL;
    gpio_config.pupd = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_config.type = MD_GPIO_TYPE_CMOS;
    gpio_config.func = MD_GPIO_FUNC_3;
    md_gpio_init(USART0_TX_PORT, USART0_TX_PIN, &gpio_config);

    /* Initialize rx pin */
    gpio_config.mode = MD_GPIO_MODE_INPUT;
    gpio_config.odos = MD_GPIO_PUSH_PULL;
    gpio_config.pupd = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_config.type = MD_GPIO_TYPE_CMOS;
    gpio_config.func = MD_GPIO_FUNC_3;
    md_gpio_init(USART0_RX_PORT, USART0_RX_PIN, &gpio_config);

    return;
}

/**
  * @brief:  Initialize the usart.
  * @param:  None
  * @retval: None
  */
static void init_mcu_usart(void)
{
    md_usart_init_t usart_config;
    md_usart_init_struct(&usart_config);
    /* Initialize USART0 */
    usart_config.baud        = 9600;
    usart_config.word_length = MD_USART_WORD_LENGTH_9B;
    usart_config.stop_bits   = MD_USART_STOP_BITS_1;
    usart_config.parity      = MD_USART_PARITY_EVEN;
    usart_config.fctl        = MD_USART_HW_FLOW_CTL_NONE;
    usart_config.mode        = MD_USART_MODE_TX_RX;
    md_usart_init(USART0, &usart_config);

    md_usart_clear_flag_txc(USART0);
    md_usart_enable_it_txc(USART0);
    md_usart_enable_it_rxne(USART0);
}

uint8_t ask_data_handle(void)
{
    if (g_rx_buf[0] != STA1 || g_rx_buf[1] != STA2 || g_rx_buf[2] != FRAME_LEN || g_rx_buf[8] != END1 || g_rx_buf[9] != END2)
    {
        return 0;
    }

    g_data_addr = (g_rx_buf[3] << 24) | (g_rx_buf[4] << 16) | (g_rx_buf[5] << 8) | (g_rx_buf[6]);
    g_data_len = g_rx_buf[7];

    return 1;
}

void answer_data_send(void)
{
    uint16_t i;
    uint8_t data_buf[512];

    memcpy(data_buf, (void *)g_data_addr, g_data_len);

    g_tx_buf[0] = STA1;
    g_tx_buf[1] = STA2;
    g_tx_buf[2] = FRAME_LEN + g_data_len;
    g_tx_buf[3] = g_rx_buf[3];
    g_tx_buf[4] = g_rx_buf[4];
    g_tx_buf[5] = g_rx_buf[5];
    g_tx_buf[6] = g_rx_buf[6];
    g_tx_buf[7] = g_data_len;

    for (i = 0; i < g_data_len; i++)
    {
        g_tx_buf[8 + i] = data_buf[i];
    }

    g_tx_buf[8 + g_data_len] = END1;
    g_tx_buf[9 + g_data_len] = END2;

    md_usart_send_data8(USART0, g_tx_buf[0]);
}

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main()
{
    /* Initialize the md library and the library clock as 1ms interrupt */
    md_init_1ms_tick();
    md_mcu_irq_config(USART0_IRQn, 0, ENABLE);

    /* Init USART */
    init_usart_pin();
    init_mcu_usart();

    while (1)
    {
        if (g_frame_flag == 1)
        {
            if (ask_data_handle() == 1)
            {
                answer_data_send();
            }

            g_frame_flag = 0;
            memset(g_rx_buf, 0, 10);
            g_tx_i = 0;
        }
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
