/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */
static md_dma_descriptor_t s_dma0_ctrl_base[2]__attribute__((aligned(512)));

/* Public Variables ---------------------------------------------------------- */
md_usart_init_t g_usart_handle;
md_dma_config_t g_tx_config, g_rx_config;

uint8_t g_rx_buf[10];

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */
/**
  * @brief  Initializate pin of uart module.
  * @retval None
  */
static void init_usart_pin(void)
{
    md_gpio_init_t gpio_config;
    md_gpio_init_struct(&gpio_config);

    /* Initialize tx pin */
    gpio_config.mode = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos = MD_GPIO_PUSH_PULL;
    gpio_config.pupd = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_config.type = MD_GPIO_TYPE_CMOS;
    gpio_config.func = MD_GPIO_FUNC_3;
    md_gpio_init(USART0_TX_PORT, USART0_TX_PIN, &gpio_config);

    /* Initialize rx pin */
    gpio_config.mode = MD_GPIO_MODE_INPUT;
    gpio_config.odos = MD_GPIO_PUSH_PULL;
    gpio_config.pupd = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_config.type = MD_GPIO_TYPE_CMOS;
    gpio_config.func = MD_GPIO_FUNC_3;
    md_gpio_init(USART0_RX_PORT, USART0_RX_PIN, &gpio_config);

    return;
}

/**
  * @brief:  Initialize the IWDT.
  * @param:  None
  * @retval: None
  */
static void init_mcu_usart(void)
{
    md_usart_init_struct(&g_usart_handle);
    /* Initialize USART0 */
    g_usart_handle.baud        = 9600;
    g_usart_handle.word_length = MD_USART_WORD_LENGTH_8B;
    g_usart_handle.stop_bits   = MD_USART_STOP_BITS_1;
    g_usart_handle.parity      = MD_USART_PARITY_NONE;
    g_usart_handle.fctl        = MD_USART_HW_FLOW_CTL_NONE;
    g_usart_handle.mode        = MD_USART_MODE_TX_RX;
    md_usart_init(USART0, &g_usart_handle);
    return;
}

/**
  * @brief  dma_start_tx init.
  * @param  None
  * @retval None
  */
void config_dma_tx(USART_TypeDef *USARTx, uint8_t *buf, uint16_t size, uint8_t channel)
{
    memset(&g_tx_config, 0x0, sizeof(g_tx_config));

    /*Initialize DMA*/
    md_dma_enable();
    md_dma_config_struct(&g_tx_config);
    g_tx_config.src     = (void *)(buf);
    g_tx_config.dst     = (void *)(&USARTx->DATA);
    g_tx_config.size    = size;
    g_tx_config.src_inc    = MD_DMA_DATA_INC_BYTE;
    g_tx_config.dst_inc    = MD_DMA_DATA_INC_NONE;
    g_tx_config.R_power    = MD_DMA_R_POWER_1;
    g_tx_config.msel       = USARTx == USART0 ? MD_DMA_MSEL_USART0 : MD_DMA_MSEL_USART1;
    g_tx_config.msigsel    = MD_DMA_MSIGSEL_USART_TXEMPTY;
    g_tx_config.channel    = channel;

    if ((g_usart_handle.word_length == MD_USART_WORD_LENGTH_9B) && (g_usart_handle.parity == MD_USART_PARITY_NONE))
    {
        g_tx_config.src_inc    = MD_DMA_DATA_INC_HALFWORD;
        g_tx_config.data_width = MD_DMA_DATA_SIZE_HALFWORD;
    }
}

/**
  * @brief  dma_recv_rx init.
  * @param  None
  * @retval None
  */
void config_dma_rx(USART_TypeDef *USARTx, uint8_t *buf, uint16_t size, uint8_t channel)
{
    memset(&g_rx_config, 0x0, sizeof(g_rx_config));

    /*Initialize DMA*/
    md_dma_enable();
    md_dma_config_struct(&g_rx_config);
    g_rx_config.src     = (void *)(&USARTx->DATA);
    g_rx_config.dst     = (void *)(buf);
    g_rx_config.size    = size;
    g_rx_config.src_inc    = MD_DMA_DATA_INC_NONE;
    g_rx_config.dst_inc    = MD_DMA_DATA_INC_BYTE;
    g_rx_config.R_power    = MD_DMA_R_POWER_1;
    g_rx_config.msel       = USARTx == USART0 ? MD_DMA_MSEL_USART0 : MD_DMA_MSEL_USART1;
    g_rx_config.msigsel    = MD_DMA_MSIGSEL_USART_RNR;
    g_rx_config.channel    = channel;

    if ((g_usart_handle.word_length == MD_USART_WORD_LENGTH_9B) && (g_usart_handle.parity == MD_USART_PARITY_NONE))
    {
        g_rx_config.src_inc    = MD_DMA_DATA_INC_HALFWORD;
        g_rx_config.data_width = MD_DMA_DATA_SIZE_HALFWORD;
    }
}

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main()
{
    /*Configure the system clock*/
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HRC_6, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);

    /*Initialize the SysTick interrupt*/
    md_init_1ms_tick();

    /*I2C clock can make*/
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    /* Init USART */
    init_usart_pin();
    init_mcu_usart();

    md_mcu_irq_config(DMA_IRQn, 2, ENABLE);
    memset(s_dma0_ctrl_base, 0x0, sizeof(s_dma0_ctrl_base));
    md_dma_set_ctrlbase((uint32_t)&s_dma0_ctrl_base);
    /* dma receive config */
    config_dma_rx(USART0, g_rx_buf, sizeof(g_rx_buf), 1);
    md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_BASIC, &g_rx_config);
    /* dma send config */
    config_dma_tx(USART0, g_rx_buf, sizeof(g_rx_buf), 0);

    /* Enable DMA channel */
    md_dma_clear_flag_done(MD_DMA_CH_1);
    md_dma_enable_channel(MD_DMA_CH_1);
    md_usart_enable_dma_req_rx(USART0);

    while (1)
    {
    }
}

/**
  * @}
  */
/**
  * @}
  */


/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
