/**********************************************************************************
 *
 * @file    smart_card.c
 * @brief   smart_card c File.
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/** @addtogroup USART_Application
  * @{
  */

/** @defgroup Smart_card Smart_card
  * @brief usart application demo for smart card
  * @{
  */

/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include "md_conf.h"
#include "smart_card.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  get string of data
  * @param  data_addr: data first address
  * @param  len: data lenth
  * @arg
  * @retval result
  */
uint8_t get_card_byte(uint8_t *data_addr, uint8_t len)
{
    uint32_t i, out_time, sys_clk_div1000;
    sys_clk_div1000 = md_cmu_get_clock() / 1000;

    while (1)
    {
        for (i = 0; i < len; i++)
        {
            for (out_time = 0x00; out_time < sys_clk_div1000; out_time++)
            {
                if (md_usart_is_active_flag_rxne(USART0))
                {
                    break;
                }
            }

            if (out_time == sys_clk_div1000)
                return CARD_ERROR + i;

            if (md_usart_is_active_flag_perr(USART0))
                return CARD_ERROR + i;

            data_addr[i] =  md_usart_recv_data8(USART0);
        }

        return CARD_OK;
    }
}

/**
  * @brief  send string of data
  * @param  data_addr: data first address
  * @param  len: data lenth
  * @arg
  * @retval result
  */
uint8_t send_card_byte(uint8_t *data_addr, uint16_t len)
{
    uint32_t i, out_time, sys_clk_div1000;
    sys_clk_div1000 = md_cmu_get_clock() / 1000;

    while (1)
    {
        for (i = 0; i < len; i++)
        {
            md_usart_send_data8(USART0, data_addr[i]);

            for (out_time = 0x00; out_time < sys_clk_div1000; out_time++)
            {
                if (md_usart_is_active_flag_txc(USART0))
                {
                    break;
                }
            }

            md_usart_clear_flag_txc(USART0);

            if (out_time == sys_clk_div1000)
                return CARD_ERROR + i;

            if (md_usart_is_active_flag_perr(USART0))
                return CARD_ERROR + i;
        }

        return CARD_OK;
    }
}

/**
  * @brief:  cold reset smart card and receive rseponse.
  * @param:  rxbuf: receive buffer address.
  * @retval: None
  */
uint8_t reset_rec(uint8_t *rxbuf)
{
    CARD_RST_L;
    md_delay_1ms(1);
    CARD_RST_H;

    if (get_card_byte(rxbuf, 2) == CARD_OK)
    {
        if ((rxbuf[0] != 0x3B) || (rxbuf[1] != 0x69))
        {
            return CARD_ERROR;
        }

        /*Ignore the reset reply*/
        while (get_card_byte(rxbuf, 31) == CARD_OK);

        return CARD_OK;
    }
    else
    {
        return CARD_ERROR;
    }
}

/**
  * @brief:  cold test smart card and receive rseponse.
  * @param:  rxbuf: receive buffer address.
  * @retval: None
  */
uint8_t instruction_test(void)
{
    uint8_t result;

    /*Instructions for the common communication, only communication*/
    g_tx_buf[0] = 0x00;
    g_tx_buf[1] = 0xA4;
    g_tx_buf[2] = 0x00;
    g_tx_buf[3] = 0x00;
    g_tx_buf[4] = 0x02;
    g_tx_buf[5] = 0x3F;
    g_tx_buf[6] = 0x00;

    /*Waiting for the ESAM free*/
    while (get_card_byte(g_rx_buf, 31) == CARD_OK);

    /*Before sending, close to receive*/
    md_usart_disable_rx(USART0);

    /*Send command head (5) to ESAM g_tx_buf [0] - g_tx_buf [4]*/
    /*Instructions category CLA code INS instruction reference symbol number of bytes of data (P1, P2, P3*/
    result = send_card_byte(g_tx_buf, 5);

    /*After sending, open to receive*/
    md_usart_enable_rx(USART0);

    /*Whether to send and receive is normal*/
    if (result == CARD_ERROR)
        return CARD_ERROR;

    /*One byte process*/
    result = get_card_byte(g_rx_buf, 1);

    if (result != CARD_OK)
        return CARD_ERROR;

    /*If the return process of instruction code byte = = INS*/
    if (g_rx_buf[0] == g_tx_buf[1])
    {
        md_usart_disable_rx(USART0);

        /*Send the rest of the data bytes, number of bytes of data = number of P3 = g_tx_buf [4]*/
        result = send_card_byte(g_tx_buf + 5, g_tx_buf[4]);

        md_usart_enable_rx(USART0);

        if (result != CARD_OK)
            return CARD_ERROR;

        /*Receive 2 status byte*/
        result = get_card_byte(g_rx_buf, 2);

        if (result != CARD_OK)
            return CARD_ERROR;

        /*0 x61xx and 0 x9000 are normal processing*/
        if (g_rx_buf[0] == 0x61)
        {
            return CARD_OK;
        }

        if (g_rx_buf[0] == 0x90 && g_rx_buf[1] == 0x00)
        {
            return CARD_OK;
        }

    }

    return CARD_ERROR;

}


/**
  * @brief:  cold test smart card and receive rseponse.
  * @param:  rxbuf: receive buffer address.
  * @retval: None
  */
uint8_t rand_instruction_test(void)
{
    uint8_t result, different_times = 0U;
    uint32_t last_rand, temp_rand = 0U;

    /*Instructions for the generated four random 8 bits of data*/
    g_tx_buf[0] = 0x00;
    g_tx_buf[1] = 0x84;
    g_tx_buf[2] = 0x00;
    g_tx_buf[3] = 0x00;
    g_tx_buf[4] = 0x04;

    last_rand = temp_rand;

    while (different_times < 5)
    {
        /*Waiting for the ESAM free*/
        while (get_card_byte(g_rx_buf, 31) == CARD_OK);

        /*Before sending, close to receive*/
        md_usart_disable_rx(USART0);

        /*Send command head (5) to ESAM g_tx_buf [0] - g_tx_buf [4]*/
        /*Instructions category CLA code INS instruction reference symbol number of bytes of data (P1, P2, P3*/
        result = send_card_byte(g_tx_buf, 5);

        /*After sending, open to receive*/
        md_usart_enable_rx(USART0);

        /*Whether to send and receive is normal*/
        if (result == CARD_ERROR)
            return CARD_ERROR;

        /*Process 1 byte + 0 (the data didn't send the instruction) + P3 state of a random number + two bytes*/
        result = get_card_byte(g_rx_buf, 1 + g_tx_buf[4] + 2);

        if (result != CARD_OK)
            return CARD_ERROR;

        /*Judge status, 0 x90 0 x00 = command execution success*/
        if (g_rx_buf[1 + g_tx_buf[4]] != 0x90 || g_rx_buf[1 + g_tx_buf[4] + 1] != 0x00)
        {
            return CARD_ERROR;
        }

        /*Will be credited to temp_rand 4 8 bits of random number*/
        memcpy(&temp_rand, g_rx_buf + 1, g_tx_buf[4]);

        /*To determine whether a random number change*/
        if (temp_rand != last_rand)
        {
            last_rand = temp_rand;
            different_times++;
        }
    }

    return CARD_OK;

}


/**
  * @brief:  cold test smart card and receive rseponse.
  * @param:  rxbuf: receive buffer address.
  * @retval: None
  */
uint8_t name_instruction_test(void)
{
    uint8_t result;

    /*Instructions for the serial number of the ESAM*/
    g_tx_buf[0] = 0x00;
    g_tx_buf[1] = 0xB0;
    g_tx_buf[2] = 0x99;
    g_tx_buf[3] = 0x01;
    g_tx_buf[4] = 0x08;

    /*Waiting for the ESAM free*/
    while (get_card_byte(g_rx_buf, 31) == CARD_OK);

    /*Before sending, close to receive*/
    md_usart_disable_rx(USART0);

    /*Send command head (5) to ESAM g_tx_buf [0] - g_tx_buf [4]*/
    /*Instructions category CLA code INS instruction reference symbol number of bytes of data (P1, P2, P3*/
    result = send_card_byte(g_tx_buf, 5);

    /*After sending, open to receive*/
    md_usart_enable_rx(USART0);

    /*Whether to send and receive is normal*/
    if (result == CARD_ERROR)
        return CARD_ERROR;

    /*Process 1 byte + 0 (the data didn't send the instruction) + P3 a serial number + 2 a status byte*/
    result = get_card_byte(g_rx_buf, 1 + g_tx_buf[4] + 2);

    if (result != CARD_OK)
        return CARD_ERROR;

    /*Judge status, 0 x90 0 x00 = command execution success*/
    if (g_rx_buf[1 + g_tx_buf[4]] != 0x90 || g_rx_buf[1 + g_tx_buf[4] + 1] != 0x00)
    {
        return CARD_ERROR;
    }

    return CARD_OK;
}

/**
  * @}Smart_card
  */

/**
  * @}Micro_Driver
  */
/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
