/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for demo
 *
 * @date    23 Sep 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Sep 2021     biyq          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */
uint8_t g_tx_buf[8] = {0x88U, 0x77U, 0x66U, 0x55U, 0x44U, 0x33U, 0x22U, 0x11U};
uint8_t g_rx_buf[8];
uint8_t g_rx_index;

/* Public Variables ---------------------------------------------------------- */
lin_node_t g_node1;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */
void uart_send_byte(uint8_t u8data);
void uart_recv_ctrl(lin_node_t *lin_node);
void lin_breakfield_send(void);
void lin_select_normal_slope(void);
void lin_select_low_slope(void);
void lin_exit_sleep(void);
void lin_enter_sleep(void);

/* Public Function ---------------------------------------------------------- */

/* Private Function ---------------------------------------------------------- */
/**
  * @brief  Init LIN pin
  * @retval None
  */
void lin_pin_init(void)
{
    md_gpio_init_t x;
    md_gpio_init_struct(&x);

    /* Initialize tx pin: PB10 */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.odos  = MD_GPIO_OPEN_DRAIN;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_CMOS;
    x.func  = MD_GPIO_FUNC_3;
    md_gpio_init(LIN0_PORT, LIN0_TX_PIN, &x);

    /* Initialize rx pin: PB11 */
    x.mode  = MD_GPIO_MODE_INPUT;
    md_gpio_init(LIN0_PORT, LIN0_RX_PIN, &x);

    /* Initialize inh pin: PB6 */
    x.mode  = MD_GPIO_MODE_INPUT;
    x.func  = MD_GPIO_FUNC_1;
    md_gpio_init(LIN0_PORT, LIN0_INH_PIN, &x);

    /* Initialize slp pin: PB7 */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.func  = MD_GPIO_FUNC_1;
    md_gpio_init(LIN0_PORT, LIN0_SLP_PIN, &x);
}

/**
  * @brief:  Initialize the lin.
  * @param:  None
  * @retval: None
  */
void lin_init(void)
{
    md_uart_init_t h_uart;
    md_uart_init_struct(&h_uart);

    /* Initialize LIN1 */
    h_uart.baud        = 9600;
    h_uart.word_length = MD_UART_WORD_LENGTH_8B;
    h_uart.stop_bits   = MD_UART_STOP_BITS_1;
    h_uart.parity      = MD_UART_PARITY_NONE;
    h_uart.fctl        = MD_UART_FLOW_CTL_DISABLE;
    h_uart.mode        = MD_UART_MODE_LIN;
    md_uart_init(UART0, &h_uart);

    /* set fifo threshold */
    md_uart_set_tx_threshold(UART0, MD_UART_TXFIFO_EMPTY);
    md_uart_set_rx_threshold(UART0, MD_UART_RXFIFO_1BYTE);

    /* enable lin break irq and rxth */
    md_uart_enable_it_linbk(UART0);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Configure system clock */
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HOSC_3, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();
    /* 2bits preempt-priority(0-3)
       2bits sub-priority(0-3). */
    NVIC_SetPriorityGrouping(5);

    /* Enable UART0 irq */
    md_mcu_irq_config(UART0_IRQn, 2, 2, ENABLE);

    /* Init Lin node */
    memset(&g_node1, 0x00, sizeof(lin_node_t));
    g_node1.id = 0x07;
    g_node1.sync_field = SYNC_SEG;
    g_node1.node_mode = MASTER;
    g_node1.node_send_bk_req = lin_breakfield_send;
    g_node1.node_send_byte = uart_send_byte;
    g_node1.node_recv_ctrl = uart_recv_ctrl;
    g_node1.node_normal_slope_mode = lin_select_normal_slope;
    g_node1.node_low_slope_mode = lin_select_low_slope;
    g_node1.node_enter_sleep = lin_enter_sleep;
    g_node1.node_exit_sleep = lin_exit_sleep;

    /* Init Lin config */
    lin_pin_init();
    lin_init();
    lin_chstat(&g_node1, NORMAL_SLOPE);   /* Due to the fact that tx pin has been pushed up, so we can only select normal slope */

    /* lin send frame */
    lin_sendframe(&g_node1, g_tx_buf, 8);
    md_delay_1ms(8000);
    lin_recvframe(&g_node1, 8);

    while (1);
}

/**
  * @brief  uart send data
  * @retval None
  */
void uart_send_byte(uint8_t u8data)
{
    uint16_t cnt;

    md_uart_set_send_data8(UART0, u8data);

    cnt = 4000;

    while (!md_uart_is_active_it_tbc(UART0) && (--cnt));

    md_uart_clear_it_tbc(UART0);

    return;
}

/**
  * @brief  uart send data
  * @retval None
  */
void uart_recv_ctrl(lin_node_t *lin_node)
{
    if (lin_node->node_mode == MASTER && lin_node->dir == L_READ)
        md_uart_enable_it_rfth(UART0);
    else if (lin_node->node_mode == SLAVE && lin_node->dir == L_WRITE)
        md_uart_enable_it_rfth(UART0);
    else
        md_uart_disable_it_rfth(UART0);

    return;
}

/**
  * @brief  send break field
  * @retval None
  */
void lin_breakfield_send(void)
{
    md_uart_enable_lin_bkreq(UART0);

    return;
}

/**
  * @brief  use normal slope mode
  * @retval None
  */
void lin_exit_sleep(void)
{
    md_gpio_write_pin(LIN0_PORT, LIN0_SLP_PIN, 1);

    return;
}

/**
  * @brief  use normal slope mode
  * @retval None
  */
void lin_enter_sleep(void)
{
    md_gpio_write_pin(LIN0_PORT, LIN0_SLP_PIN, 0);

    return;
}

/**
  * @brief  use normal slope mode
  * @retval None
  */
void lin_select_normal_slope(void)
{

    return;
}

/**
  * @brief  use low slope mode
  * @retval None
  */

void lin_select_low_slope(void)
{

    return;
}
/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
