/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for demo
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

md_dma_descriptor_t g_dma0_ctrl_base[2] __attribute__((aligned(512)));
uint8_t g_rx_buf[32];
md_dma_config_t g_dma_tx_config, g_dma_rx_config;
/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Init UART pin
  * @retval None
  */
void init_uart_pin(void)
{
    md_gpio_init_t gpio_init;
    md_gpio_init_struct(&gpio_init);

    /* Initialize tx pin: PB10 */
    gpio_init.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.podrv = MD_GPIO_OUT_DRIVE_1;
    gpio_init.nodrv = MD_GPIO_OUT_DRIVE_1;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_3;
    md_gpio_init(UART1_TX_PORT, UART1_TX_PIN, &gpio_init);

    /* Initialize rx pin: PB11 */
    gpio_init.mode  = MD_GPIO_MODE_INPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.podrv = MD_GPIO_OUT_DRIVE_1;
    gpio_init.nodrv = MD_GPIO_OUT_DRIVE_1;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_3;
    md_gpio_init(UART1_RX_PORT, UART1_RX_PIN, &gpio_init);
}

/**
  * @brief:  Initialize the uart.
  * @param:  None
  * @retval: None
  */
void init_uart(void)
{
    md_uart_init_t uart_init;
    md_uart_init_struct(&uart_init);

    /* Initialize UART1 */
    uart_init.baud        = 9600;
    uart_init.word_length = MD_UART_WORD_LENGTH_8B;
    uart_init.stop_bits   = MD_UART_STOP_BITS_1;
    uart_init.parity      = MD_UART_PARITY_NONE;
    uart_init.fctl        = MD_UART_FLOW_CTL_DISABLE;
    uart_init.mode        = MD_UART_MODE;
    md_uart_init(UART1, &uart_init);
}

/**
  * @brief  dma_start_tx init.
  * @param  None
  * @retval None
  */
void dma_send_tx(void)
{
    memset(&g_dma_tx_config, 0x0, sizeof(md_dma_config_t));
    md_dma_set_ctrlbase((uint32_t)g_dma0_ctrl_base);
    md_dma_enable();

    md_dma_config_struct(&g_dma_tx_config);

    g_dma_tx_config.channel = 1;
    g_dma_tx_config.data_width = MD_DMA_DATA_SIZE_BYTE;
    g_dma_tx_config.primary = ENABLE;
    g_dma_tx_config.R_power = MD_DMA_R_POWER_1;
    g_dma_tx_config.dst     = (void *)&UART1->TXBUF;
    g_dma_tx_config.src     = (void *)(g_rx_buf);
    g_dma_tx_config.dst_inc = MD_DMA_DATA_INC_NONE;
    g_dma_tx_config.src_inc = MD_DMA_DATA_INC_BYTE;
    g_dma_tx_config.size    = 32;
    g_dma_tx_config.msel    = MD_DMA_MSEL_UART1;
    g_dma_tx_config.msigsel = MD_DMA_MSIGSEL_UART_TXEMPTY;
    g_dma_tx_config.burst   = ENABLE;
    g_dma_tx_config.interrupt = ENABLE;

    md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_BASIC, &g_dma_tx_config);
}

/**
  * @brief  dma_start_rx init.
  * @param  None
  * @retval None
  */
void dma_recv_rx(void)
{
    memset(&g_dma_rx_config, 0x0, sizeof(md_dma_config_t));
    md_dma_set_ctrlbase((uint32_t)g_dma0_ctrl_base);

    /* Enable DMA */
    md_dma_enable();

    g_dma_rx_config.data_width = MD_DMA_DATA_SIZE_BYTE;
    g_dma_rx_config.src_inc    = MD_DMA_DATA_INC_NONE;
    g_dma_rx_config.dst_inc    = MD_DMA_DATA_INC_BYTE;
    g_dma_rx_config.R_power    = MD_DMA_R_POWER_1;
    g_dma_rx_config.primary    = ENABLE;
    g_dma_rx_config.burst      = DISABLE;
    g_dma_rx_config.high_prio  = DISABLE;
    g_dma_rx_config.interrupt  = ENABLE;

    g_dma_rx_config.channel = 0;
    g_dma_rx_config.src     = (void *)(&UART1->RXBUF);
    g_dma_rx_config.dst     = (void *)(g_rx_buf);
    g_dma_rx_config.size    = 32;
    g_dma_rx_config.msel    = MD_DMA_MSEL_UART1;
    g_dma_rx_config.msigsel = MD_DMA_MSIGSEL_UART_RNR;
    md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_BASIC, &g_dma_rx_config);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Configure system clock */
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HOSC_3, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();
    /* 2bits preempt-priority(0-3)
       2bits sub-priority(0-3). */
    NVIC_SetPriorityGrouping(5);

    NVIC_SetPriority(DMA_IRQn, 2);
    NVIC_EnableIRQ(DMA_IRQn);

    init_uart_pin();
    init_uart();

    memset(g_dma0_ctrl_base, 0x0, sizeof(g_dma0_ctrl_base));
    /* send */
    dma_send_tx();
    /* receive */
    dma_recv_rx();

    md_dma_clear_flag_done(MD_DMA_CH_1);
    md_dma_clear_flag_done(MD_DMA_CH_0);
    md_dma_enable_channel(MD_DMA_CH_0);
    md_uart_enable_rxdma(UART1);

    while (1);
}


/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
