/**********************************************************************************
 *
 * @file    smart_card.c
 * @brief   smart_card c File.
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/** @addtogroup UART_Application
  * @{
  */

/** @defgroup Smart_card Smart_card
  * @brief usart application demo for smart card
  * @{
  */

/* Includes ------------------------------------------------------------------ */

#include "ald_gpio.h"
#include "ald_uart.h"
#include "ald_cmu.h"
#include "smart_card.h"
#include "string.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  get string of data
  * @param  data_addr: data first address
  * @param  len: data lenth
  * @arg
  * @retval result
  */
uint8_t get_card_byte(uint8_t *data_addr, uint8_t len)
{
    uint16_t cnt;
    uint32_t i, out_time, sys_clk_div1000;
    sys_clk_div1000 = ald_cmu_get_clock() / 1000;

    while (1)
    {
        for (i = 0; i < len; i++)
        {
            for (out_time = 0x00; out_time < sys_clk_div1000; out_time++)
            {
                if (READ_BIT(g_h_uart.perh->STAT, UART_STAT_RFTH_MSK) == UART_STAT_RFTH_MSK)
                    break;
            }

            if (out_time == sys_clk_div1000)
                return CARD_ERROR + i;

            if (READ_BIT(g_h_uart.perh->STAT, UART_STAT_PERR_MSK) == UART_STAT_PERR_MSK)
                return CARD_ERROR + i;

            cnt = 4000;

            while (!ald_uart_get_status(&g_h_uart, UART_STATUS_RFEMPTY) && (--cnt))
            {
                data_addr[i] = g_h_uart.perh->RXBUF;
            }
        }

        return CARD_OK;
    }
}

/**
  * @brief  send string of data
  * @param  data_addr: data first address
  * @param  len: data lenth
  * @arg
  * @retval result
  */
uint8_t send_card_byte(uint8_t *data_addr, uint16_t len)
{
    uint16_t cnt;
    uint32_t i, out_time, sys_clk_div1000;
    sys_clk_div1000 = ald_cmu_get_clock() / 1000;

    while (1)
    {
        for (i = 0; i < len; i++)
        {
            cnt = 4000;

            while (ald_uart_get_status(&g_h_uart, UART_STATUS_TFFULL) && (--cnt));

            g_h_uart.perh->TXBUF = data_addr[i];

            for (out_time = 0x00; out_time < sys_clk_div1000; out_time++)
            {
                cnt = 4000;

                while (ald_uart_get_status(&g_h_uart, UART_STATUS_TSBUSY) && (--cnt));

                break;
            }

            ald_uart_clear_flag_status(&g_h_uart, UART_IF_TBC);

            if (out_time == sys_clk_div1000)
                return CARD_ERROR + i;

            if (READ_BIT(g_h_uart.perh->STAT, UART_STAT_PERR_MSK) == UART_STAT_PERR_MSK)
                return CARD_ERROR + i;
        }

        return CARD_OK;
    }
}

/**
  * @brief:  cold reset smart card and receive rseponse.
  * @param:  rxbuf: receive buffer address.
  * @retval: None
  */
uint8_t reset_rec(uint8_t *rxbuf)
{
    uint16_t cnt;
    CARD_RST_L;
    ald_delay_ms(1);
    CARD_RST_H;

    if (get_card_byte(rxbuf, 2) == CARD_OK)
    {
        if ((rxbuf[0] != 0x3B) || (rxbuf[1] != 0x69))
            return CARD_ERROR;

        cnt = 4000;

        /*Ignore the reset reply*/
        while ((get_card_byte(rxbuf, 31) == CARD_OK) && (--cnt));

        if (cnt > 0)
            return CARD_OK;
        else
            return CARD_ERROR;
    }
    else
    {
        return CARD_ERROR;
    }
}

/**
  * @brief:  cold test smart card and receive rseponse.
  * @param:  rxbuf: receive buffer address.
  * @retval: None
  */
uint8_t instruction_test(void)
{
    uint16_t cnt;
    uint8_t result;

    /*Instructions for the common communication, only communication*/
    g_tx_buf[0] = 0x00;
    g_tx_buf[1] = 0xA4;
    g_tx_buf[2] = 0x00;
    g_tx_buf[3] = 0x00;
    g_tx_buf[4] = 0x02;
    g_tx_buf[5] = 0x3F;
    g_tx_buf[6] = 0x00;

    cnt = 4000;

    /*Waiting for the ESAM free*/
    while ((get_card_byte(g_rx_buf, 31) == CARD_OK) && (--cnt));

    if (cnt < 1)
        return CARD_ERROR;

    /*Before sending, close to receive*/
    CLEAR_BIT(g_h_uart.perh->LCON, UART_LCON_RXEN_MSK);

    /*Send command head (5) to ESAM g_tx_buf [0] - g_tx_buf [4]*/
    /*Instructions category CLA code INS instruction reference symbol number of bytes of data (P1, P2, P3*/
    result = send_card_byte(g_tx_buf, 5);

    /*After sending, open to receive*/
    SET_BIT(g_h_uart.perh->LCON, UART_LCON_RXEN_MSK);

    /*Whether to send and receive is normal*/
    if (result == CARD_ERROR)
        return CARD_ERROR;

    /*One byte process*/
    result = get_card_byte(g_rx_buf, 1);

    if (result != CARD_OK)
        return CARD_ERROR;

    /*If the return process of instruction code byte = = INS*/
    if (g_rx_buf[0] == g_tx_buf[1])
    {
        CLEAR_BIT(g_h_uart.perh->LCON, UART_LCON_RXEN_MSK);

        /*Send the rest of the data bytes, number of bytes of data = number of P3 = g_tx_buf [4]*/
        result = send_card_byte(g_tx_buf + 5, g_tx_buf[4]);

        SET_BIT(g_h_uart.perh->LCON, UART_LCON_RXEN_MSK);

        if (result != CARD_OK)
            return CARD_ERROR;

        /*Receive 2 status byte*/
        result = get_card_byte(g_rx_buf, 2);

        if (result != CARD_OK)
            return CARD_ERROR;

        /*0 x61xx and 0 x9000 are normal processing*/
        if (g_rx_buf[0] == 0x61)
            return CARD_OK;

        if (g_rx_buf[0] == 0x90 && g_rx_buf[1] == 0x00)
            return CARD_OK;

    }

    return CARD_ERROR;

}


/**
  * @brief:  cold test smart card and receive rseponse.
  * @param:  rxbuf: receive buffer address.
  * @retval: None
  */
uint8_t rand_instruction_test(void)
{
    uint16_t cnt;
    uint8_t result, different_times = 0U;
    uint32_t last_rand, temp_rand = 0U;

    /*Instructions for the generated four random 8 bits of data*/
    g_tx_buf[0] = 0x00;
    g_tx_buf[1] = 0x84;
    g_tx_buf[2] = 0x00;
    g_tx_buf[3] = 0x00;
    g_tx_buf[4] = 0x04;

    last_rand = temp_rand;

    while (different_times < 5)
    {
        cnt = 4000;

        /*Waiting for the ESAM free*/
        while ((get_card_byte(g_rx_buf, 31) == CARD_OK) && (--cnt));

        if (cnt < 1)
            return CARD_ERROR;

        /*Before sending, close to receive*/
        CLEAR_BIT(g_h_uart.perh->LCON, UART_LCON_RXEN_MSK);

        /*Send command head (5) to ESAM g_tx_buf [0] - g_tx_buf [4]*/
        /*Instructions category CLA code INS instruction reference symbol number of bytes of data (P1, P2, P3*/
        result = send_card_byte(g_tx_buf, 5);

        /*After sending, open to receive*/
        SET_BIT(g_h_uart.perh->LCON, UART_LCON_RXEN_MSK);

        /*Whether to send and receive is normal*/
        if (result == CARD_ERROR)
            return CARD_ERROR;

        /*Process 1 byte + 0 (the data didn't send the instruction) + P3 state of a random number + two bytes*/
        result = get_card_byte(g_rx_buf, 1 + g_tx_buf[4] + 2);

        if (result != CARD_OK)
            return CARD_ERROR;

        /*Judge status, 0 x90 0 x00 = command execution success*/
        if (g_rx_buf[1 + g_tx_buf[4]] != 0x90 || g_rx_buf[1 + g_tx_buf[4] + 1] != 0x00)
            return CARD_ERROR;

        /*Will be credited to temp_rand 4 8 bits of random number*/
        memcpy(&temp_rand, g_rx_buf + 1, g_tx_buf[4]);

        /*To determine whether a random number change*/
        if (temp_rand != last_rand)
        {
            last_rand = temp_rand;
            different_times++;
        }
    }

    return CARD_OK;
}


/**
  * @brief:  cold test smart card and receive rseponse.
  * @param:  rxbuf: receive buffer address.
  * @retval: None
  */
uint8_t name_instruction_test(void)
{
    uint16_t cnt;
    uint8_t result;

    /*Instructions for the serial number of the ESAM*/
    g_tx_buf[0] = 0x00;
    g_tx_buf[1] = 0xB0;
    g_tx_buf[2] = 0x99;
    g_tx_buf[3] = 0x01;
    g_tx_buf[4] = 0x08;

    cnt = 4000;

    /*Waiting for the ESAM free*/
    while ((get_card_byte(g_rx_buf, 31) == CARD_OK) && (--cnt));

    if (cnt < 1)
        return CARD_ERROR;

    /*Before sending, close to receive*/
    CLEAR_BIT(g_h_uart.perh->LCON, UART_LCON_RXEN_MSK);

    /*Send command head (5) to ESAM g_tx_buf [0] - g_tx_buf [4]*/
    /*Instructions category CLA code INS instruction reference symbol number of bytes of data (P1, P2, P3*/
    result = send_card_byte(g_tx_buf, 5);

    /*After sending, open to receive*/
    SET_BIT(g_h_uart.perh->LCON, UART_LCON_RXEN_MSK);

    /*Whether to send and receive is normal*/
    if (result == CARD_ERROR)
        return CARD_ERROR;

    /*Process 1 byte + 0 (the data didn't send the instruction) + P3 a serial number + 2 a status byte*/
    result = get_card_byte(g_rx_buf, 1 + g_tx_buf[4] + 2);

    if (result != CARD_OK)
        return CARD_ERROR;

    /*Judge status, 0 x90 0 x00 = command execution success*/
    if (g_rx_buf[1 + g_tx_buf[4]] != 0x90 || g_rx_buf[1 + g_tx_buf[4] + 1] != 0x00)
        return CARD_ERROR;

    return CARD_OK;
}

/**
  * @}Smart_card
  */

/**
  * @}Micro_Driver
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
