/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

uart_handle_t g_uart_init;

uint8_t g_rx_buf[256];
uint32_t g_rx_len = sizeof(g_rx_buf);

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initializate pin of uart module.
  * @retval None
  */
void init_uart_pin(void)
{
    gpio_init_t gpio_init;
    memset(&gpio_init, 0x00, sizeof(gpio_init_t));

    /* Initialize tx pin: PC0 */
    gpio_init.mode = GPIO_MODE_OUTPUT;
    gpio_init.odos = GPIO_OPEN_DRAIN;
    gpio_init.pupd = GPIO_PUSH_UP;
    gpio_init.podrv = GPIO_OUT_DRIVE_1;
    gpio_init.nodrv = GPIO_OUT_DRIVE_0_1;
    gpio_init.flt  = GPIO_FILTER_DISABLE;
    gpio_init.type = GPIO_TYPE_CMOS;
    gpio_init.func = GPIO_FUNC_3;
    ald_gpio_init(UART_TX_PORT, UART_TX_PIN, &gpio_init);

    return;
}

/**
  * @brief  Send message complete.
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None.
  */
void uart_send_complete(uart_handle_t *arg)
{
    uint16_t cnt = 4000U;

    while (ald_uart_get_status(&g_uart_init, UART_STATUS_TSBUSY) && (--cnt));

    UART_TX_DISABLE(&g_uart_init);
    UART_RX_ENABLE(&g_uart_init);
    ald_uart_recv_by_it(arg, g_rx_buf, 1);

    return;
}

/**
  * @brief  Receive a message complete.
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None.
  */
void uart_recv_complete(uart_handle_t *arg)
{
    uint16_t cnt;
    static uint32_t s_i;
    s_i++;

    if ((s_i > 1) && (*(arg->rx_buf - 2) == 0x0D) && (*(arg->rx_buf - 1) == 0x0A))
    {
        cnt = 4000;

        while (ald_uart_get_status(&g_uart_init, UART_STATUS_RSBUSY) && (--cnt));

        ald_delay_us(70);
        UART_RX_DISABLE(&g_uart_init);
        UART_TX_ENABLE(&g_uart_init);
        ald_uart_send_by_it(&g_uart_init, g_rx_buf, s_i);
        s_i = 0;
    }
    else
    {
        if (s_i >= g_rx_len)
            s_i = 0;

        ald_uart_recv_by_it(arg, g_rx_buf + s_i, 1);
    }

    return;
}

/**
  * @brief  Occurs error.
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None.
  */
void uart_error(uart_handle_t *arg)
{
    return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Initialize ALD */
    ald_cmu_init();
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_72M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
    ald_mcu_irq_config(UART1_IRQn, 3, 3, ENABLE);

    /* Initialize pin */
    init_uart_pin();

    memset(&g_uart_init, 0x00, sizeof(uart_handle_t));
    /* Initialize uart */
    g_uart_init.perh             = UART1;
    g_uart_init.init.baud        = 9600;
    g_uart_init.init.word_length = UART_WORD_LENGTH_8B;
    g_uart_init.init.stop_bits   = UART_STOP_BITS_1;
    g_uart_init.init.parity      = UART_PARITY_NONE;
    g_uart_init.init.mode        = UART_MODE_HDSEL;
    g_uart_init.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
    g_uart_init.rx_cplt_cbk      = uart_recv_complete;
    g_uart_init.tx_cplt_cbk      = uart_send_complete;
    ald_uart_init(&g_uart_init);

    UART_TX_DISABLE(&g_uart_init);

    ald_uart_rx_fifo_config(&g_uart_init, UART_RXFIFO_1BYTE);
    ald_uart_tx_fifo_config(&g_uart_init, UART_TXFIFO_EMPTY);

    ald_uart_recv_by_it(&g_uart_init, g_rx_buf, 1);

    while (1);
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
