/**********************************************************************************
 *
 * @file    main.c
 * @brief   capture_dma C file
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "main.h"

/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Variables --------------------------------------------------------- */

/* Private Macros ------------------------------------------------------------ */
/* Buffer size */
#define BUFFER_SIZE  10

#define CAPTURE_CH1_PORT GPIOC
#define CAPTURE_CH1_PIN  MD_GPIO_PIN_6

/* Private Variables ---------------------------------------------------------- */
md_dma_descriptor_t dma_ctrl_base[2] __attribute__((aligned(512)));
md_dma_config_t g_dma_config_duty;
md_dma_config_t g_dma_config_cycle;

/* duty/cycle data buffer */
uint16_t g_duty_buffer[BUFFER_SIZE];
uint16_t g_cycle_buffer[BUFFER_SIZE];

/* Public Variables ---------------------------------------------------------- */
uint8_t g_cplt_cnt = 0U;

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initialize Capture port.
  * @param  None
  * @retval None
  */
void init_gpio(void)
{
    md_gpio_init_t gpio_init;

    /* Initialize CH1IN */
    md_gpio_init_struct(&gpio_init);         /* initialize the gpio_init */
    gpio_init.mode = MD_GPIO_MODE_INPUT;     /* input */
    gpio_init.odos = MD_GPIO_PUSH_PULL;
    gpio_init.pupd = MD_GPIO_FLOATING;       /* no pull up and pull donw */
    gpio_init.podrv= MD_GPIO_OUT_DRIVE_0_1;
    gpio_init.nodrv= MD_GPIO_OUT_DRIVE_0_1;
    gpio_init.flt = MD_GPIO_FILTER_DISABLE;  /* no filter */
    gpio_init.type = MD_GPIO_TYPE_CMOS;      /* CMOS */
    gpio_init.func = MD_GPIO_FUNC_2;         /* GPIO function 2 */
    md_gpio_init(CAPTURE_CH1_PORT, CAPTURE_CH1_PIN, &gpio_init);
}

/**
  * @brief:  Initialize AD16C4T to capture the period and duty cycle
  *          of square wave in CH1_IN port
  * @param:  None
  * @retval: None
  */
static void init_timer(void)
{
    md_timer_base_init_t ad16c4t_init;
    md_timer_ic_init_t ic_init;

    md_timer_base_struct_init(&ad16c4t_init);          /* initialize the ad16c4t_init  */
    ad16c4t_init.prescaler  = 72 - 1;                  /* clk_count: 1MHz */
    ad16c4t_init.clk_div    = MD_TIMER_CLOCK_DIV1;     /* working clock of dead time and filter */
    ad16c4t_init.mode       = MD_TIMER_CNT_MODE_UP;    /* count up */
    ad16c4t_init.period     = 0xFFFF;                  /* set max period */
    ad16c4t_init.re_cnt     = 0;                       /* no repeat count */
    md_timer_base_set_config(AD16C4T1, &ad16c4t_init);

    md_timer_ic_struct_init(&ic_init);                 /* initialize the ic_init  */
    ic_init.filter      = 0;                           /* 0 filter */
    ic_init.polarity    = MD_TIMER_IC_POLARITY_RISE;   /* capture rising edge */
    ic_init.psc         = MD_TIMER_IC_PSC_DIV1;        /* no prescaler in capture channel */
    ic_init.sel         = MD_TIMER_IC_SEL_DIRECT;      /* capture this channel */
    md_timer_ic_init(AD16C4T1, MD_TIMER_CHANNEL_1, &ic_init);
    md_timer_ic_struct_init(&ic_init);
    ic_init.filter      = 0;
    ic_init.polarity    = MD_TIMER_IC_POLARITY_FALL;   /* capture falling edge */
    ic_init.psc         = MD_TIMER_IC_PSC_DIV1;        /* no prescaler in capture channel */
    ic_init.sel         = MD_TIMER_IC_SEL_INDIRECT;    /* capture adjacent channel */
    md_timer_ic_init(AD16C4T1, MD_TIMER_CHANNEL_2, &ic_init);

    md_timer_set_slave_mode_smods(AD16C4T1, 4);        /* 0x4: Reset Mode */
    md_timer_set_slave_trigger_tssel(AD16C4T1, 5);     /* 0x5: Filtered Timer Input 1 */

    md_timer_set_cc_dma_select_ccdmasel(AD16C4T1, 0);  /* CCn DMA request sent when CCx event occurs */
    md_timer_enable_cc1dma(AD16C4T1);        /* enable CC1 DMA request */
    md_timer_enable_cc2dma(AD16C4T1);        /* enable CC2 DMA request */

    md_timer_enable_cc1it_interrupt(AD16C4T1);         /* enable CC1 interrupt request */
    md_timer_enable_cc2it_interrupt(AD16C4T1);         /* enable CC1 interrupt request */
    md_timer_enable_uit_interrupt(AD16C4T1);           /* enable update interrupt request */

    md_mcu_irq_config(AD16C4T1_UP_IRQn, 0, 0, ENABLE); /* enable NVIC AD16C4T1 updata interrupt */
    md_mcu_irq_config(AD16C4T1_CC_IRQn, 0, 0, ENABLE); /* enable NVIC AD16C4T1 capture and compare interrupt */

    md_timer_enable_counter_cnten(AD16C4T1);           /* enable AD16C4T1 */
}

/**
  * @brief:  Use MD library to config DMA.
  * @param:  None
  * @retval: None
  */
static void init_dma(void)
{
    md_dma_set_ctrlbase((uint32_t)dma_ctrl_base);
    md_dma_enable();

    md_dma_config_struct(&g_dma_config_cycle);
    g_dma_config_cycle.src      = (void *)(&AD16C4T1->CCVAL1);  /* Source data begin pointer */
    g_dma_config_cycle.dst      = (void *)g_cycle_buffer;         /* Destination data begin pointer */
    g_dma_config_cycle.size     = BUFFER_SIZE;                  /* The total number of DMA transfers that DMA cycle contains */
    g_dma_config_cycle.data_width = MD_DMA_DATA_SIZE_HALFWORD;  /* Data width */
    g_dma_config_cycle.src_inc  = MD_DMA_DATA_INC_NONE;         /* Source increment type */
    g_dma_config_cycle.dst_inc  = MD_DMA_DATA_INC_HALFWORD;     /* Destination increment type */
    g_dma_config_cycle.R_power  = MD_DMA_R_POWER_1;             /* Control how many DMA transfers can occur before re-arbitrates */
    g_dma_config_cycle.primary  = ENABLE;                       /* Use primary descriptor or alternate descriptor */
    g_dma_config_cycle.burst    = DISABLE;                      /* Uses the alternate data structure when complete a DMA cycle */
    g_dma_config_cycle.high_prio    = DISABLE;                  /* High priority or default priority */
    g_dma_config_cycle.interrupt    = ENABLE;                   /* Enable/disable interrupt */
    g_dma_config_cycle.msel     = MD_DMA_MSEL_TIMER1;           /* Input source to DMA channel */
    g_dma_config_cycle.msigsel  = MD_DMA_MSIGSEL_TIMER_CH1;     /* Input signal to DMA channel */
    g_dma_config_cycle.channel  = 0;                            /* Channel index */
    md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_BASIC, &g_dma_config_cycle);

    md_dma_set_ctrlbase((uint32_t)dma_ctrl_base);
    md_dma_config_struct(&g_dma_config_duty);
    g_dma_config_duty.src      = (void *)(&AD16C4T1->CCVAL2);  /* Source data begin pointer */
    g_dma_config_duty.dst      = (void *)g_duty_buffer;          /* Destination data begin pointer */
    g_dma_config_duty.size     = BUFFER_SIZE;                  /* The total number of DMA transfers that DMA cycle contains */
    g_dma_config_duty.data_width = MD_DMA_DATA_SIZE_HALFWORD;  /* Data width */
    g_dma_config_duty.src_inc  = MD_DMA_DATA_INC_NONE;         /* Source increment type */
    g_dma_config_duty.dst_inc  = MD_DMA_DATA_INC_HALFWORD;     /* Destination increment type */
    g_dma_config_duty.R_power  = MD_DMA_R_POWER_1;             /* Control how many DMA transfers can occur before re-arbitrates */
    g_dma_config_duty.primary  = ENABLE;                       /* Use primary descriptor or alternate descriptor */
    g_dma_config_duty.burst    = DISABLE;                      /* Uses the alternate data structure when complete a DMA cycle */
    g_dma_config_duty.high_prio    = DISABLE;                  /* High priority or default priority */
    g_dma_config_duty.interrupt    = ENABLE;                   /* Enable/disable interrupt */
    g_dma_config_duty.msel     = MD_DMA_MSEL_TIMER1;           /* Input source to DMA channel */
    g_dma_config_duty.msigsel  = MD_DMA_MSIGSEL_TIMER_CH2;     /* Input signal to DMA channel */
    g_dma_config_duty.channel  = 1;                            /* Channel index */
    md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_BASIC, &g_dma_config_duty);

    md_mcu_irq_config(DMA_IRQn, 0, 0, ENABLE); /* enable NVIC DMA interrupt */
    md_dma_enable_channel(MD_DMA_CH_0);        /* enable DMA channel 0 */
    md_dma_enable_channel(MD_DMA_CH_1);        /* enable DMA channel 1 */
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Configure system clock */
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HOSC_3, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();
    /* 2bits preempt-priority(0-3)
       2bits sub-priority(0-3). */
    NVIC_SetPriorityGrouping(5);

    /* Initialize pin */
    init_gpio();
    /* Initialize timer */
    init_timer();
    /* Initialize dma */
    init_dma();

    while (1)
    {
        if (g_cplt_cnt == 2)
        {
            g_cplt_cnt = 0;

            /* reconfig dma */
            g_dma_config_cycle.src      = (void *)(&AD16C4T1->CCVAL1);
            g_dma_config_cycle.size     = BUFFER_SIZE;
            md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_BASIC, &g_dma_config_cycle);
            g_dma_config_duty.src      = (void *)(&AD16C4T1->CCVAL2);
            g_dma_config_duty.size     = BUFFER_SIZE;
            md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_BASIC, &g_dma_config_duty);
            md_dma_enable_channel(MD_DMA_CH_0);
            md_dma_enable_channel(MD_DMA_CH_1);
        }
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
