/**********************************************************************************
 *
 * @file    svpwm.c
 * @brief   Interrupt handler
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "es32f36xx.h"
#include "stdint.h"
#include "ald_timer.h"
#include "svpwm.h"

/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

extern uint16_t g_Acnt;
extern uint16_t g_Bcnt;
extern uint16_t g_Ccnt;

/* Private Macros ------------------------------------------------------------ */
/* 空间电压矢量各扇区对应角度用无符号16位数表示如下： */
#define SECTOR_1    0                   /* 0° */
#define SECTOR_2    ((uint16_t)10922)   /* 60° */
#define SECTOR_3    ((uint16_t)21845)   /* 120° */
#define SECTOR_4    ((uint16_t)32768)   /* 180° */
#define SECTOR_5    ((uint16_t)43690)   /* 240° */
#define SECTOR_6    ((uint16_t)54613)   /* 300° */
#define DEG_60      ((uint16_t)10922)    /* 60° */
#define LIMIT_UPPER_VOLT 32767           /* 80% */

/* 0°~60°范围正弦表，该正弦表对应170个角度，精度为0.35°，具体表示如下： */
const int sinetable[] =
{
    0, 201, 401, 602, 803, 1003, 1204, 1404, 1605, 1805, 2005, 2206, 2406, 2606, 2806, 3006, 3205, 3405, 3605, 3804,
    4003, 4202, 4401, 4600, 4799, 4997, 5195, 5393, 5591, 5789, 5986, 6183, 6380, 6577, 6773, 6970, 7166, 7361, 7557, 7752,
    7947, 8141, 8335, 8529, 8723, 8916, 9109, 9302, 9494, 9686, 9877, 10068, 10259, 10449, 10639, 10829, 11018, 11207, 11395, 11583,
    11771, 11958, 12144, 12331, 12516, 12701, 12886, 13070, 13254, 13437, 13620, 13802, 13984, 14165, 14346, 14526, 14706, 14885, 15063, 15241,
    15419, 15595, 15772, 15947, 16122, 16297, 16470, 16643, 16816, 16988, 17159, 17330, 17500, 17669, 17838, 18006, 18173, 18340, 18506, 18671,
    18835, 18999, 19162, 19325, 19487, 19647, 19808, 19967, 20126, 20284, 20441, 20598, 20753, 20908, 21062, 21216, 21368, 21520, 21671, 21821,
    21970, 22119, 22266, 22413, 22559, 22704, 22848, 22992, 23134, 23276, 23417, 23557, 23696, 23834, 23971, 24107, 24243, 24377, 24511, 24644,
    24776, 24906, 25036, 25165, 25293, 25420, 25547, 25672, 25796, 25919, 26042, 26163, 26283, 26403, 26521, 26638, 26755, 26870, 26984, 27098,
    27210, 27321, 27431, 27541, 27649, 27756, 27862, 27967, 28071, 28174, 28276, 28377
};

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  The function calculates three duty cycle values basedon the hT0, hT1, and hT2 times.
  * @param  int16_t volts,uint16_t Angle.
  * @retval None.
  */
void Cal_SVPWM(int32_t volts, uint16_t Angle)
{
    uint16_t angle1, angle2;
    uint16_t hT0_div2, hT1, hT2, hTs;

    hTs = AD16C4T1->AR;

    if (volts > LIMIT_UPPER_VOLT)
        volts = LIMIT_UPPER_VOLT;

    if (Angle < SECTOR_2)
    {
        angle2 = Angle - SECTOR_1;                      /* 计算扇区基准角度 */
        angle1 = DEG_60 - angle2;                       /* 计算电压矢量与扇区基准夹角 */

        hT1 = sinetable[(unsigned char)(angle1 >> 6)];  /* 查正弦表 */
        hT2 = sinetable[(unsigned char)(angle2 >> 6)];
        hT1 = ((uint32_t)hT1 * (uint32_t)volts) >> 15;  /* 电压幅值归一化 */
        hT1 = ((uint32_t)hT1 * (uint32_t)hTs) >> 15;    /* 占空比归一化 */
        hT2 = ((uint32_t)hT2 * (uint32_t)volts) >> 15;
        hT2 = ((uint32_t)hT2 * (uint32_t)hTs) >> 15;
        hT0_div2 = (hTs - hT1 - hT2) >> 1;              /* 计算零矢量作用时间 */

        AD16C4T1->CCVAL1 = hT1 + hT2 + hT0_div2;        /* 计算占空比 */
        AD16C4T1->CCVAL2 = hT2 + hT0_div2;
        AD16C4T1->CCVAL3 = hT0_div2;
    }
    else if (Angle < SECTOR_3)
    {
        angle2 = Angle - SECTOR_2;                      /* 计算扇区基准角度 */
        angle1 = DEG_60 - angle2;                       /* 计算电压矢量与扇区基准夹角 */

        hT1 = sinetable[(unsigned char)(angle1 >> 6)];  /* 查正弦表 */
        hT2 = sinetable[(unsigned char)(angle2 >> 6)];

        hT1 = ((uint32_t)hT1 * (uint32_t)volts) >> 15;  /* 电压幅值归一化 */
        hT1 = ((uint32_t)hT1 * (uint32_t)hTs) >> 15;    /* 占空比归一化 */
        hT2 = ((uint32_t)hT2 * (uint32_t)volts) >> 15;
        hT2 = ((uint32_t)hT2 * (uint32_t)hTs) >> 15;
        hT0_div2 = (hTs - hT1 - hT2) >> 1;              /* 计算零矢量作用时间 */

        AD16C4T1->CCVAL1 = hT1 + hT0_div2;              /* 计算占空比 */
        AD16C4T1->CCVAL2 = hT1 + hT2 + hT0_div2;
        AD16C4T1->CCVAL3 = hT0_div2;
    }
    else if (Angle < SECTOR_4)
    {
        angle2 = Angle - SECTOR_3;                      /* 计算扇区基准角度 */
        angle1 = DEG_60 - angle2;                       /* 计算电压矢量与扇区基准夹角 */

        hT1 = sinetable[(unsigned char)(angle1 >> 6)];  /* 查正弦表 */
        hT2 = sinetable[(unsigned char)(angle2 >> 6)];

        hT1 = ((uint32_t)hT1 * (uint32_t)volts) >> 15;  /* 电压幅值归一化 */
        hT1 = ((uint32_t)hT1 * (uint32_t)hTs) >> 15;    /* 占空比归一化 */
        hT2 = ((uint32_t)hT2 * (uint32_t)volts) >> 15;
        hT2 = ((uint32_t)hT2 * (uint32_t)hTs) >> 15;
        hT0_div2 = (hTs - hT1 - hT2) >> 1;              /* 计算零矢量作用时间 */

        AD16C4T1->CCVAL1 = hT0_div2;        /* 计算占空比 */
        AD16C4T1->CCVAL2 = hT1 + hT2 + hT0_div2;
        AD16C4T1->CCVAL3 = hT2 + hT0_div2;
    }
    else if (Angle < SECTOR_5)
    {
        angle2 = Angle - SECTOR_4;                      /* 计算扇区基准角度 */
        angle1 = DEG_60 - angle2;                       /* 计算电压矢量与扇区基准夹角 */

        hT1 = sinetable[(unsigned char)(angle1 >> 6)];  /* 查正弦表 */
        hT2 = sinetable[(unsigned char)(angle2 >> 6)];

        hT1 = ((uint32_t)hT1 * (uint32_t)volts) >> 15;  /* 电压幅值归一化 */
        hT1 = ((uint32_t)hT1 * (uint32_t)hTs) >> 15;    /* 占空比归一化 */
        hT2 = ((uint32_t)hT2 * (uint32_t)volts) >> 15;
        hT2 = ((uint32_t)hT2 * (uint32_t)hTs) >> 15;
        hT0_div2 = (hTs - hT1 - hT2) >> 1;              /* 计算零矢量作用时间 */

        AD16C4T1->CCVAL1 = hT0_div2;                    /* 计算占空比 */
        AD16C4T1->CCVAL2 = hT1 + hT0_div2;
        AD16C4T1->CCVAL3 = hT1 + hT2 + hT0_div2;
    }
    else if (Angle < SECTOR_6)
    {
        angle2 = Angle - SECTOR_5;                      /* 计算扇区基准角度 */
        angle1 = DEG_60 - angle2;                       /* 计算电压矢量与扇区基准夹角 */

        hT1 = sinetable[(unsigned char)(angle1 >> 6)];  /* 查正弦表 */
        hT2 = sinetable[(unsigned char)(angle2 >> 6)];

        hT1 = ((uint32_t)hT1 * (uint32_t)volts) >> 15;  /* 电压幅值归一化 */
        hT1 = ((uint32_t)hT1 * (uint32_t)hTs) >> 15;    /* 占空比归一化 */
        hT2 = ((uint32_t)hT2 * (uint32_t)volts) >> 15;
        hT2 = ((uint32_t)hT2 * (uint32_t)hTs) >> 15;
        hT0_div2 = (hTs - hT1 - hT2) >> 1;              /* 计算零矢量作用时间 */

        AD16C4T1->CCVAL1 = hT2 + hT0_div2;              /* 计算占空比 */
        AD16C4T1->CCVAL2 = hT0_div2;
        AD16C4T1->CCVAL3 = hT1 + hT2 + hT0_div2;

    }
    else
    {
        angle2 = Angle - SECTOR_6;                      /* 计算扇区基准角度 */
        angle1 = DEG_60 - angle2;                       /* 计算电压矢量与扇区基准夹角 */

        hT1 = sinetable[(unsigned char)(angle1 >> 6)];  /* 查正弦表 */
        hT2 = sinetable[(unsigned char)(angle2 >> 6)];

        hT1 = ((uint32_t)hT1 * (uint32_t)volts) >> 15;  /* 电压幅值归一化 */
        hT1 = ((uint32_t)hT1 * (uint32_t)hTs) >> 15;    /* 占空比归一化 */
        hT2 = ((uint32_t)hT2 * (uint32_t)volts) >> 15;
        hT2 = ((uint32_t)hT2 * (uint32_t)hTs) >> 15;
        hT0_div2 = (hTs - hT1 - hT2) >> 1;              /* 计算零矢量作用时间 */

        AD16C4T1->CCVAL1 = hT1 + hT2 + hT0_div2;        /* 计算占空比 */
        AD16C4T1->CCVAL2 = hT0_div2;
        AD16C4T1->CCVAL3 = hT1 + hT0_div2;
    }

    g_Acnt = AD16C4T1->CCVAL1;
    g_Bcnt = AD16C4T1->CCVAL2;
    g_Ccnt = AD16C4T1->CCVAL3;
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
