/**********************************************************************************
 *
 * @file    main.c
 * @brief   dma_input_capture C file
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "main.h"
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Variables --------------------------------------------------------- */
/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/* Public Variables ---------------------------------------------------------- */
timer_handle_t g_h_tim;
dma_handle_t g_hperh;
timer_clock_config_t g_clock;
timer_ic_init_t g_init;
timer_slave_config_t g_slave_config;

/* Private Macros ------------------------------------------------------------ */
/* duty/cycle data buffer */
#define BUFFER_SIZE 32
uint16_t g_duty_buffer[BUFFER_SIZE] __attribute__((aligned(4))) = {0U};
uint16_t g_cycle_buffer[BUFFER_SIZE] __attribute__((aligned(4))) = {0U};

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initializate pin of tim module.
  * @retval None
  */
void timer_pin_g_init(void)
{
    gpio_init_t x;

    /* Initialize tim0 ch1 pin */
    x.mode = GPIO_MODE_INPUT;
    x.pupd = GPIO_PUSH_UP;
    x.flt  = GPIO_FILTER_DISABLE;
    x.type = GPIO_TYPE_CMOS;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.func = GPIO_FUNC_2;
    ald_gpio_init(TIMER_INPUTCAPTURE_PORT, TIMER_INPUTCAPTURE_PIN, &x);
}

/**
  * @brief  Initializate DMA chanel0 and channel1.
  * @retval None
  */
void g_init_dma(void)
{
    g_hperh.perh = DMA0;
    ald_dma_config_struct(&g_hperh.config);

    g_hperh.config.src     = (void *)(&AD16C4T0->CCVAL1);
    g_hperh.config.dst     = (void *)g_cycle_buffer;
    g_hperh.config.size    = BUFFER_SIZE;
    g_hperh.config.data_width = DMA_DATA_SIZE_HALFWORD;
    g_hperh.config.src_inc = DMA_DATA_INC_NONE;
    g_hperh.config.dst_inc = DMA_DATA_INC_HALFWORD;
    g_hperh.config.R_power = DMA_R_POWER_1;

    g_hperh.config.msel    = DMA_MSEL_TIMER0;
    g_hperh.config.msigsel = DMA_MSIGSEL_TIMER_CH1;

    g_hperh.cplt_cbk       = NULL;
    g_hperh.cplt_arg       = NULL;
    g_hperh.err_cbk        = NULL;
    g_hperh.err_arg        = NULL;
    g_hperh.config.channel = 0;
    ald_dma_config_basic(&g_hperh);

    g_hperh.perh = DMA0;
    ald_dma_config_struct(&g_hperh.config);

    g_hperh.config.src     = (void *)(&AD16C4T0->CCVAL2);
    g_hperh.config.dst     = (void *)g_duty_buffer;
    g_hperh.config.size    = BUFFER_SIZE;
    g_hperh.config.data_width = DMA_DATA_SIZE_HALFWORD;
    g_hperh.config.src_inc = DMA_DATA_INC_NONE;
    g_hperh.config.dst_inc = DMA_DATA_INC_HALFWORD;
    g_hperh.config.R_power = DMA_R_POWER_1;

    g_hperh.config.msel    = DMA_MSEL_TIMER0;
    g_hperh.config.msigsel = DMA_MSIGSEL_TIMER_CH2;

    g_hperh.cplt_cbk       = NULL;
    g_hperh.cplt_arg       = NULL;
    g_hperh.err_cbk        = NULL;
    g_hperh.err_arg        = NULL;
    g_hperh.config.channel = 1;
    ald_dma_config_basic(&g_hperh);
}

/**
  * @brief  Main function
  * @retval Status
  */
int main(void)
{
    /* Initialize ALD */
    ald_cmu_init();
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_72M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
    /* Initialize pin and set the default state*/
    timer_pin_g_init();

    /* Initialize AD16C4T0 */
    g_h_tim.perh           = AD16C4T0;
    g_h_tim.init.prescaler = 0;
    g_h_tim.init.mode      = TIMER_CNT_MODE_UP;
    g_h_tim.init.period    = 0xFFFF;
    g_h_tim.init.clk_div   = TIMER_CLOCK_DIV1;
    g_h_tim.init.re_cnt    = 0;
    g_h_tim.capture_cbk    = NULL;
    ald_timer_ic_init(&g_h_tim);

    /* Initialize g_clock source */
    g_clock.source     = TIMER_SRC_INTER;
    ald_timer_config_clock_source(&g_h_tim, &g_clock);

    /* Initialize input capture channel */
    g_init.polarity = TIMER_IC_POLARITY_RISE;
    g_init.sel      = TIMER_IC_SEL_DIRECT;
    g_init.psc      = TIMER_IC_PSC_DIV1;
    g_init.filter   = 3;
    ald_timer_ic_config_channel(&g_h_tim, &g_init, TIMER_CHANNEL_1);

    /* Initialize input capture channel */
    g_init.polarity = TIMER_IC_POLARITY_FALL;
    g_init.sel      = TIMER_IC_SEL_INDIRECT;
    g_init.psc      = TIMER_IC_PSC_DIV1;
    g_init.filter   = 2;
    ald_timer_ic_config_channel(&g_h_tim, &g_init, TIMER_CHANNEL_2);

    g_slave_config.mode     = TIMER_MODE_RESET;
    g_slave_config.input    = TIMER_TS_TI1FP1;
    g_slave_config.polarity     = TIMER_CLK_POLARITY_RISE;
    g_slave_config.psc          = TIMER_ETR_PSC_DIV1;
    g_slave_config.filter       = 4;
    ald_timer_slave_config_sync(&g_h_tim, &g_slave_config);

    g_init_dma();

    TIMER_CCx_DMA_REQ_CCx(&g_h_tim);
    AD16C4T0->DMAEN |= (1 << 1);    /* TIMER DMA channel0 enable */
    AD16C4T0->DMAEN |= (1 << 2);    /* TIMER DMA channel1 enable */

    /* Start input capture */
    ald_timer_ic_start(&g_h_tim, TIMER_CHANNEL_1);
    ald_timer_ic_start(&g_h_tim, TIMER_CHANNEL_2);

    while (1)
    {
        ald_delay_ms(1000);
    }
}
/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
