/**********************************************************************************
 *
 * @file    main.c
 * @brief   capture_dma C file
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */
#define BUF_LEN         10
/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
/* Private Variables --------------------------------------------------------- */
timer_handle_t g_ad16c4t_init, g_ad16c4t_init_2;
timer_ic_init_t g_ic_init;
timer_clock_config_t g_ad16c4t_clock_config;
gpio_init_t g_gpio_init;

/* duty/cycle data buffer */
uint16_t g_duty_buffer[BUF_LEN] = {0U};
uint16_t g_cycle_buffer[BUF_LEN] = {0U};

/* Public Variables ---------------------------------------------------------- */
uint8_t g_cplt_cnt = 0U;

/* Private Function ---------------------------------------------------------- */
/**
  * @brief  DMA callback  function
  * @retval None
  */
void ald_dma_cplt_callback(timer_handle_t *tim)
{
    if (ald_dma_get_flag_status(DMA0, 0))
    {
        TIMER_CCx_DISABLE(tim, TIMER_CHANNEL_1);
        ald_timer_clear_flag_status(tim, TIMER_FLAG_CC1);
        g_cplt_cnt++;
    }

    if (ald_dma_get_flag_status(DMA0, 1))
    {
        TIMER_CCx_DISABLE(tim, TIMER_CHANNEL_2);
        ald_timer_clear_flag_status(tim, TIMER_FLAG_CC2);
        g_cplt_cnt++;
    }

    return;
}

/**
  * @brief  Initialize TOGGLE IO.
  * @param  None.
  * @retval None.
  */
void init_gpio(void)
{
    /* Initialize P as input */
    memset(&g_gpio_init, 0x0, sizeof(g_gpio_init));
    g_gpio_init.mode = GPIO_MODE_INPUT;
    g_gpio_init.odos = GPIO_PUSH_PULL;
    g_gpio_init.pupd = GPIO_FLOATING;
    g_gpio_init.podrv= GPIO_OUT_DRIVE_0_1;
    g_gpio_init.nodrv= GPIO_OUT_DRIVE_0_1;
    g_gpio_init.flt  = GPIO_FILTER_DISABLE;
    g_gpio_init.type = GPIO_TYPE_CMOS;
    g_gpio_init.func = GPIO_FUNC_2;
    ald_gpio_init(CAPTURE_CH1_PORT, CAPTURE_CH1_PIN, &g_gpio_init);
}

/**
  * @brief:  Initialize AD16C4T to timely overflow funtion
  * @param:  None
  * @retval: None
  */
static void init_timer(void)
{
    /* Initialize AD16C4T1 */
    memset(&g_ad16c4t_init, 0x0, sizeof(g_ad16c4t_init));   /* initialize the g_ad16c4t_init */
    g_ad16c4t_init.perh = AD16C4T1;
    g_ad16c4t_init.init.prescaler    = 72 - 1;              /* clk_count: 1MHz */
    g_ad16c4t_init.init.mode         = TIMER_CNT_MODE_UP;   /* count up */
    g_ad16c4t_init.init.period       = 0xFFFF;              /* period is max count clock */
    g_ad16c4t_init.init.clk_div      = TIMER_CLOCK_DIV1;    /* working clock of dead time and filter */
    g_ad16c4t_init.init.re_cnt       = 1 - 1;               /* 1 repeat count */
    g_ad16c4t_init.capture_cbk       = ald_dma_cplt_callback;
    ald_timer_base_init(&g_ad16c4t_init);
    /* Initialize clock source */
    g_ad16c4t_clock_config.source = TIMER_SRC_INTER;        /**< TIMER clock sources */
    ald_timer_config_clock_source(&g_ad16c4t_init, &g_ad16c4t_clock_config);

    /* Initialize input capture channel 1 */
    g_ic_init.polarity = TIMER_IC_POLARITY_RISE;
    g_ic_init.sel      = TIMER_IC_SEL_DIRECT;
    g_ic_init.psc      = TIMER_IC_PSC_DIV1;
    g_ic_init.filter   = 0;
    ald_timer_ic_config_channel(&g_ad16c4t_init, &g_ic_init, TIMER_CHANNEL_1);

    /* Initialize input capture channel 2 */
    g_ic_init.polarity = TIMER_IC_POLARITY_FALL;
    g_ic_init.sel      = TIMER_IC_SEL_INDIRECT;
    g_ic_init.psc      = TIMER_IC_PSC_DIV1;
    g_ic_init.filter   = 0;
    ald_timer_ic_config_channel(&g_ad16c4t_init, &g_ic_init, TIMER_CHANNEL_2);

    ald_timer_pwm_set_input(&g_ad16c4t_init, TIMER_CHANNEL_1);
    MODIFY_REG(g_ad16c4t_init.perh->CCEP, TIMER_CCEP_CC1POL_MSK, TIMER_IC_POLARITY_RISE << TIMER_CCEP_CC1POL_POS);
    MODIFY_REG(g_ad16c4t_init.perh->CCEP, TIMER_CCEP_CC1NPOL_MSK, TIMER_IC_POLARITY_RISE << TIMER_CCEP_CC1NPOL_POS);
    MODIFY_REG(g_ad16c4t_init.perh->CCEP, TIMER_CCEP_CC2POL_MSK, TIMER_IC_POLARITY_FALL << TIMER_CCEP_CC2POL_POS);
    MODIFY_REG(g_ad16c4t_init.perh->CCEP, TIMER_CCEP_CC2NPOL_MSK, TIMER_IC_POLARITY_RISE << TIMER_CCEP_CC2NPOL_POS);

    if (g_ad16c4t_init.hdma1.perh == NULL)
        g_ad16c4t_init.hdma1.perh = DMA0;

    g_ad16c4t_init_2 = g_ad16c4t_init;
    g_ad16c4t_init_2.state = TIMER_STATE_READY;

    SET_BIT(DMA0->CFG, DMA_CFG_MASTER_ENABLE_MSK);
    ald_timer_ic_start(&g_ad16c4t_init_2, TIMER_CHANNEL_2);
    ald_timer_ic_start_by_dma(&g_ad16c4t_init_2, TIMER_CHANNEL_2, g_cycle_buffer, BUF_LEN, 1);

    ald_timer_ic_start(&g_ad16c4t_init, TIMER_CHANNEL_1);
    ald_timer_ic_start_by_dma(&g_ad16c4t_init, TIMER_CHANNEL_1, g_duty_buffer, BUF_LEN, 0);

    ald_timer_clear_flag_status(&g_ad16c4t_init_2, TIMER_FLAG_CC1);
    ald_timer_clear_flag_status(&g_ad16c4t_init_2, TIMER_FLAG_CC2);

    return;
}


/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Initialize ALD */
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_72M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

    /* Initialize pin */
    init_gpio();

    /* Initialize timer */
    init_timer();

    while (1)
    {
        if (g_cplt_cnt == 2)
        {
            g_cplt_cnt = 0;
            init_timer();
        }
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
