/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */
md_dma_descriptor_t g_dma0_ctrl_base[2] __attribute__((aligned(512)));
md_dma_config_t g_dma_tx_config, g_dma_rx_config;
md_spi_init_t g_h_spi;

uint8_t g_recv_buf[20] = {0U};
uint8_t g_send_len = sizeof(g_recv_buf);
uint8_t g_recv_len = sizeof(g_recv_buf);

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initializate pin of spi module.
  * @retval None
  */
void spi_pin_init(void)
{
    md_gpio_init_t x;
    md_gpio_init_struct(&x);

    /* Initialize nss pin */
    x.mode  = MD_GPIO_MODE_INPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_CMOS;
    x.func  = MD_GPIO_FUNC_4;
    md_gpio_init(GPIO_PORT, SPI_NSS_PIN, &x);

    /* Initialize sck pin */
    x.mode  = MD_GPIO_MODE_INPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_CMOS;
    x.func  = MD_GPIO_FUNC_4;
    md_gpio_init(GPIO_PORT, SPI_SCK_PIN, &x);

    /* Initialize miso pin */
    x.mode  = MD_GPIO_MODE_OUTPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_CMOS;
    x.func  = MD_GPIO_FUNC_4;
    md_gpio_init(GPIO_PORT, SPI_MISO_PIN, &x);

    /* Initialize mosi pin */
    x.mode  = MD_GPIO_MODE_INPUT;
    x.odos  = MD_GPIO_PUSH_PULL;
    x.pupd  = MD_GPIO_PUSH_UP;
    x.podrv = MD_GPIO_OUT_DRIVE_1;
    x.nodrv = MD_GPIO_OUT_DRIVE_1;
    x.flt   = MD_GPIO_FILTER_DISABLE;
    x.type  = MD_GPIO_TYPE_CMOS;
    x.func  = MD_GPIO_FUNC_4;
    md_gpio_init(GPIO_PORT, SPI_MOSI_PIN, &x);

    return;
}

/**
  * @brief  dma_start_tx init.
  * @param  None
  * @retval None
  */
void dma_send_tx(void)
{
    memset(&g_dma_tx_config, 0x0, sizeof(md_dma_config_t));
    md_dma_set_ctrlbase((uint32_t)g_dma0_ctrl_base);

    /* Enable DMA */
    md_dma_enable();

    md_dma_config_struct(&g_dma_tx_config);
    g_dma_tx_config.data_width = MD_DMA_DATA_SIZE_BYTE;
    g_dma_tx_config.src_inc    = MD_DMA_DATA_INC_BYTE;
    g_dma_tx_config.dst_inc    = MD_DMA_DATA_INC_NONE;
    g_dma_tx_config.R_power    = MD_DMA_R_POWER_1;
    g_dma_tx_config.primary    = ENABLE;
    g_dma_tx_config.burst      = ENABLE;
    g_dma_tx_config.high_prio  = DISABLE;
    g_dma_tx_config.interrupt  = ENABLE;
    g_dma_tx_config.channel    = 0;
    g_dma_tx_config.src        = (void *)(g_recv_buf);
    g_dma_tx_config.dst        = (void *)(&SPI1->DATA);
    g_dma_tx_config.size       = g_send_len;
    g_dma_tx_config.msel       = MD_DMA_MSEL_SPI1;
    g_dma_tx_config.msigsel    = MD_DMA_MSIGSEL_SPI_TXEMPTY;
    md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_BASIC, &g_dma_tx_config);

    md_dma_clear_flag_done(MD_DMA_CH_0);
    /* Enable DMA channel */
    md_dma_enable_channel(MD_DMA_CH_0);
    /* enable dma tx */
    md_spi_enable_txdma(SPI1);
}

/**
  * @brief  dma_start_rx init.
  * @param  None
  * @retval None
  */
void dma_recv_rx(void)
{
    memset(&g_dma_rx_config, 0x0, sizeof(md_dma_config_t));
    md_dma_set_ctrlbase((uint32_t)g_dma0_ctrl_base);

    /* Enable DMA */
    md_dma_enable();

    md_dma_config_struct(&g_dma_rx_config);
    g_dma_rx_config.data_width = MD_DMA_DATA_SIZE_BYTE;
    g_dma_rx_config.src_inc    = MD_DMA_DATA_INC_NONE;
    g_dma_rx_config.dst_inc    = MD_DMA_DATA_INC_BYTE;
    g_dma_rx_config.R_power    = MD_DMA_R_POWER_1;
    g_dma_rx_config.primary    = ENABLE;
    g_dma_rx_config.burst      = ENABLE;
    g_dma_rx_config.high_prio  = DISABLE;
    g_dma_rx_config.interrupt  = ENABLE;
    g_dma_rx_config.channel    = 1;
    g_dma_rx_config.src        = (void *)(&SPI1->DATA);
    g_dma_rx_config.dst        = (void *)(g_recv_buf);
    g_dma_rx_config.size       = g_recv_len;
    g_dma_rx_config.msel       = MD_DMA_MSEL_SPI1;
    g_dma_rx_config.msigsel    = MD_DMA_MSIGSEL_SPI_RNR;
    md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_BASIC, &g_dma_rx_config);

    md_dma_clear_flag_done(MD_DMA_CH_1);
    /* Enable DMA channel */
    md_dma_enable_channel(MD_DMA_CH_1);
    /* enable dma rx */
    md_spi_enable_rxdma(SPI1);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HOSC_3, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);

    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();
    /* 2bits preempt-priority(0-3)
       2bits sub-priority(0-3). */
    NVIC_SetPriorityGrouping(5);

    spi_pin_init();

    md_spi_struct_init(&g_h_spi);
    g_h_spi.SPIx      = SPI1;
    g_h_spi.mode      = MD_SPI_MODE_SLAVER;
    g_h_spi.dir       = MD_SPI_DIRECTION_2LINES;
    g_h_spi.data_size = MD_SPI_DATA_SIZE_8;
    g_h_spi.baud      = MD_SPI_BAUD_2;
    g_h_spi.phase     = MD_SPI_CPHA_SECOND;
    g_h_spi.polarity  = MD_SPI_CPOL_HIGH;
    g_h_spi.first_bit = MD_SPI_FIRSTBIT_MSB;
    g_h_spi.ss_en     = DISABLE;
    g_h_spi.crc_calc  = DISABLE;
    g_h_spi.crc_poly  = 0;
    md_spi_init(&g_h_spi);

    NVIC_SetPriority(DMA_IRQn, 2);
    NVIC_EnableIRQ(DMA_IRQn);
    memset(g_dma0_ctrl_base, 0x0, sizeof(g_dma0_ctrl_base));

    dma_send_tx();
    dma_recv_rx();

    while (1);
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
