/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

md_spi_init_t g_spi_config;
uint8_t g_recv_buf[20] = {0U};
uint8_t g_send_buf[20] = {0U};
uint8_t g_send_len = sizeof(g_send_buf);
uint8_t g_recv_len = sizeof(g_recv_buf);

md_dma_descriptor_t g_dma0_ctrl_base[2] __attribute__((aligned(512)));
md_dma_config_t g_dma_tx_config, g_dma_rx_config;
uint8_t g_complete;
/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initializate pin of spi module.
  * @retval None
  */
void init_spi_pin(void)
{
    md_gpio_init_t gpio_init;
    md_gpio_init_struct(&gpio_init);

    /* Initialize nss pin */
    gpio_init.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.podrv = MD_GPIO_OUT_DRIVE_1;
    gpio_init.nodrv = MD_GPIO_OUT_DRIVE_1;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_1;
    md_gpio_init(GPIO_PORT, SPI_NSS_PIN, &gpio_init);
    md_gpio_set_pin_high(GPIO_PORT, SPI_NSS_PIN);

    /* Initialize sck pin */
    gpio_init.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.podrv = MD_GPIO_OUT_DRIVE_1;
    gpio_init.nodrv = MD_GPIO_OUT_DRIVE_1;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_4;
    md_gpio_init(GPIO_PORT, SPI_SCK_PIN, &gpio_init);

    /* Initialize miso pin */
    gpio_init.mode  = MD_GPIO_MODE_INPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.podrv = MD_GPIO_OUT_DRIVE_1;
    gpio_init.nodrv = MD_GPIO_OUT_DRIVE_1;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_4;
    md_gpio_init(GPIO_PORT, SPI_MISO_PIN, &gpio_init);

    /* Initialize mosi pin */
    gpio_init.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.podrv = MD_GPIO_OUT_DRIVE_1;
    gpio_init.nodrv = MD_GPIO_OUT_DRIVE_1;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_4;
    md_gpio_init(GPIO_PORT, SPI_MOSI_PIN, &gpio_init);

    return;
}

/**
  * @brief  Initializate spi module.
  * @retval None
  */
void init_spi(void)
{
    md_spi_struct_init(&g_spi_config);
    g_spi_config.SPIx      = SPI1;
    g_spi_config.mode      = MD_SPI_MODE_MASTER;
    g_spi_config.dir       = MD_SPI_DIRECTION_2LINES;
    g_spi_config.data_size = MD_SPI_DATA_SIZE_8;
    g_spi_config.baud      = MD_SPI_BAUD_128;
    g_spi_config.phase     = MD_SPI_CPHA_FIRST;
    g_spi_config.polarity  = MD_SPI_CPOL_HIGH;
    g_spi_config.first_bit = MD_SPI_FIRSTBIT_MSB;
    g_spi_config.ss_en     = DISABLE;
    g_spi_config.crc_calc  = DISABLE;
    g_spi_config.crc_poly  = 0;
    md_spi_init(&g_spi_config);
}

/**
  * @brief  dma_start_tx init.
  * @param  None
  * @retval None
  */
void dma_send_tx(void)
{
    memset(&g_dma_tx_config, 0x0, sizeof(md_dma_config_t));
    md_dma_set_ctrlbase((uint32_t)g_dma0_ctrl_base);

    /* Enable DMA */
    md_dma_enable();

    md_dma_config_struct(&g_dma_tx_config);
    g_dma_tx_config.data_width = MD_DMA_DATA_SIZE_BYTE;
    g_dma_tx_config.src_inc    = MD_DMA_DATA_INC_BYTE;
    g_dma_tx_config.dst_inc    = MD_DMA_DATA_INC_NONE;
    g_dma_tx_config.R_power    = MD_DMA_R_POWER_1;
    g_dma_tx_config.primary    = ENABLE;
    g_dma_tx_config.burst      = ENABLE;
    g_dma_tx_config.high_prio  = DISABLE;
    g_dma_tx_config.interrupt  = ENABLE;
    g_dma_tx_config.channel    = 0;
    g_dma_tx_config.src        = (void *)(g_send_buf);
    g_dma_tx_config.dst        = (void *)(&SPI1->DATA);
    g_dma_tx_config.size       = g_send_len;
    g_dma_tx_config.msel       = MD_DMA_MSEL_SPI1;
    g_dma_tx_config.msigsel    = MD_DMA_MSIGSEL_SPI_TXEMPTY;
    md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_BASIC, &g_dma_tx_config);
}

/**
  * @brief  dma_start_rx init.
  * @param  None
  * @retval None
  */
void dma_recv_rx(void)
{
    memset(&g_dma_rx_config, 0x0, sizeof(md_dma_config_t));
    md_dma_set_ctrlbase((uint32_t)g_dma0_ctrl_base);

    /* Enable DMA */
    md_dma_enable();

    md_dma_config_struct(&g_dma_rx_config);
    g_dma_rx_config.data_width = MD_DMA_DATA_SIZE_BYTE;
    g_dma_rx_config.src_inc    = MD_DMA_DATA_INC_NONE;
    g_dma_rx_config.dst_inc    = MD_DMA_DATA_INC_BYTE;
    g_dma_rx_config.R_power    = MD_DMA_R_POWER_1;
    g_dma_rx_config.primary    = ENABLE;
    g_dma_rx_config.burst      = ENABLE;
    g_dma_rx_config.high_prio  = DISABLE;
    g_dma_rx_config.interrupt  = ENABLE;
    g_dma_rx_config.channel    = 1;
    g_dma_rx_config.src        = (void *)(&SPI1->DATA);
    g_dma_rx_config.dst        = (void *)(g_recv_buf);
    g_dma_rx_config.size       = g_recv_len;
    g_dma_rx_config.msel       = MD_DMA_MSEL_SPI1;
    g_dma_rx_config.msigsel    = MD_DMA_MSIGSEL_SPI_RNR;
    md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_BASIC, &g_dma_rx_config);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    uint16_t cnt;

    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HOSC_3, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);

    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();
    /* 2bits preempt-priority(0-3)
       2bits sub-priority(0-3). */
    NVIC_SetPriorityGrouping(5);

    init_spi_pin();

    init_spi();

    NVIC_SetPriority(DMA_IRQn, 2);
    NVIC_EnableIRQ(DMA_IRQn);
    memset(g_dma0_ctrl_base, 0x0, sizeof(g_dma0_ctrl_base));
    dma_send_tx();
    dma_recv_rx();

    while (1)
    {
        memcpy(g_send_buf, g_recv_buf, g_recv_len);
        md_gpio_set_pin_low(GPIO_PORT, SPI_NSS_PIN);
        /* Enable DMA channel */
        md_dma_enable_channel(MD_DMA_CH_1);
        md_dma_enable_channel(MD_DMA_CH_0);
        /* enable dma rx */
        md_spi_enable_rxdma(SPI1);

        g_complete = 0;
        /* enable dma tx */
        md_spi_enable_txdma(SPI1);

        cnt = 4000;

        while (!g_complete && (--cnt));

        md_gpio_set_pin_high(GPIO_PORT, SPI_NSS_PIN);

        md_delay_1ms(1000);
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
