/**********************************************************************************
 *
 * @file    wm8978.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ----------------------------------------------------------------- */

#include "wm8978.h"
#include "stdio.h"
/* Private Macros ----------------------------------------------------------- */

/* Private Variables--------------------------------------------------------- */
static void *wm_i2c = NULL;
static uint16_t s_WM8978_REGVAL[58] =
{
    0X0000, 0X0000, 0X0000, 0X0000, 0X0050, 0X0000, 0X0140, 0X0000,
    0X0000, 0X0000, 0X0000, 0X00FF, 0X00FF, 0X0000, 0X0100, 0X00FF,
    0X00FF, 0X0000, 0X012C, 0X002C, 0X002C, 0X002C, 0X002C, 0X0000,
    0X0032, 0X0000, 0X0000, 0X0000, 0X0000, 0X0000, 0X0000, 0X0000,
    0X0038, 0X000B, 0X0032, 0X0000, 0X0008, 0X000C, 0X0093, 0X00E9,
    0X0000, 0X0000, 0X0000, 0X0000, 0X0003, 0X0010, 0X0010, 0X0100,
    0X0100, 0X0002, 0X0001, 0X0001, 0X0039, 0X0039, 0X0039, 0X0039,
    0X0001, 0X0001
};

/* Public Variables -------------------------------------------------------- */

/* Private Constants -------------------------------------------------------- */

/* Private function prototypes ---------------------------------------------- */

/* Private Function---------------------------------------------------------- */

uint8_t WM8978_Write_Reg(uint8_t reg, uint16_t val)
{
    uint8_t res;
    uint8_t udata[2];
    udata[0] = (reg << 1) | ((uint8_t)((val >> 8) & 0x01));
    udata[1] = (uint8_t)val;
    res = WM_I2C_WRITE(wm_i2c, WM8978_ADDR, udata, 2);

    if (!res)
        s_WM8978_REGVAL[reg] = val;
    else
        printf("I2C write failed,res=%d\r\n", res);

    return res;
}

uint16_t WM8978_Read_Reg(uint8_t reg)
{
    return s_WM8978_REGVAL[reg];
}

uint8_t WM8978_Init(void *i2c_dev)
{
    /*The I2C bus must be finished before the call this function to initialize*/
    wm_i2c = i2c_dev;

    if (!wm_i2c)
        return 1;

    uint8_t Res;
    Res = WM8978_Write_Reg(0, 0);                       /*Soft reset WM8978*/

    if (Res)
        return 1;                                       /*Send instructions, failure WM8978 anomalies*/

    WM8978_HPvol_Set(0, 0);
    WM8978_SPKvol_Set(0);/* 50 */
    WM8978_Write_Reg(1, 0X1B);                          /*R1, MICEN is set to 1 (MIC), BIASEN is set to 1 (simulator, VMIDSEL (1-0) is set to: 11 (5 k)*/
    WM8978_Write_Reg(2, 0X1B0);                         /*R2, ROUT1 LOUT1 output enabled (headphones can work), BOOSTENR, BOOSTENL can make*/
    WM8978_Write_Reg(3, 0X6C);                          /*R3, LOUT2 and ROUT2 output can work (horn), RMIX, LMIX can make*/
    WM8978_Write_Reg(6, 0);                             /*R6, MCLK is provided by external*/
    WM8978_Write_Reg(43, 1 << 4);                       /*R43 INVROUT2 reverse, drive the horn*/
    WM8978_Write_Reg(47, 1 << 8);                       /*R47 Settings, PGABOOSTL, left channel MIC gain 20 times*/
    WM8978_Write_Reg(48, 1 << 8);                       /*R48 Settings, PGABOOSTR, right channel MIC gain 20 times*/
    WM8978_Write_Reg(49, 1 << 1);                       /*R49 TSDEN, open overheating protection*/
    WM8978_Write_Reg(10, 1 << 3);                       /*R10, SOFTMUTE closed, 128 x sampling, the best SNR*/
    WM8978_Write_Reg(14, 1 << 3);                       /*R14, ADC, 128 x sampling rate*/

    WM8978_I2S_Cfg(2, 0);                               /*Figures do not need to set the I2S interface mode, data set, broadcast from the device does not use*/
    /*The play is set*/
    WM8978_ADDA_Cfg(1, 0);                              /*Open the DAC*/
    WM8978_Input_Cfg(0, 0, 0);                          /*Close the input channel*/
    WM8978_Output_Cfg(1, 0);                            /*Open the DAC output*/
    /*The recording setting
    Open the ADC
    Open input channel (MIC&LINE IN)
    Open the BYPASS output
    MIC gain Settings
    */
    WM8978_HPvol_Set(30, 30);
    WM8978_SPKvol_Set(30);/* 50 */

    return 0;
}
/*WM8978 DAC/ADC configuration
Adc can make adcen: (1)/off (0)
Dacen: dac can make (1)/off (0)
*/
void WM8978_ADDA_Cfg(uint8_t dacen, uint8_t adcen)
{
    uint16_t regval;
    regval = WM8978_Read_Reg(3);                        /*Read the R3*/

    if (dacen)
        regval |= 3 << 0;                                               /*R3 minimum 2 bits set to 1, open DACR&DACL*/
    else
        regval &= ~(3 << 0);                                        /*R3 minimum 2 bits reset, close DACR&DACL.*/

    WM8978_Write_Reg(3, regval);                        /*Set the R3*/
    regval = WM8978_Read_Reg(2);                        /*Read the R2*/

    if (adcen)
        regval |= 3 << 0;                                       /*R2 minimum 2 bits set to 1, open ADCR&ADCL*/
    else
        regval &= ~(3 << 0);                                    /*R2 minimum 2 bits reset, close ADCR&ADCL.*/

    WM8978_Write_Reg(2, regval);                        /*Set the R2*/
}
/*WM8978 input channel configuration
Micen: open MIC (1)/off (0)
Lineinen: Line (1) In open/close (0)
Auxen: aux (1) open/close (0)
*/
void WM8978_Input_Cfg(uint8_t micen, uint8_t lineinen, uint8_t auxen)
{
    uint16_t regval;
    regval = WM8978_Read_Reg(2);                        /*Read the R2*/

    if (micen)
        regval |= 3 << 2;                                               /*Open INPPGAENR INPPGAENL PGA amplification (MIC)*/
    else
        regval &= ~(3 << 2);                                        /*Close the INPPGAENR INPPGAENL.*/

    WM8978_Write_Reg(2, regval);                        /*Set the R2*/
    regval = WM8978_Read_Reg(44);                       /*Read the R44*/

    if (micen)
        regval |= 3 << 4 | 3 << 0;                              /* LIN2INPPGA,LIP2INPGA,RIN2INPPGA,RIP2INPGA. */
    else
        regval &= ~(3 << 4 | 3 << 0);                           /* رLIN2INPPGA,LIP2INPGA,RIN2INPPGA,RIP2INPGA. */

    WM8978_Write_Reg(44, regval);                       /*Set the R44*/

    if (lineinen)
        WM8978_LINEIN_Gain(5);                              /*The LINE IN 0 db gain*/
    else
        WM8978_LINEIN_Gain(0);                              /*Shut down the LINE IN*/

    if (auxen)
        WM8978_AUX_Gain(7);                                     /*AUX 6 db gain*/
    else
        WM8978_AUX_Gain(0);                                     /*Close the AUX input*/
}
/*WM8978 MIC gain Settings (not including the BOOST of 20 db, MIC - > gain of ADC input section)
Gain: 0 ~ 63, corresponding to 12 dB ~ 35.25 dB and 0.75 dB/Step
*/
void WM8978_MIC_Gain(uint8_t gain)
{
    gain &= 0X3F;
    WM8978_Write_Reg(45, gain);                         /*R45, left channel PGA Settings*/
    WM8978_Write_Reg(46, gain | 1 << 8);            /*R46, right channel PGA Settings*/
}
/*WM8978 L2 / R2 (Line In) gain Settings (part of the L2 / R2 -- > ADC input gain)
Gain: 0 ~ 7, 0 say channel ban, 1 ~ 7, corresponding to 12 db ~ 6 db, 3 db/Step
*/
void WM8978_LINEIN_Gain(uint8_t gain)
{
    uint16_t regval;
    gain &= 0X07;
    regval = WM8978_Read_Reg(47);                       /*Read R47*/
    regval &= ~(7 << 4);                                            /*Remove the original Settings*/
    WM8978_Write_Reg(47, regval | gain << 4);   /*Set the R47*/
    regval = WM8978_Read_Reg(48);                       /*Read the R48*/
    regval &= ~(7 << 4);                                            /*Remove the original Settings*/
    WM8978_Write_Reg(48, regval | gain << 4);   /*Set the R48*/
}
/*WM8978 AUXR, AUXL (PWM) audio gain Settings (AUXR/L - > gain of ADC input section)
Gain: 0 ~ 7, 0 say channel ban, 1 ~ 7, corresponding to 12 db ~ 6 db, 3 db/Step
*/
void WM8978_AUX_Gain(uint8_t gain)
{
    uint16_t regval;
    gain &= 0X07;
    regval = WM8978_Read_Reg(47);                       /*Read R47*/
    regval &= ~(7 << 0);                                            /*Remove the original Settings*/
    WM8978_Write_Reg(47, regval | gain << 0);   /*Set the R47*/
    regval = WM8978_Read_Reg(48);                       /*Read the R48*/
    regval &= ~(7 << 0);                                            /*Remove the original Settings*/
    WM8978_Write_Reg(48, regval | gain << 0);   /*Set the R48*/
}

/*WM8978 output configuration
Dacen: DAC output (sound) (1) open/close (0)
Bpsen: Bypass output (recording, including the MIC, the LINE IN, AUX, etc.) (1) open/close (0)
*/
void WM8978_Output_Cfg(uint8_t dacen, uint8_t bpsen)
{
    uint16_t regval = 0U;

    if (dacen)
        regval |= 1 << 0;                                               /*The DAC output can make*/

    if (bpsen)
    {
        regval |= 1 << 1;                                               /*BYPASS can make*/
        regval |= 5 << 2;                                               /*0 db gain*/
    }

    WM8978_Write_Reg(50, regval);                       /*R50 set*/
    WM8978_Write_Reg(51, regval);                       /*R51 set*/
}
/*Set the left and right channel headphone volume
Voll: left sound channel volume (0 ~ 63)
Volr: right sound channel volume (0 ~ 63)
*/
void WM8978_HPvol_Set(uint8_t voll, uint8_t volr)
{
    voll &= 0X3F;
    volr &= 0X3F;                                                       /*Limit the scope of*/

    if (voll == 0)voll |= 1 << 6;                           /*The volume of 0,.mute directly*/

    if (volr == 0)volr |= 1 << 6;                           /*The volume of 0,.mute directly*/

    WM8978_Write_Reg(52, voll);                         /*R52, left sound channel headphone volume Settings*/
    WM8978_Write_Reg(53, volr | (1 << 8));          /*R53, right channel headphone volume Settings, synchronous update (HPVU = 1)*/
}
/*Set the speaker volume
Voll: left sound channel volume (0 ~ 63)
*/
void WM8978_SPKvol_Set(uint8_t volx)
{
    volx &= 0X3F; /*Limit the scope of*/

    if (volx == 0)volx |= 1 << 6;                           /*The volume of 0,.mute directly*/

    WM8978_Write_Reg(54, volx);                         /*R54, horn left sound channel volume Settings*/
    WM8978_Write_Reg(55, volx | (1 << 8));          /*R55, horn right channel volume Settings, synchronous update (SPKVU = 1)*/
}
/*Set the I2S working mode
3, PCM/DSP;
3, 32 bit;
*/
void WM8978_I2S_Cfg(uint8_t fmt, uint8_t len)
{
    fmt &= 0x03;
    len &= 0x03;                                                        /*Limit the scope of*/
    WM8978_Write_Reg(4, (fmt << 3) | (len << 5)); /*R4, WM8978 work mode setting*/
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
