/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ----------------------------------------------------------------- */

#include <string.h>
#include "main.h"
#include <ald_i2c.h>
#include "wm8978.h"
#include "sample_wav.h"
#include "stdio.h"
#include "bsp_uart.h"
#include "wm_i2s.h"
/* Private Macros ----------------------------------------------------------- */

#define U2C4(X) (X)&0xFF,(X>>8)&0xFF,(X>>16)&0xFF,(X>>24)&0xFF
/* Private Variables--------------------------------------------------------- */

typedef struct
{
    uint32_t   ChunkID;       /* 0 */
    uint32_t   FileSize;      /* 4 */
    uint32_t   FileFormat;    /* 8 */

    uint32_t   SubChunk1ID;   /* 12 */
    uint32_t   SubChunk1Size; /* 16*/
    uint16_t   AudioFormat;   /* 20 */
    uint16_t   NbrChannels;   /* 22 */
    uint32_t   SampleRate;    /* 24 */

    uint32_t   ByteRate;      /* 28 */
    uint16_t   BlockAlign;    /* 32 */
    uint16_t   BitPerSample;  /* 34 */
    uint32_t   SubChunk2ID;   /* 36 */
    uint32_t   SubChunk2Size; /* 40 */

} wave_formattypedef_t;
#define WAVF_RIFF 0x46464952
#define WAVF_WAVE 0x45564157
#define WAVF_fmt  0x20746D66
#define WAVF_data 0x61746164

/* Public Variables -------------------------------------------------------- */

/* Private Constants -------------------------------------------------------- */
static i2c_handle_t g_hi2c;
extern uart_handle_t g_h_uart;
/* Private function prototypes ---------------------------------------------- */

void print_wav_info(wave_formattypedef_t *wav);
void i2s0_pin_init(void);
void i2c0_init(void);
int sd_test(void);
/* Private Function---------------------------------------------------------- */

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/**
  * @brief  Initializate pin of spi module.
  * @retval None
  */

/* DMA使用的双缓冲区 4K bytes*/
int16_t audio_dma_buffer[2 * 1024] __attribute((aligned(4)));
/* 录音数据暂存区 64K bytes*/
int16_t audio_record_buffer[32 * 1024] __attribute((aligned(4)));

/* 发送信息 */
const uint8_t *g_audio_data;
uint32_t g_audio_length;
uint32_t g_current_pos = 0;
uint32_t audio_play_channels;

/* 接收缓存 */
uint8_t *record_buffer;
uint32_t record_size;
uint32_t record_length = 0;
/* NOTE 缓冲区的size 单位是int16而不是byte */

/* Demo控制用的数据 */
uint8_t audio_volume = 30;
volatile int record_full = 0;
uint32_t wm8978_status = 0;

uint16_t audio_send_data(void *buf, uint16_t length)
{
    uint32_t new_pos;

    if (g_current_pos == g_audio_length)
        g_current_pos = 0;

    new_pos = g_current_pos + length;

    if (new_pos > g_audio_length)
        length = (g_audio_length - g_current_pos);

    int16_t *dst = (int16_t *)buf;
    int16_t *src = (int16_t *)(g_audio_data + g_current_pos);

    if (audio_play_channels == 1)
    {
        for (int i = 0; i < length / 2; i++)
        {
            dst[i * 2] = src[i];
            dst[i * 2 + 1] = src[i];
        }
    }
    else
    {
        memcpy(dst, src, length * 2);
    }

    g_current_pos += length;
    return length;
}
uint16_t audio_recv_data(void *buf, uint16_t length)
{
    uint32_t new_pos;
    uint16_t res = length;

    length = length / 2;
    new_pos = record_length + length;

    if (new_pos >= record_size)
    {
        wm_stop();
        length = (record_size - record_length);
        res = 0;
        record_full = 1;
    }

    int16_t *dst = (int16_t *)(record_buffer + record_length);
    int16_t *src = (int16_t *)buf;

    for (int i = 0; i < length; i++)
        dst[i] = src[i * 2];

    record_length += length;

    return res;
}

int play_wav(wave_formattypedef_t *wav)
{
    uint32_t wav_sample_rate = 0;
    uint32_t wav_channels = 0;
    uint32_t wav_audio_fmt = 0;
    uint32_t wav_sample_bits = 0;

    wav_audio_fmt = wav->AudioFormat;
    wav_channels = wav->NbrChannels;
    wav_sample_bits = wav->BitPerSample;
    wav_sample_rate = wav->SampleRate;

    audio_play_channels = wav_channels;

    if (wav->ChunkID != WAVF_RIFF || wav->FileFormat != WAVF_WAVE
            || wav->SubChunk1ID != WAVF_fmt || wav->SubChunk2ID != WAVF_data)
        return -1;

    if (wav_sample_bits != 16)
        return -2;

    if (wav_channels != 1 && wav_channels != 2)
        return -3;

    if (wav_audio_fmt != 1)
        return -4;

    if (wm_set_sample_rate(wav_sample_rate))
        return -5;

    return 0;
}

void test_wm(void)
{
    g_audio_data = (g_sample_wav_data + 44);
    g_audio_length = g_sample_wav_data_size - 44;
    /* 初始化WM8978 */
    i2c0_init();

    if (WM8978_Init(&g_hi2c))
    {
        printf("Error WM8978 init\r\n");
        return ;
    }

    wm8978_status = 1;
    /* 初始化I2S */
    i2s0_pin_init();
    /* 音频初始化信息 */
    struct wm_i2s_info wm_info;
    uint8_t *g_audio_buffer;

    g_audio_buffer = (uint8_t *)audio_dma_buffer;
    wm_info.buffers[0] = g_audio_buffer;
    wm_info.buffers[1] = g_audio_buffer + 1024 * 2;
    wm_info.buffer_length = 1024;
    wm_info.spi_x = SPI0;
    wm_info.feed_data = audio_send_data;
    wm_info.sample_rate = 16000;
    wm_info.dma_tx_ch = 2;
    wm_info.dma_rx_ch = 3;

    if (wm_i2s_init(&wm_info))
    {
        printf("I2S init failed\r\n");
    }

    printf("Audio init ok\r\n");
    audio_play_channels = 1;
    record_buffer = (uint8_t *)audio_record_buffer;
    record_size = sizeof(audio_record_buffer) / 2;
    record_length = 0;
    printf("Record buffer (size:%d)\r\n", record_size);
}

/**
  * @brief  对上位机发来的各种命令的处理
  */
void do_msg(char *msg)
{
    printf("[msg:%s]\r\n", msg);

    if (!strcmp(msg, "play"))
    {
        /* 播放 */
        g_audio_data = g_sample_wav_data;
        g_audio_length = g_sample_wav_data_size;
        g_current_pos = 0;
        wm_set_callback(audio_send_data);
        wm_set_mode(WM_SEND);
        wm_play();
    }
    else if (!strcmp(msg, "pause"))
    {
        /* 暂停 */
        wm_pause();
    }
    else if (!strcmp(msg, "resume"))
    {
        /* 继续 */
        wm_resume();
    }
    else if (!strcmp(msg, "stop"))
    {
        /* 停止 */
        wm_stop();
    }
    else if (!strcmp(msg, "record"))
    {
        /* 录音 */
        record_length = 0;
        wm_set_mode(WM_RECV);
        wm_set_callback(audio_recv_data);
        wm_record();
    }
    else if (!strcmp(msg, "replay"))
    {
        /* 重放内存里的录音 */
        printf("Replay length= %d\r\n", record_length);

        g_audio_data = record_buffer;
        g_audio_length = record_length;
        g_current_pos = 0;
        wm_set_mode(WM_SEND);
        wm_set_callback(audio_send_data);
        wm_play();
    }
    else if (!strcmp(msg, "volh"))
    {
        /* 调高音量 */
        audio_volume = audio_volume >= 90 ? 100 : audio_volume + 10;
        wm_set_vol(0x7, audio_volume);
        printf("Volume = %d \r\n", audio_volume);
    }
    else if (!strcmp(msg, "voll"))
    {
        /* 调低音量 */
        audio_volume = audio_volume <= 10 ? 0 : audio_volume - 10;
        wm_set_vol(0x7, audio_volume);
        printf("Volume = %d \r\n", audio_volume);
    }
    else if (!strcmp(msg, "info") || !strcmp(msg, "status"))
    {
        /* 输出当前信息 */
        printf("WM8978     = %d\r\n", wm8978_status);
        printf("I2S        = %d\r\n", wm_status());
        printf("Current pos= %d\r\n", g_current_pos);
        printf("Record len = %d\r\n", record_length);
    }
    else
    {
        printf("Unknown command\r\n");
    }
}

/**
** @brief 事件循环，处理上位机消息，读取/写入SD卡
*/
void event_loop(void)
{
    char recv_buffer[65];
    uint32_t recv_ptr = 0;

    while (1)
    {
        recv_ptr = bsp_uart_recved(recv_buffer, 64);

        if (recv_ptr)
        {
            recv_buffer[recv_ptr] = 0;

            for (int i = recv_ptr - 1; i >= 0; i--)
            {
                switch (recv_buffer[i])
                {
                    case ' ':
                    case '\n':
                    case '\r':
                    case '\t':
                        recv_buffer[i] = 0;
                        break;

                    default:
                        i = -1;
                }
            }

            do_msg((char *)recv_buffer);
        }

        if (record_full)
        {
            printf("Record buffer full, stopped\r\n");
            record_full = 0;
        }
    }
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Initialize ALD */
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_72M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
    init_uart();
    printf("Startup...\r\n");
    print_wav_info((wave_formattypedef_t *)g_sample_wav_data);
    test_wm();
    ald_delay_ms(100);

    event_loop();
}
/**
  * @brief 输出WAV信息
*/
void print_wav_info(wave_formattypedef_t *wav)
{
    printf("Chunk ID     = %d,%c%c%c%c\r\n", wav->ChunkID, U2C4(wav->ChunkID));
    printf("FileSize     = %d\r\n", wav->FileSize);
    printf("FileFormat   = %d,%c%c%c%c\r\n", wav->FileFormat, U2C4(wav->FileFormat));
    printf("SubChunk1ID  = %d,%c%c%c%c\r\n", wav->SubChunk1ID, U2C4(wav->SubChunk1ID));
    printf("SubChunk1Size= %d\r\n", wav->SubChunk1Size);
    printf("AudioFormat  = %d\r\n", wav->AudioFormat);
    printf("NbrChannels  = %d\r\n", wav->NbrChannels);
    printf("SampleRate   = %d\r\n", wav->SampleRate);
    printf("ByteRate     = %d\r\n", wav->SampleRate);
    printf("BlockAlign   = %d\r\n", wav->BlockAlign);
    printf("BitPerSample = %d\r\n", wav->BitPerSample);
    printf("SubChunk2ID  = %d,%c%c%c%c\r\n", wav->SubChunk2ID, U2C4(wav->SubChunk2ID));
    printf("SubChunk2Size= %d\r\n", wav->SubChunk2Size);
}
void i2s0_pin_init(void)
{
    gpio_init_t x;

    /* Initialize NSS/WS pin */
    x.mode  = GPIO_MODE_OUTPUT;
    x.odos  = GPIO_PUSH_PULL;
    x.pupd  = GPIO_PUSH_DOWN;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_CMOS;
    x.func  = GPIO_FUNC_6;
    ald_gpio_init(I2S_LRCK_PORT, I2S_LRCK_PIN, &x);

    /* Initialize SCK/CK pin */
    x.mode  = GPIO_MODE_OUTPUT;
    x.odos  = GPIO_PUSH_PULL;
    x.pupd  = GPIO_PUSH_DOWN;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_CMOS;
    x.func  = GPIO_FUNC_4;
    ald_gpio_init(I2S_SCLK_PORT, I2S_SCLK_PIN, &x);

    /* Initialize MISO/MCLK pin */
    x.mode  = GPIO_MODE_OUTPUT;
    x.odos  = GPIO_PUSH_PULL;
    x.pupd  = GPIO_PUSH_DOWN;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_CMOS;
    x.func  = GPIO_FUNC_4;
    ald_gpio_init(I2S_MCLK_PORT, I2S_MCLK_PIN, &x);

    /* Initialize MOSI/SD pin */
    x.mode  = GPIO_MODE_OUTPUT;
    x.odos  = GPIO_PUSH_PULL;
    x.pupd  = GPIO_PUSH_DOWN;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_CMOS;
    x.func  = GPIO_FUNC_4;
    ald_gpio_init(I2S_SD_PORT, I2S_SD_PIN, &x);

    return;
}
void i2c0_init(void)
{
    gpio_init_t x;

    /* Initialize SCK pin */
    x.mode  = GPIO_MODE_OUTPUT;
    x.odos  = GPIO_OPEN_DRAIN;
    x.pupd  = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_CMOS;
    x.func  = GPIO_FUNC_5;
    ald_gpio_init(I2C_SCL_PORT, I2C_SCL_PIN, &x);

    /* Initialize SDA pin */
    x.mode  = GPIO_MODE_OUTPUT;
    x.odos  = GPIO_OPEN_DRAIN;
    x.pupd  = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_CMOS;
    x.func  = GPIO_FUNC_5;
    ald_gpio_init(I2C_SDA_PORT, I2C_SDA_PIN, &x);

    memset(&g_hi2c, 0, sizeof(g_hi2c));
    g_hi2c.perh = I2C0;
    g_hi2c.init.module  = I2C_MODULE_MASTER;
    g_hi2c.init.addr_mode    = I2C_ADDR_7BIT;
    g_hi2c.init.clk_speed    = 100000;
    g_hi2c.init.dual_addr    = I2C_DUALADDR_ENABLE;
    g_hi2c.init.general_call = I2C_GENERALCALL_ENABLE;
    g_hi2c.init.no_stretch   = I2C_NOSTRETCH_DISABLE;
    g_hi2c.init.own_addr1    = 0xA0;
    ald_i2c_init(&g_hi2c);
}
/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
