/**
  *********************************************************************************
  *
  * @file    bsp_uart.c
  * @brief   UART for debug printf
  *
  * @version V1.0
  * @date    12 Aug 2021
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Aug 2021     shiwa           The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */


/* Includes ----------------------------------------------------------------- */

#include "bsp_uart.h"
#include "string.h"

/* Private Macros ----------------------------------------------------------- */

/* Public Variables -------------------------------------------------------- */

/* Private Variables--------------------------------------------------------- */
static uint32_t s_uart_enabled = 0;
uart_handle_t g_h_uart;
static uint32_t s_rx_num = 0;
static uint32_t s_rx_cplted = 0;
uint8_t g_rbuf[4 * 64];
uint32_t g_rlen[4];
/* Private Constants -------------------------------------------------------- */

/* Private function prototypes ---------------------------------------------- */

/* Private Function---------------------------------------------------------- */
void bsp_uart_recv(uint8_t *data, uint16_t len, uint16_t timeout)
{
    ald_uart_rx_fifo_config(&g_h_uart, UART_RXFIFO_1BYTE);
    ald_uart_recv_frame_by_it(&g_h_uart, data, len, timeout);
}
uint32_t bsp_uart_recved(char *buf, int length)
{
    uint32_t len = length > g_rlen[s_rx_num] ? g_rlen[s_rx_num] : length;

    if (len)
    {
        memcpy(buf, g_rbuf + s_rx_num * 64, len);
        g_rlen[s_rx_num] = 0;
        s_rx_num++;

        if (s_rx_num >= 4)
            s_rx_num = 0;
    }

    return len;
}
void uart_rx_cplt(uart_handle_t *uart)
{
    g_rlen[s_rx_cplted++] = uart->rx_count;

    if (s_rx_cplted >= 4)
        s_rx_cplted = 0;

    bsp_uart_recv(g_rbuf + s_rx_cplted * 64, 64, 100);
}
/**
  * @brief  Initialize UART for debug output/input
  * @retval None
  */
void init_uart(void)
{
    gpio_init_t x;

    memset(&x, 0, sizeof(x));
    /* Initialize tx pin*/
    x.mode  = GPIO_MODE_OUTPUT;
    x.odos  = GPIO_PUSH_PULL;
    x.pupd  = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_1;
    x.nodrv = GPIO_OUT_DRIVE_1;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_CMOS;
    x.func  = GPIO_FUNC_3;
    ald_gpio_init(M_UART_TX_PORT, M_UART_TX_PIN, &x);

    /* Initialize rx pin*/
    x.mode  = GPIO_MODE_INPUT;
    x.odos  = GPIO_PUSH_PULL;
    x.pupd  = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_1;
    x.nodrv = GPIO_OUT_DRIVE_1;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_CMOS;
    x.func  = GPIO_FUNC_3;
    ald_gpio_init(M_UART_RX_PORT, M_UART_RX_PIN, &x);

    /* Initialize UART */
    memset(&g_h_uart, 0x00, sizeof(uart_handle_t));
    /* Initialize uart */
    g_h_uart.perh             = M_UART;
    g_h_uart.init.baud        = 115200;
    g_h_uart.init.word_length = UART_WORD_LENGTH_8B;
    g_h_uart.init.stop_bits   = UART_STOP_BITS_1;
    g_h_uart.init.parity      = UART_PARITY_NONE;
    g_h_uart.init.mode        = UART_MODE_UART;
    g_h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
    g_h_uart.rx_cplt_cbk = uart_rx_cplt;
    ald_uart_init(&g_h_uart);

    ald_uart_rx_fifo_config(&g_h_uart, UART_RXFIFO_1BYTE);
    ald_uart_tx_fifo_config(&g_h_uart, UART_TXFIFO_EMPTY);
    ald_mcu_irq_config(M_UART_IRQ, 3, 3, ENABLE);

    s_uart_enabled = 1;
    bsp_uart_recv(g_rbuf, 64, 100);
}

/**
  * @brief  Redirect fputc to UART0
  * @param  ch Character to print
  * @param  f Not used
  * @retval None
  */
#ifdef __GNUC__     /* GCC Compiler */
int _write(int file, const void *ptr, int len)
{
    uint16_t cnt;
    (void)file;
    uint8_t *ptr_char = (uint8_t *)ptr;

    int i = 0;

    for (i = 0 ; i < len ; i++)
    {
        if (*ptr_char == '\n')
        {
            cnt = 4000;

            while (!ald_uart_get_status(&g_h_uart, UART_STATUS_TFEMPTY) && (--cnt));

            g_h_uart.perh->TXBUF = '\r';
        }

        cnt = 4000;

        while (!ald_uart_get_status(&g_h_uart, UART_STATUS_TFEMPTY) && (--cnt));

        g_h_uart.perh->TXBUF = *ptr_char++;
    }

    return len;
}

void *_sbrk(intptr_t increment)
{
    extern char end asm("end");
    /* Defined by the linker. */
    static char *heap_end;
    char *prev_heap_end;

    heap_end += increment;

    return (void *)prev_heap_end;
}

void _read()
{

}

void _lseek()
{

}

void _close()
{

}

void _fstat()
{

}

void _isatty()
{

}

void _getpid()
{

}

void _kill()
{

}

void _exit()
{

}

#else
int fputc(int ch, FILE *f)
{
    if (!s_uart_enabled)
        return 0;

    uint8_t x = (uint8_t)ch;
    /* Send one byte */
    ald_uart_send(&g_h_uart, &x, 1, 1000);

    return (ch);
}

#endif /* __GNUC__ */

void M_UART_HANDLER(void)
{
    ald_uart_irq_handler(&g_h_uart);
    return;
}
