/**********************************************************************************
 *
 * @file    spi_flash.c
 * @brief   spi_flash C file
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "md_conf.h"
#include "spi_flash.h"

/* Private Macros ------------------------------------------------------------ */
/* Public Variables ---------------------------------------------------------- */
/* Private Constants --------------------------------------------------------- */

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
/* Private Variables --------------------------------------------------------- */
static md_spi_init_t s_spi;

/* Private function prototypes ----------------------------------------------- */
static void delay(int i);
static void init_spi_pin(void);

/* Private Function ---------------------------------------------------------- */
/**
  * @brief  send byte in poll mode
  * @param  SPIx: Specific spi.
  * @param  data: Specific data.
  * @retval Status.
  */
static int32_t md_spi_send_byte_fast(SPI_TypeDef *SPIx, uint8_t data)
{
    uint16_t cnt = 5000U, temp;

    SPIx->DATA = data;

    while (((SPIx->STAT & SPI_STAT_TXBE_MSK) == 0) && (--cnt));

    cnt = 5000;

    while (((SPIx->STAT & SPI_STAT_RXBNE_MSK) == 0) && (--cnt));

    temp = SPIx->DATA;
    UNUSED(temp);

    return cnt == 0 ? -1 : 0;
}

/**
  * @brief  receive byte in poll mode
  * @param  SPIx: Specific spi.
  * @param  data: Specific data.
  * @retval Status.
  */
static uint8_t md_spi_recv_byte_fast(SPI_TypeDef *SPIx, int *status)
{
    uint16_t cnt = 5000U;

    if (READ_BIT(SPIx->CON1, SPI_CON1_MSTREN_MSK) == SPI_CON1_MSTREN_MSK)
    {
        SPIx->DATA = 0xFF;

        while (((SPIx->STAT & SPI_STAT_TXBE_MSK) == 0) && (--cnt));
    }

    cnt = 5000;

    while (((SPIx->STAT & SPI_STAT_RXBNE_MSK) == 0) && (--cnt));

    *status = cnt == 0 ? -1 : 0;

    return (uint8_t)SPIx->DATA;
}

/**
  * @brief  flash sector erase function
  * @param  addr: Specific address which sector to be erase.
  * @retval Status.
  */
md_status_t erase_flash_sector(uint32_t addr)
{
    uint8_t cmd_buf[4];
    uint8_t i = 0U;

    cmd_buf[0] = FLASH_ERASE;       /*Flash sector erase command*/
    cmd_buf[1] = (addr >> 16) & 0xff;   /*24 bit Flash address*/
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;

    CLR_FLASH_CS();     /*Choose lower, the selected Flash*/

    if (md_spi_send_byte_fast(SPI1, FLASH_WRITE_ENABLE) != MD_OK)  /*First write send enabling instruction*/
    {
        SET_FLASH_CS();
        return MD_ERROR;
    }

    SET_FLASH_CS();    /*Pick up and release of Flash*/

    delay(100);
    CLR_FLASH_CS();

    for (i = 0; i < sizeof(cmd_buf); i++)     /*Send the sector erase instructions and Flash address three bytes*/
    {
        if (md_spi_send_byte_fast(SPI1, cmd_buf[i]) != MD_OK)
        {
            SET_FLASH_CS();
            return MD_ERROR;
        }
    }

    SET_FLASH_CS();

    return MD_OK;
}

/**
  * @brief  transmit an amount of data in blocking mode.
  * @param  addr: Specific address which to be write.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref md_status_t.
  */
md_status_t write_flash(uint32_t addr, char *buf, uint8_t size)
{
    uint8_t cmd_buf[4];
    uint8_t i = 0U;

    if (buf == NULL)
        return MD_ERROR;

    cmd_buf[0] = FLASH_PROGRAM;
    cmd_buf[1] = (addr >> 16) & 0xff;
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;

    CLR_FLASH_CS();     /*Choose lower, the selected Flash*/

    if (md_spi_send_byte_fast(SPI1, FLASH_WRITE_ENABLE) != MD_OK)
    {
        SET_FLASH_CS();
        return MD_ERROR;
    }

    SET_FLASH_CS();    /*Pick up and release of Flash*/

    delay(100);
    CLR_FLASH_CS();

    for (i = 0; i < sizeof(cmd_buf); i++)     /*Send the programming instructions and Flash address three bytes*/
    {
        if (md_spi_send_byte_fast(SPI1, cmd_buf[i]) != MD_OK)
        {
            SET_FLASH_CS();
            return MD_ERROR;
        }
    }

    for (i = 0; i < size; i++)  /*To write the data sent to the Flash*/
    {
        if (md_spi_send_byte_fast(SPI1, buf[i]) != MD_OK)
        {
            SET_FLASH_CS();
            return MD_ERROR;
        }
    }

    SET_FLASH_CS();

    return MD_OK;
}

/**
  * @brief  Receive an amount of data in blocking mode.
  * @param  addr: address of flash where want to read.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be received
  * @retval Status, see @ref md_status_t.
  */
md_status_t read_flash(uint32_t addr, char *buf, uint16_t size)
{
    uint8_t cmd_buf[4];
    uint8_t i = 0U;
    int r_flag = 0;

    if (buf == NULL)
        return MD_ERROR;

    cmd_buf[0] = FLASH_READ;
    cmd_buf[1] = (addr >> 16) & 0xff;
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;

    CLR_FLASH_CS();     /*Choose lower, the selected Flash*/

    for (i = 0; i < sizeof(cmd_buf); i++)   /*Send the editor & reader instructions and Flash address three bytes*/
    {
        if (md_spi_send_byte_fast(SPI1, cmd_buf[i]) != MD_OK)
        {
            SET_FLASH_CS();     /*Pick up and release of Flash*/
            return MD_ERROR;
        }
    }

    for (i = 0; i < size; i++)  /*Send the programming instructions and Flash address three bytes*/
    {
        buf[i] = md_spi_recv_byte_fast(SPI1, &r_flag);

        if (r_flag != MD_OK)
        {
            SET_FLASH_CS();
            return MD_ERROR;
        }
    }

    SET_FLASH_CS();

    return MD_OK;
}

/**
  * @brief  wait until flash unbusy.
  * @retval Status, see @ref md_status_t.
  */
md_status_t wait_flash_unbusy(void)
{
    uint8_t status;
    int r_flag = 0;

    CLR_FLASH_CS(); /*Choose lower, the selected Flash*/

    if (md_spi_send_byte_fast(SPI1, (uint8_t)FLASH_STATUS) != MD_OK)   /*Send to read status command*/
    {
        SET_FLASH_CS();     /*Pick up and release of Flash*/
        return MD_ERROR;
    }

    do
    {
        status = md_spi_recv_byte_fast(SPI1, &r_flag);

        if (r_flag != MD_OK)
        {
            SET_FLASH_CS();
            return MD_ERROR;
        }
    }
    while (status & 0x01);

    SET_FLASH_CS();

    return MD_OK;
}

/**
  * @brief  Read flash id in blocking mode.
  * @retval flash id.
  */
uint32_t read_flash_id(void)
{
    uint8_t i;
    int r_flag = 0;
    uint8_t flash_id[4] = {0U};

    flash_id[0] = FLASH_ID;

    CLR_FLASH_CS(); /*Choose lower, the selected Flash*/

    for (i = 0; i < sizeof(flash_id); i++)
    {
        if (md_spi_send_byte_fast(SPI1, flash_id[i]) != MD_OK)
        {
            SET_FLASH_CS();     /*Pick up and release of Flash*/
            return MD_ERROR;
        }
    }

    for (i = 0; i < 3; i++)
    {
        flash_id[i] = md_spi_recv_byte_fast(SPI1, &r_flag);

        if (r_flag != MD_OK)
        {
            SET_FLASH_CS();
            return MD_ERROR;
        }
    }

    SET_FLASH_CS();

    return ((flash_id[0] << 16) | (flash_id[1] << 8) | (flash_id[2]));  /*Manufacturer ID flash_id [0] and device ID flash_id [1]*/
}

/**
  * @brief spi function
  * @retval None.
  */
void init_mcu_spi(void)
{
    init_spi_pin();

    s_spi.mode      = MD_SPI_MODE_MASTER;   /* SPI host mode */
    s_spi.baud      = MD_SPI_BAUD_8;        /* fclk / 8 */
    s_spi.data_size = MD_SPI_DATA_SIZE_8;   /* 8 bit pattern */
    s_spi.polarity  = MD_SPI_CPOL_HIGH;     /* Free high level */
    s_spi.phase     = MD_SPI_CPHA_SECOND;   /* The second edge receiving data */
    s_spi.first_bit = MD_SPI_FIRSTBIT_MSB;  /* Send the MSB first */
    s_spi.dir       = MD_SPI_DIRECTION_2LINES;
    s_spi.ss_en     = DISABLE;
    s_spi.crc_calc  = DISABLE;
    s_spi.crc_poly  = 0;
    md_spi_init(SPI1, &s_spi);              /*According to the parameter initialization SPI peripherals*/

    return;
}

/**
  * @brief  Initializate spi flash pin
  * @retval None.
  */
static void init_spi_pin(void)
{
    md_gpio_init_t gpio_config;

    gpio_config.type  = MD_GPIO_TYPE_CMOS;
    gpio_config.odos  = MD_GPIO_PUSH_PULL;
    gpio_config.pupd  = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_config.func  = MD_GPIO_FUNC_1;
    gpio_config.mode  = MD_GPIO_MODE_OUTPUT;
    md_gpio_init(SPI_NSS_PORT, SPI_NSS_PIN, &gpio_config);    /* Initialize NSS Pin */
    SET_FLASH_CS();   /*Choose the output high, release of Flash*/

    gpio_config.func  = MD_GPIO_FUNC_6;
    md_gpio_init(SPI_SCK_PORT, SPI_SCK_PIN, &gpio_config);    /* Initialize SCK Pin */

    md_gpio_init(SPI_MOSI_PORT, SPI_MOSI_PIN, &gpio_config);    /* Initialize MOSI pin */

    gpio_config.mode  = MD_GPIO_MODE_INPUT;
    md_gpio_init(SPI_MISO_PORT, SPI_MISO_PIN, &gpio_config);    /* Initialize MISO pins */
    return;
}

/**
  * @brief  delay some time.
  * @retval None.
  */
static void delay(int i)
{
    while (i--) ;
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
