/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for demo
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

#define HIGH_SECOND     /*The clock polarity and clock phase*/

/* Private Variables --------------------------------------------------------- */
static md_dma_descriptor_t s_dma0_ctrl_base[2]__attribute__((aligned(512)));
uint8_t g_xfer_buf[20] = {0U};
uint8_t g_xfer_len = sizeof(g_xfer_buf);

/* Public Variables ---------------------------------------------------------- */
md_dma_config_t g_tx_config, g_rx_config;


/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples_SPI
  * @{
  */

/*From machine mode communication pins initialization function*/
void init_spi_pin(void)
{
    md_gpio_init_t gpio_config;

    /*From machine initialization, SCK pin configuration for the input mode*/
    gpio_config.mode = MD_GPIO_MODE_INPUT;
    gpio_config.odos = MD_GPIO_PUSH_PULL;
    gpio_config.pupd = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_config.type = MD_GPIO_TYPE_CMOS;
    gpio_config.func = MD_GPIO_FUNC_6;
    md_gpio_init(SPI_SCK_PORT, SPI_SCK_PIN, &gpio_config);

    /*From machine initialization MISO pins, configured to output mode*/
    gpio_config.mode = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos = MD_GPIO_PUSH_PULL;
    gpio_config.pupd = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_config.type = MD_GPIO_TYPE_CMOS;
    gpio_config.func = MD_GPIO_FUNC_6;
    md_gpio_init(SPI_MISO_PORT, SPI_MISO_PIN, &gpio_config);

    /*From machine initialization, MOSI pin configured as input mode*/
    gpio_config.mode = MD_GPIO_MODE_INPUT;
    gpio_config.odos = MD_GPIO_PUSH_PULL;
    gpio_config.pupd = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_config.type = MD_GPIO_TYPE_CMOS;
    gpio_config.func = MD_GPIO_FUNC_6;
    md_gpio_init(SPI_MOSI_PORT, SPI_MOSI_PIN, &gpio_config);

    /*From machine initialization, NSS pin configured as input mode*/
    gpio_config.mode = MD_GPIO_MODE_INPUT;
    gpio_config.odos = MD_GPIO_PUSH_PULL;
    gpio_config.pupd = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_config.type = MD_GPIO_TYPE_CMOS;
    gpio_config.func = MD_GPIO_FUNC_6;
    md_gpio_init(SPI_NSS_PORT, SPI_NSS_PIN, &gpio_config);

    return;
}

/*SPI configuration function*/
void md_spi_struct_user(md_spi_init_t *init)
{
    init->mode      = MD_SPI_MODE_SLAVER;       /*From the machine model*/
    init->dir       = MD_SPI_DIRECTION_2LINES;      /*Full duplex*/
    init->data_size = MD_SPI_DATA_SIZE_8;
    init->baud      = MD_SPI_BAUD_2;

#ifdef HIGH_FIRST       /*Free clock high level, the first clock edge sampling data*/
    init->phase     = MD_SPI_CPHA_FIRST;
    init->polarity  = MD_SPI_CPOL_HIGH;

#elif defined HIGH_SECOND       /*Free clock high level, the second clock edge sampling data*/
    init->phase     = MD_SPI_CPHA_SECOND;
    init->polarity  = MD_SPI_CPOL_HIGH;

#elif defined LOW_FIRST     /*Free clock low level, the first clock edge sampling data*/
    init->phase     = MD_SPI_CPHA_FIRST;
    init->polarity  = MD_SPI_CPOL_LOW;

#elif defined LOW_SECOND        /*Free clock low level, the second clock edge sampling data*/
    init->phase     = MD_SPI_CPHA_SECOND;
    init->polarity  = MD_SPI_CPOL_LOW;
#endif  /*HIGH_FIRST*/

    init->first_bit = MD_SPI_FIRSTBIT_MSB;      /*High in the former*/
    init->ss_en     = DISABLE;
    init->crc_calc  = DISABLE;
    init->crc_poly  = 0;
    return;
}

void init_mcu_spi(void)
{
    md_spi_init_t spi_config;
    
    /*Empty the SPI initialization structure*/
    memset(&spi_config, 0x0, sizeof(spi_config));
    /*The SPI initialization structure is set to the default values*/
    md_spi_struct_init(&spi_config);
    /*User configuration SPI*/
    md_spi_struct_user(&spi_config);
    /*SPI initialization*/
    md_spi_init(SPI1, &spi_config);
    
    init_spi_pin();
    
    return;
}

/*The DMA TX initialization function*/
void config_dma_tx(void)
{
    memset(&g_tx_config, 0x0, sizeof(g_tx_config));

    /*To enable the DMA*/
    md_dma_enable();

    /*User configuration DMA*/
    md_dma_config_struct(&g_tx_config);
    g_tx_config.data_width = MD_DMA_DATA_SIZE_BYTE;
    g_tx_config.src_inc    = MD_DMA_DATA_INC_BYTE;       /*The source data address byte increments*/
    g_tx_config.dst_inc    = MD_DMA_DATA_INC_NONE;       /*The target data address no increment*/
    g_tx_config.R_power    = MD_DMA_R_POWER_1;
    g_tx_config.primary    = ENABLE;
    g_tx_config.burst      = DISABLE;
    g_tx_config.high_prio  = DISABLE;
    g_tx_config.interrupt  = ENABLE;
    g_tx_config.channel    = 0;
    g_tx_config.src        = (void *)(g_xfer_buf);          /*The source data*/
    g_tx_config.dst        = (void *)(&SPI1->DATA);         /*The target data*/
    g_tx_config.size       = g_xfer_len;                    /*The length of the data*/
    g_tx_config.msel       = MD_DMA_MSEL_SPI1;
    g_tx_config.msigsel    = MD_DMA_MSIGSEL_SPI_TXEMPTY;       /*SPI0 send*/
    md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_BASIC, &g_tx_config);
    md_dma_clear_flag_done(MD_DMA_CH_0);

    /*Enable the DMA channel*/
    md_dma_enable_channel(MD_DMA_CH_0);
    /*Enabling SPI DMA to send*/
    md_spi_enable_tx_dma(SPI1);

    return;
}

/*DMA_RX initialization function*/
void config_dma_rx(void)
{
    memset(&g_rx_config, 0x0, sizeof(g_rx_config));

    /*To enable the DMA*/
    md_dma_enable();

    /*User configuration DMA*/
    md_dma_config_struct(&g_rx_config);
    g_rx_config.data_width = MD_DMA_DATA_SIZE_BYTE;
    g_rx_config.src_inc    = MD_DMA_DATA_INC_NONE;       /*Source data is no address increment*/
    g_rx_config.dst_inc    = MD_DMA_DATA_INC_BYTE;       /*The target data address byte increments*/
    g_rx_config.R_power    = MD_DMA_R_POWER_1;
    g_rx_config.primary    = ENABLE;
    g_rx_config.burst      = DISABLE;
    g_rx_config.high_prio  = DISABLE;
    g_rx_config.interrupt  = ENABLE;
    g_rx_config.channel    = 1;
    g_rx_config.src        = (void *)(&SPI1->DATA);         /*The source data*/
    g_rx_config.dst        = (void *)(g_xfer_buf);          /*The target data*/
    g_rx_config.size       = g_xfer_len;                    /*The length of the data*/
    g_rx_config.msel       = MD_DMA_MSEL_SPI1;
    g_rx_config.msigsel    = MD_DMA_MSIGSEL_SPI_RNR;       /*SPI0 receive*/
    md_dma_config_base(DMA0, MD_DMA_CYCLE_CTRL_BASIC, &g_rx_config);
    md_dma_clear_flag_done(MD_DMA_CH_1);

    /*Enable the DMA channel*/
    md_dma_enable_channel(MD_DMA_CH_1);
    /*Enabling SPI DMA*/
    md_spi_enable_rx_dma(SPI1);
    return;
}


/*The main function*/
int main()
{
    /*Configure the system clock*/
    md_cmu_clock_config(MD_CMU_CLOCK_HRC, 24000000);
    /*Initialize the SysTick interrupt*/
    md_init_1ms_tick();

    /* Enable SPI clock */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_spi1();
    SYSCFG_LOCK();

    /* SPI initialization*/
    init_mcu_spi();

    /* Config DMA */
    NVIC_SetPriority(DMA_IRQn, 2);
    NVIC_EnableIRQ(DMA_IRQn);
    memset(s_dma0_ctrl_base, 0x0, sizeof(s_dma0_ctrl_base));
    md_dma_set_ctrlbase((uint32_t)&s_dma0_ctrl_base);
    config_dma_tx();
    config_dma_rx();

    while (1)
    {
        md_delay_1ms(1000);
    }
}

/**
  * @} Examples_SPI
  */

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
