/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for demo
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

#define HIGH_SECOND     /*The clock polarity and clock phase*/

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */
uint8_t g_send_buf[20] = {0U};
uint8_t g_recv_buf[20] = {0U};

volatile uint8_t g_complete_flag;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/*Host mode communication pins initialization function*/
void init_spi_pin(void)
{
    md_gpio_init_t gpio_config;

    /*Host SCK pin initialization, configuration for the output mode*/
    gpio_config.mode = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos = MD_GPIO_PUSH_PULL;
    gpio_config.pupd = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_config.type = MD_GPIO_TYPE_CMOS;
    gpio_config.func = MD_GPIO_FUNC_6;
    md_gpio_init(SPI_SCK_PORT, SPI_SCK_PIN, &gpio_config);

    /*Host MISO pins initialization, configuration for the input mode*/
    gpio_config.mode = MD_GPIO_MODE_INPUT;
    gpio_config.odos = MD_GPIO_PUSH_PULL;
    gpio_config.pupd = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_config.type = MD_GPIO_TYPE_CMOS;
    gpio_config.func = MD_GPIO_FUNC_6;
    md_gpio_init(SPI_MISO_PORT, SPI_MISO_PIN, &gpio_config);

    /*Host MOSI pin initialization, configuration for the output mode*/
    gpio_config.mode = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos = MD_GPIO_PUSH_PULL;
    gpio_config.pupd = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_config.type = MD_GPIO_TYPE_CMOS;
    gpio_config.func = MD_GPIO_FUNC_6;
    md_gpio_init(SPI_MOSI_PORT, SPI_MOSI_PIN, &gpio_config);

    /*Host NSS pin initialization, configuration for the output mode*/
    gpio_config.mode = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos = MD_GPIO_PUSH_PULL;
    gpio_config.pupd = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_config.type = MD_GPIO_TYPE_CMOS;
    gpio_config.func = MD_GPIO_FUNC_1;        /* GPIO */
    md_gpio_init(SPI_NSS_PORT, SPI_NSS_PIN, &gpio_config);
    md_gpio_set_pin_high(SPI_NSS_PORT, SPI_NSS_PIN);
    
    return;
}

/*SPI configuration function*/
void spi_struct_user(md_spi_init_t *init)
{
    init->mode      = MD_SPI_MODE_MASTER;       /*Host mode*/
    init->dir       = MD_SPI_DIRECTION_2LINES;      /*Full duplex*/
    init->data_size = MD_SPI_DATA_SIZE_8;
    init->baud      = MD_SPI_BAUD_32;

#ifdef HIGH_FIRST       /*Free clock high level, the first clock edge sampling data*/
    init->phase     = MD_SPI_CPHA_FIRST;
    init->polarity  = MD_SPI_CPOL_HIGH;

#elif defined HIGH_SECOND       /*Free clock high level, the second clock edge sampling data*/
    init->phase     = MD_SPI_CPHA_SECOND;
    init->polarity  = MD_SPI_CPOL_HIGH;

#elif defined LOW_FIRST     /*Free clock low level, the first clock edge sampling data*/
    init->phase     = MD_SPI_CPHA_FIRST;
    init->polarity  = MD_SPI_CPOL_LOW;

#elif defined LOW_SECOND        /*Free clock low level, the second clock edge sampling data*/
    init->phase     = MD_SPI_CPHA_SECOND;
    init->polarity  = MD_SPI_CPOL_LOW;
#endif   /* HIGH_FIRST */

    init->first_bit = MD_SPI_FIRSTBIT_MSB;      /*High in the former*/
    init->ss_en     = DISABLE;
    init->crc_calc  = DISABLE;
    init->crc_poly  = 0;
    
    return;
}

void init_mcu_spi(void)
{
    md_spi_init_t spi_config;
    /*Empty the SPI initialization structure*/
    memset(&spi_config, 0, sizeof(spi_config));
    /*The SPI initialization structure is set to the default values*/
    md_spi_struct_init(&spi_config);
    /*User configuration SPI*/
    spi_struct_user(&spi_config);
    /*SPI initialization*/
    md_spi_init(SPI1, &spi_config);

    init_spi_pin();
    
    /*Make to receive the air break*/
    md_spi_enable_rxne_interrupt(SPI1);
    
    /*SPI interrupt initialization: set the interrupt priority and can interrupt*/
    NVIC_SetPriority(SPI1_IRQn, 0);
    NVIC_EnableIRQ(SPI1_IRQn);
    return;
}

/*The main function*/
int main()
{
    /*Configure the system clock*/
    md_cmu_clock_config(MD_CMU_CLOCK_HRC, 24000000);

    /*Initialize the SysTick interrupt*/
    md_init_1ms_tick();

    /*The SPI clock can make*/
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_spi1();
    SYSCFG_LOCK();

    /* spi initialization*/
    init_mcu_spi();

    while (1)
    {
        memcpy(g_send_buf, g_recv_buf, sizeof(g_send_buf));
        
        g_complete_flag = 0;
        md_gpio_set_pin_low(SPI_NSS_PORT, SPI_NSS_PIN);
        md_spi_enable_txe_interrupt(SPI1);
        while (!g_complete_flag);
        md_gpio_set_pin_high(SPI_NSS_PORT, SPI_NSS_PIN);
      
        md_delay_1ms(3000);
    }
}

/**
  * @} Examples_SPI
  */

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
