/**********************************************************************************
 *
 * @file    spi_flash.c
 * @brief   spi_flash C file
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "spi_flash.h"

/* Private Macros ------------------------------------------------------------ */
/* Public Variables ---------------------------------------------------------- */
/* Private Constants --------------------------------------------------------- */

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */
/* Private Variables --------------------------------------------------------- */
static spi_handle_t s_spi;

/* Private function prototypes ----------------------------------------------- */
static void delay(int i);
static void init_spi_pin(void);

/* Private Function ---------------------------------------------------------- */
/**
  * @brief  flash sector erase function
  * @param  addr: Specific address which sector to be erase.
  * @retval Status.
  */
ald_status_t erase_flash_sector(uint32_t addr)
{
    uint8_t cmd_buf[4];
    uint8_t i = 0U;

    cmd_buf[0] = FLASH_ERASE;       /*Flash sector erase command*/
    cmd_buf[1] = (addr >> 16) & 0xff;   /*24 bit Flash address*/
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;

    CLR_FLASH_CS();     /*Choose lower, the selected Flash*/

    if (ald_spi_send_byte_fast(&s_spi, FLASH_WRITE_ENABLE) != OK)  /*First write send enabling instruction*/
    {
        SET_FLASH_CS();
        return ERROR;
    }

    SET_FLASH_CS();    /*Pick up and release of Flash*/

    delay(100);
    CLR_FLASH_CS();

    for (i = 0; i < sizeof(cmd_buf); i++)     /*Send the sector erase instructions and Flash address three bytes*/
    {
        if (ald_spi_send_byte_fast(&s_spi, cmd_buf[i]) != OK)
        {
            SET_FLASH_CS();
            return ERROR;
        }
    }

    SET_FLASH_CS();

    return OK;
}

/**
  * @brief  transmit an amount of data in blocking mode.
  * @param  addr: Specific address which to be write.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be sent
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t write_flash(uint32_t addr, char *buf, uint8_t size)
{
    uint8_t cmd_buf[4];
    uint8_t i = 0U;

    if (buf == NULL)
        return ERROR;

    cmd_buf[0] = FLASH_PROGRAM;
    cmd_buf[1] = (addr >> 16) & 0xff;
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;

    CLR_FLASH_CS();     /*Choose lower, the selected Flash*/

    if (ald_spi_send_byte_fast(&s_spi, FLASH_WRITE_ENABLE) != OK)
    {
        SET_FLASH_CS();
        return ERROR;
    }

    SET_FLASH_CS();    /*Pick up and release of Flash*/

    delay(100);
    CLR_FLASH_CS();

    for (i = 0; i < sizeof(cmd_buf); i++)     /*Send the programming instructions and Flash address three bytes*/
    {
        if (ald_spi_send_byte_fast(&s_spi, cmd_buf[i]) != OK)
        {
            SET_FLASH_CS();
            return ERROR;
        }
    }

    for (i = 0; i < size; i++)  /*To write the data sent to the Flash*/
    {
        if (ald_spi_send_byte_fast(&s_spi, buf[i]) != OK)
        {
            SET_FLASH_CS();
            return ERROR;
        }
    }

    SET_FLASH_CS();

    return OK;
}

/**
  * @brief  Receive an amount of data in blocking mode.
  * @param  addr: address of flash where want to read.
  * @param  buf: Pointer to data buffer
  * @param  size: Amount of data to be received
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t read_flash(uint32_t addr, char *buf, uint16_t size)
{
    uint8_t cmd_buf[4];
    uint8_t i = 0U;

    if (buf == NULL)
        return ERROR;

    cmd_buf[0] = FLASH_READ;
    cmd_buf[1] = (addr >> 16) & 0xff;
    cmd_buf[2] = (addr >> 8) & 0xff;
    cmd_buf[3] = addr & 0xff;

    CLR_FLASH_CS();     /*Choose lower, the selected Flash*/

    for (i = 0; i < sizeof(cmd_buf); i++)   /*Send the editor & reader instructions and Flash address three bytes*/
    {
        if (ald_spi_send_byte_fast(&s_spi, cmd_buf[i]) != OK)
        {
            SET_FLASH_CS();     /*Pick up and release of Flash*/
            return ERROR;
        }
    }

    for (i = 0; i < size; i++)  /*Send the programming instructions and Flash address three bytes*/
    {
        buf[i] = ald_spi_recv_byte_fast(&s_spi);

    }

    SET_FLASH_CS();

    return OK;
}

/**
  * @brief  wait until flash unbusy.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t wait_flash_unbusy(void)
{
    uint8_t status;
    int r_flag = 0;

    CLR_FLASH_CS(); /*Choose lower, the selected Flash*/

    if (ald_spi_send_byte_fast(&s_spi, (uint8_t)FLASH_STATUS) != OK)   /*Send to read status command*/
    {
        SET_FLASH_CS();     /*Pick up and release of Flash*/
        return ERROR;
    }

    do
    {
        status = ald_spi_recv_byte_fast(&s_spi);

        if (r_flag != OK)
        {
            SET_FLASH_CS();
            return ERROR;
        }
    }
    while (status & 0x01);

    SET_FLASH_CS();

    return OK;
}

/**
  * @brief  Read flash id in blocking mode.
  * @retval flash id.
  */
uint32_t read_flash_id(void)
{
    uint8_t i;
    int r_flag = 0;
    uint8_t flash_id[4] = {0U};

    flash_id[0] = FLASH_ID;

    CLR_FLASH_CS(); /*Choose lower, the selected Flash*/

    for (i = 0; i < sizeof(flash_id); i++)
    {
        if (ald_spi_send_byte_fast(&s_spi, flash_id[i]) != OK)
        {
            SET_FLASH_CS();     /*Pick up and release of Flash*/
            return ERROR;
        }
    }

    for (i = 0; i < 3; i++)
    {
        flash_id[i] = ald_spi_recv_byte_fast(&s_spi);

        if (r_flag != OK)
        {
            SET_FLASH_CS();
            return ERROR;
        }
    }

    SET_FLASH_CS();

    return ((flash_id[0] << 16) | (flash_id[1] << 8) | (flash_id[2]));  /*Manufacturer ID flash_id [0] and device ID flash_id [1]*/
}

/**
  * @brief spi function
  * @retval None.
  */
void init_mcu_spi(void)
{
    init_spi_pin();

    s_spi.perh           = SPI1;           /*Using SPI0*/
    s_spi.init.mode      = SPI_MODE_MASTER;    /*SPI host mode*/
    s_spi.init.baud      = SPI_BAUD_8;         /*clock / 4*/
    s_spi.init.data_size = SPI_DATA_SIZE_8;    /*8 bit pattern*/
    s_spi.init.polarity  = SPI_CPOL_HIGH;      /*Free high level*/
    s_spi.init.phase     = SPI_CPHA_SECOND;    /*The second edge receiving data*/
    s_spi.init.first_bit = SPI_FIRSTBIT_MSB;   /*Send the MSB first*/
    s_spi.init.dir       = SPI_DIRECTION_2LINES;
    s_spi.init.ss_en     = DISABLE;
    s_spi.init.crc_calc  = DISABLE;

    ald_spi_init(&s_spi);  /*According to the parameter initialization SPI peripherals*/

    return;
}

/**
  * @brief  Initializate spi flash pin
  * @retval None.
  */
static void init_spi_pin(void)
{
    gpio_init_t gpio_config;

    /*SCK pin initialization, configuration for the output mode*/
    gpio_config.mode = GPIO_MODE_OUTPUT;
    gpio_config.odos = GPIO_PUSH_PULL;
    gpio_config.pupd = GPIO_PUSH_UP;
    gpio_config.podrv = GPIO_OUT_DRIVE_1;
    gpio_config.nodrv = GPIO_OUT_DRIVE_1;
    gpio_config.flt  = GPIO_FILTER_DISABLE;
    gpio_config.type = GPIO_TYPE_CMOS;
    gpio_config.func = GPIO_FUNC_6;
    ald_gpio_init(SPI_SCK_PORT, SPI_SCK_PIN, &gpio_config);

    /*MISO pins initialization, configuration for the input mode*/
    gpio_config.mode = GPIO_MODE_INPUT;
    gpio_config.odos = GPIO_PUSH_PULL;
    gpio_config.pupd = GPIO_PUSH_UP;
    gpio_config.podrv = GPIO_OUT_DRIVE_1;
    gpio_config.nodrv = GPIO_OUT_DRIVE_1;
    gpio_config.flt  = GPIO_FILTER_DISABLE;
    gpio_config.type = GPIO_TYPE_CMOS;
    gpio_config.func = GPIO_FUNC_6;
    ald_gpio_init(SPI_MISO_PORT, SPI_MISO_PIN, &gpio_config);

    /*Initialization, MOSI pin is configured to output mode*/
    gpio_config.mode = GPIO_MODE_OUTPUT;
    gpio_config.odos = GPIO_PUSH_PULL;
    gpio_config.pupd = GPIO_PUSH_UP;
    gpio_config.podrv = GPIO_OUT_DRIVE_1;
    gpio_config.nodrv = GPIO_OUT_DRIVE_1;
    gpio_config.flt  = GPIO_FILTER_DISABLE;
    gpio_config.type = GPIO_TYPE_CMOS;
    gpio_config.func = GPIO_FUNC_6;
    ald_gpio_init(SPI_MOSI_PORT, SPI_MOSI_PIN, &gpio_config);

    /*CS choose pin initialization, configuration for the output mode*/
    gpio_config.mode = GPIO_MODE_OUTPUT;
    gpio_config.odos = GPIO_PUSH_PULL;
    gpio_config.pupd = GPIO_PUSH_UP;
    gpio_config.podrv = GPIO_OUT_DRIVE_1;
    gpio_config.nodrv = GPIO_OUT_DRIVE_1;
    gpio_config.flt  = GPIO_FILTER_DISABLE;
    gpio_config.type = GPIO_TYPE_CMOS;
    gpio_config.func = GPIO_FUNC_1;
    ald_gpio_init(SPI_NSS_PORT, SPI_NSS_PIN, &gpio_config);
    ald_gpio_write_pin(SPI_NSS_PORT, SPI_NSS_PIN, 1);

    SET_FLASH_CS();
    return;
}

/**
  * @brief  delay some time.
  * @retval None.
  */
static void delay(int i)
{
    while (i--) ;
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
