/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <stdarg.h>
#include <stdio.h>
#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

rtc_time_t g_time;
rtc_date_t g_date;
rtc_alarm_t g_alarm;
uart_handle_t g_h_uart;
char g_uart_buf[128];
uint8_t g_alarm_flag = 0U;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

QSH_VAR_REG(g_alarm_flag, "uint8_t g_alarm_cnt", "u8");

/**
  * @brief  Output debug information via UART.
  * @param  fmt: Varibale parameter
  * @retval None
  */
void printf_e(const char *fmt, ...)
{
    va_list args;

    va_start(args, fmt);
    vsnprintf(g_uart_buf, 128, fmt, args);
    va_end(args);

    ald_uart_send(&g_h_uart, (uint8_t *)g_uart_buf, strlen(g_uart_buf), 1000);
    return;
}

void time_set(uint8_t hour_set, uint8_t minute_set, uint8_t second_set)
{
    memset(&g_time, 0x00, sizeof(rtc_time_t));
    g_time.hour    = hour_set;
    g_time.minute  = minute_set;
    g_time.second  = second_set;
    ald_rtc_set_time(&g_time, RTC_FORMAT_DEC);
}

QSH_FUN_REG(time_set, "void time_set(uint8_t hour_set, uint8_t minute_set, uint8_t second_set)");

void date_set(uint8_t year_set, uint8_t month_set, uint8_t day_set, uint8_t week_set)
{
    memset(&g_date, 0x00, sizeof(rtc_date_t));
    g_date.day     = day_set;
    g_date.month   = month_set;
    g_date.year    = year_set;
    g_date.week    = week_set;
    ald_rtc_set_date(&g_date, RTC_FORMAT_DEC);
}

QSH_FUN_REG(date_set, "void date_set(uint8_t year_set, uint8_t month_set, uint8_t day_set, uint8_t week_set)");


void alarm_set(uint8_t day_set, uint8_t hour_set, uint8_t minute_set, uint8_t second_set)
{
    g_alarm.day          = day_set;
    g_alarm.time.hour    = hour_set;
    g_alarm.time.minute  = minute_set;
    g_alarm.time.second  = second_set;
    ald_rtc_set_alarm(&g_alarm, RTC_FORMAT_DEC);
}

QSH_FUN_REG(alarm_set, "void alarm_set(uint8_t day_set, uint8_t hour_set, uint8_t minute_set, uint8_t second_set)");

void time_now(void)
{
    ald_rtc_get_time(&g_time, RTC_FORMAT_DEC);
    ald_rtc_get_date(&g_date, RTC_FORMAT_DEC);
    printf_e("%d-%d-%d %d:%d:%d ", g_date.year + 2000, g_date.month, g_date.day, g_time.hour, g_time.minute, g_time.second);

    if (g_date.week == 0)
        printf_e("Sun.\r\n");
    else if (g_date.week == 1)
        printf_e("Mon.\r\n");
    else if (g_date.week == 2)
        printf_e("Tues.\r\n");
    else if (g_date.week == 3)
        printf_e("Wed.\r\n");
    else if (g_date.week == 4)
        printf_e("Thur.\r\n");
    else if (g_date.week == 5)
        printf_e("Fri.\r\n");
    else if (g_date.week == 6)
        printf_e("Sat.\r\n");
    else
        printf_e("Error\r\n");
}

QSH_FUN_REG(time_now, "void time_now(void)");

/**
  * @brief  Initialize the UART
  * @retval None
  */
void uart_stdio_init(void)
{
    gpio_init_t x;
    memset(&x, 0x00, sizeof(gpio_init_t));

    /* Initialize tx pin */
    x.mode  = GPIO_MODE_OUTPUT;
    x.odos  = GPIO_PUSH_PULL;
    x.pupd  = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_CMOS;
    x.func  = UART_FUNC_TX;
    ald_gpio_init(UART_PORT_TX, UART_PIN_TX, &x);

    /* Initialize rx pin */
    x.mode  = GPIO_MODE_INPUT;
    x.odos  = GPIO_PUSH_PULL;
    x.pupd  = GPIO_PUSH_UP;
    x.podrv = GPIO_OUT_DRIVE_6;
    x.nodrv = GPIO_OUT_DRIVE_6;
    x.flt   = GPIO_FILTER_DISABLE;
    x.type  = GPIO_TYPE_CMOS;
    x.func  = UART_FUNC_RX;
    ald_gpio_init(UART_PORT_RX, UART_PIN_RX, &x);

    memset(&g_h_uart, 0x00, sizeof(uart_handle_t));
    /* Initialize uart */
    g_h_uart.perh             = UART1;
    g_h_uart.init.baud        = 115200;
    g_h_uart.init.word_length = UART_WORD_LENGTH_8B;
    g_h_uart.init.stop_bits   = UART_STOP_BITS_1;
    g_h_uart.init.parity      = UART_PARITY_NONE;
    g_h_uart.init.mode        = UART_MODE_UART;
    g_h_uart.init.fctl        = UART_HW_FLOW_CTL_DISABLE;
    ald_uart_init(&g_h_uart);

    ald_uart_interrupt_config(&g_h_uart, UART_IT_RFTH, ENABLE);

    return;
}

/**
  * @brief  Test RTC time and date functions
  * @retval Status.
  */
int main(void)
{
    rtc_init_t rtc_init;

    /* Initialize ALD */
    ald_cmu_init();
    /* Configure system clock */
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HRC_6, CMU_PLL1_OUTPUT_48M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
    ald_rtc_source_select(RTC_SOURCE_LOSC);
    /* Enable interrupt */
    ald_mcu_irq_config(RTC_IRQn, 0, 0, ENABLE);
    ald_mcu_irq_config(UART1_IRQn, 1, 1, ENABLE);

    /* Initialize stdio */
    uart_stdio_init();

    printf_e("RTC demo, please enter:\r\n");
    printf_e("  date_set(year,month,day,week)\r\n");
    printf_e("  time_set(hour,minute,second)\r\n");
    printf_e("  alarm_set(day,hour,minute,second)\r\n");
    printf_e("  time_now()\r\n\n");

    memset(&rtc_init, 0x00, sizeof(rtc_init_t));
    /* Initialize RTC */
    rtc_init.asynch_pre_div  = 0;
    rtc_init.synch_pre_div   = 32767;
    rtc_init.hour_format     = RTC_HOUR_FORMAT_24;
    rtc_init.output          = RTC_OUTPUT_DISABLE;
    rtc_init.output_polarity = RTC_OUTPUT_POLARITY_HIGH;
    ald_rtc_init(&rtc_init);

    /* Set current time and date */
    memset(&g_time, 0x00, sizeof(rtc_time_t));
    g_time.hour    = 0;
    g_time.minute  = 0;
    g_time.second  = 0;
    ald_rtc_set_time(&g_time, RTC_FORMAT_DEC);

    memset(&g_date, 0x00, sizeof(rtc_date_t));
    g_date.day     = 1;
    g_date.month   = 1;
    g_date.year    = 1;
    g_date.week    = 1;
    ald_rtc_set_date(&g_date, RTC_FORMAT_DEC);

    /* Set alarm */
    memset(&g_alarm, 0x00, sizeof(rtc_alarm_t));
    g_alarm.idx          = RTC_ALARM_A;
    g_alarm.mask         = RTC_ALARM_MASK_NONE;
    g_alarm.sel          = RTC_SELECT_DAY;
    g_alarm.ss_mask      = RTC_ALARM_SS_MASK_ALL;
    g_alarm.day          = 30;
    g_alarm.time.hour    = 0;
    g_alarm.time.minute  = 0;
    g_alarm.time.second  = 0;
    ald_rtc_set_alarm(&g_alarm, RTC_FORMAT_DEC);

    /* Enable alarm-a interrupt */
    ald_rtc_interrupt_config(RTC_IT_ALMA, ENABLE);

    while (1)
    {
        if (g_alarm_flag == 1)
        {
            ald_rtc_set_wakeup(RTC_WAKEUP_CLOCK_DIV_8, 32767);
            ald_rtc_interrupt_config(RTC_IT_WU, ENABLE);
            ald_pmu_stop1_enter();
        }

        ald_rtc_get_time(&g_time, RTC_FORMAT_DEC);
        ald_rtc_get_date(&g_date, RTC_FORMAT_DEC);
        ald_delay_ms(1000);
    }
}

/**
  * @brief  RTC IRQ handler
  * @retval None
  */
void rtc_irq_handler(void)
{
    if (ald_rtc_get_flag_status(RTC_IF_ALMA) != RESET)
    {
        ald_rtc_clear_flag_status(RTC_IF_ALMA);

        g_alarm_flag = 1;
    }

    if (ald_rtc_get_flag_status(RTC_IF_WU) != RESET)
    {
        ald_rtc_clear_flag_status(RTC_IF_WU);

        ald_rtc_interrupt_config(RTC_IT_WU, DISABLE);

        g_alarm_flag = 0;
    }

    return;
}


/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
