/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

rtc_time_t g_time;
rtc_date_t g_date;

rtc_time_t g_stamp_time;
rtc_date_t g_stamp_date;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

void tamper_pin_init(void)
{
    gpio_init_t GPIO_InitStructure;
    memset(&GPIO_InitStructure, 0x00, sizeof(gpio_init_t));

    /* Initialize PA0_WKUP-TAMPER1_RTC #0 pin */
    GPIO_InitStructure.mode = GPIO_MODE_INPUT;
    GPIO_InitStructure.odos = GPIO_OPEN_SOURCE;
    GPIO_InitStructure.pupd = GPIO_PUSH_UP;
    GPIO_InitStructure.flt  = GPIO_FILTER_DISABLE;
    GPIO_InitStructure.nodrv = GPIO_OUT_DRIVE_6;
    GPIO_InitStructure.podrv = GPIO_OUT_DRIVE_6;
    GPIO_InitStructure.type = GPIO_TYPE_CMOS;
    GPIO_InitStructure.func = GPIO_FUNC_1;

    ald_gpio_init(RTC_TAMPER_PORT, RTC_TAMPER_PIN, &GPIO_InitStructure);
}

void init_rtc(void)
{
    rtc_init_t rtc_init;

    memset(&rtc_init, 0x00, sizeof(rtc_init_t));
    /* Initialize RTC */
    rtc_init.asynch_pre_div  = 0;
    rtc_init.synch_pre_div   = 32767;
    rtc_init.hour_format     = RTC_HOUR_FORMAT_24;
    rtc_init.output          = RTC_OUTPUT_DISABLE;
    rtc_init.output_polarity = RTC_OUTPUT_POLARITY_HIGH;
    ald_rtc_init(&rtc_init);
}

/**
  * @brief  Test RTC time and date functions
  * @retval Status.
  */
int main(void)
{
    rtc_tamper_t tamper;

    /* Initialize ALD */
    ald_cmu_init();
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_48M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
    ald_rtc_source_select(RTC_SOURCE_LOSC);

    /* Enable RTC interrupt */
    ald_mcu_irq_config(RTC_IRQn, 0, 3, ENABLE);

    tamper_pin_init();

    init_rtc();

    /* Set current time and date */
    memset(&g_time, 0x00, sizeof(rtc_time_t));
    g_time.hour    = 9;
    g_time.minute  = 30;
    g_time.second  = 00;
    memset(&g_date, 0x00, sizeof(rtc_date_t));
    g_date.day     = 23;
    g_date.month   = 7;
    g_date.year    = 20;
    g_date.week    = 4;
    ald_rtc_set_time(&g_time, RTC_FORMAT_DEC);
    ald_rtc_set_date(&g_date, RTC_FORMAT_DEC);

    memset(&tamper, 0x00, sizeof(rtc_tamper_t));
    /* Configure tamper parameters */
    tamper.idx  = RTC_TAMPER_1;
    tamper.trig = RTC_TAMPER_TRIGGER_LOW;
    tamper.freq = RTC_TAMPER_SAMPLING_FREQ_256;
    tamper.dur  = RTC_TAMPER_DURATION_1;
    tamper.ts   = ENABLE;                          /*Tamper trigger timestamp*/
    ald_rtc_set_tamper(&tamper);

    ald_rtc_clear_flag_status(RTC_IF_TS);

    /* Configure time stamp parameters */
    ald_rtc_set_time_stamp(RTC_TS_SIGNAL_SEL_TAMPER1, RTC_TS_RISING_EDGE);

    ald_rtc_interrupt_config(RTC_IT_TS, ENABLE);

    while (1)
    {
        ald_rtc_get_time(&g_time, RTC_FORMAT_DEC);
        ald_rtc_get_date(&g_date, RTC_FORMAT_DEC);
        ald_delay_ms(1000);
    }
}

/**
  * @brief  RTC IRQ handler
  * @retval None
  */
void rtc_irq_handler(void)
{
    /* Check and clear interrupt flag */
    if (ald_rtc_get_flag_status(RTC_IF_TS) == SET)
    {
        ald_rtc_clear_flag_status(RTC_IF_TS);
        /* Cancel time stamp */
        ald_rtc_cancel_time_stamp();
        /* Get time stamp */
        ald_rtc_get_time_stamp(&g_stamp_time, &g_stamp_date, RTC_FORMAT_DEC);
    }
}
/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
