/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    17 Sep 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          17 Sep 2021     biyq          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ----------------------------------------------------------------- */

#include <string.h>
#include "main.h"
#include "flm_qspi.h"


/* Private Macros ----------------------------------------------------------- */
#define QSPI_PORT0  GPIOB
#define QSPI_PORT1  GPIOB
#define QSPI_NSS0_PIN   GPIO_PIN_12
#define QSPI_SCK_PIN    GPIO_PIN_13
#define QSPI_IO0_PIN    GPIO_PIN_14
#define QSPI_IO1_PIN    GPIO_PIN_15
#define QSPI_IO2_PIN    GPIO_PIN_10
#define QSPI_IO3_PIN    GPIO_PIN_11

/* Exported Variables -------------------------------------------------------- */


/* Exported Constants ------------------------------------------------------- */

/* Exported Functions ------------------------------------------------------ */


/* Private Variables--------------------------------------------------------- */

/* Public Variables -------------------------------------------------------- */

/* Private Constants -------------------------------------------------------- */

/* Private function prototypes ---------------------------------------------- */

/* Private Function---------------------------------------------------------- */

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/** @addtogroup Examples
  * @{
  */


/** @addtogroup config flash size parameter
  * @{
  */
void config_flash_param(qspi_handle_t *g_qspi)
{
    qspi_device_size_t l_dsize;

    l_dsize.cs0 = QSPI_NSS_512M;
    l_dsize.cs1 = QSPI_NSS_512M;
    l_dsize.cs2 = QSPI_NSS_512M;
    l_dsize.cs3 = QSPI_NSS_512M;
    l_dsize.addr = 2;       /*有2+1个地址字节*/
    l_dsize.page = 256;     /*页大小*/
    l_dsize.blk = 16;       /*Block大小64KB*/

    ald_qspi_device_size_config(g_qspi, &l_dsize);    /*配置Flash参数*/

    return;
}

/** @addtogroup Delay time
  * @{
  */
static void delay(int i)
{
    while (i--) ;
}

/**
  * @brief  Initializate pin of qspi module.
  * @retval None
  */
void qspi_pin_init(void)
{
    gpio_init_t gpio_config;

    gpio_config.odos  = GPIO_PUSH_PULL;
    gpio_config.pupd  = GPIO_PUSH_UP;
    gpio_config.nodrv = GPIO_OUT_DRIVE_6;
    gpio_config.podrv = GPIO_OUT_DRIVE_6;
    gpio_config.flt   = GPIO_FILTER_DISABLE;
    gpio_config.type  = GPIO_TYPE_TTL;
    gpio_config.func  = GPIO_FUNC_6;

    gpio_config.mode  = GPIO_MODE_OUTPUT;
    ald_gpio_init(QSPI_PORT0, QSPI_NSS0_PIN, &gpio_config); /*初始化PB12为片选管脚*/
    ald_gpio_init(QSPI_PORT0, QSPI_SCK_PIN, &gpio_config); /*初始化PB13为时钟输出管脚*/

    gpio_config.mode = GPIO_MODE_INPUT;
    ald_gpio_init(QSPI_PORT0, QSPI_IO0_PIN, &gpio_config); /*初始化PB14为IO0管脚*/
    ald_gpio_init(QSPI_PORT0, QSPI_IO1_PIN, &gpio_config); /*初始化PB15为IO1管脚*/
    ald_gpio_init(QSPI_PORT1, QSPI_IO2_PIN, &gpio_config); /*初始化PD8为IO2管脚*/
    ald_gpio_init(QSPI_PORT1, QSPI_IO3_PIN, &gpio_config); /*初始化PD9为IO3管脚*/
}

/* BSP for W25Q128FV */
/**
  * @brief  Enable written.
  * @param  hqspi: QSPI handle
  * @retval Status
  */
ald_status_t bsp_w25q128_wr_enable(void)
{
    uint32_t timecnt;
    uint32_t trycounter;

    /* Send write enable command */
    timecnt = 0xFFFFFF;
    MODIFY_REG(QSPI->FCR, QSPI_FCR_OPCODE_MSK | QSPI_FCR_DUMNUM_MSK | QSPI_FCR_RDEN_MSK | QSPI_FCR_WREN_MSK, \
               ((0x06 << 24) | (0 << 7) | (0 << 23) | (0 << 15)));
    SET_BIT(QSPI->FCR, QSPI_FCR_CMDT_MSK);

    do
    {
        delay(100);
    }
    while (READ_BIT(QSPI->FCR, QSPI_FCR_CMDS_MSK) && (timecnt--));

    /* Read register1 to check state */
    timecnt = 0xFFFFFF;
    MODIFY_REG(QSPI->FCR, QSPI_FCR_OPCODE_MSK | QSPI_FCR_DUMNUM_MSK, ((0x05 << 24) | (0 << 7)));
    MODIFY_REG(QSPI->FCR, QSPI_FCR_RDEN_MSK | QSPI_FCR_RDNUM_MSK, ((0x01 << 23) | (0 << 20)));

    do
    {
        SET_BIT(QSPI->FCR, QSPI_FCR_CMDT_MSK);
        trycounter = 0xFFFF;

        while (READ_BIT(QSPI->FCR, QSPI_FCR_CMDS_MSK) && (trycounter--));
    }
    while (((QSPI->FCRLR & 0x01) || !(QSPI->FCRLR & 0x02)) && (timecnt--));

    if (!(QSPI->FCRLR & 0x01) && (QSPI->FCRLR & 0x02))
        return OK;
    else
        return ERROR;
}

/**
  * @brief  Reset the QSPI memory.
  * @param  hqspi: QSPI handle
  * @retval Status
  */
ald_status_t bsp_w25q128_reset(void)
{
    uint32_t timecnt;
    uint32_t trycounter;

    /* Before Erase Sector,must send write enable */
    if (bsp_w25q128_wr_enable() != OK)
        return ERROR;

    /* Send reset enable command */
    timecnt = 0xFFFF;
    MODIFY_REG(QSPI->FCR, QSPI_FCR_OPCODE_MSK | QSPI_FCR_DUMNUM_MSK, ((0x66 << 24) | (0 << 7)));
    SET_BIT(QSPI->FCR, QSPI_FCR_CMDT_MSK);

    while (READ_BIT(QSPI->FCR, QSPI_FCR_CMDS_MSK) && (timecnt--));

    /* Read register1 */
    MODIFY_REG(QSPI->FCR, QSPI_FCR_OPCODE_MSK | QSPI_FCR_DUMNUM_MSK, ((0x05 << 24) | (0 << 7)));
    MODIFY_REG(QSPI->FCR, QSPI_FCR_RDEN_MSK | QSPI_FCR_RDNUM_MSK, ((0x01 << 23) | (0 << 20)));
    SET_BIT(QSPI->FCR, QSPI_FCR_CMDT_MSK);
    timecnt = 0x1FFF;

    do
    {
        SET_BIT(QSPI->FCR, QSPI_FCR_CMDT_MSK);
        trycounter = 0xFFFF;

        while (READ_BIT(QSPI->FCR, QSPI_FCR_CMDS_MSK) && (trycounter--));
    }
    while ((QSPI->FCRLR & 0x01) && (timecnt--));

    if (QSPI->FCRLR & 0x01)
        return ERROR;

    /* Send the reset memory command */
    timecnt = 0xFFFF;
    MODIFY_REG(QSPI->FCR, QSPI_FCR_OPCODE_MSK | QSPI_FCR_DUMNUM_MSK, ((0x99UL << 24) | (0 << 7)));
    SET_BIT(QSPI->FCR, QSPI_FCR_CMDT_MSK);

    while (READ_BIT(QSPI->FCR, QSPI_FCR_CMDS_MSK) && (timecnt--));

    /* Read register1 */
    SET_BIT(QSPI->FCR, QSPI_FCR_CMDT_MSK);
    timecnt = 0x1FFF;

    do
    {
        SET_BIT(QSPI->FCR, QSPI_FCR_CMDT_MSK);
        trycounter = 0xFFFF;

        while (READ_BIT(QSPI->FCR, QSPI_FCR_CMDS_MSK) && (trycounter--));
    }
    while ((QSPI->FCRLR & 0x01) && (timecnt--));

    if (!(QSPI->FCRLR & 0x01))
        return OK;
    else
        return ERROR;
}

/**
  * @brief  Erase Chip
  * @param  hqspi: QSPI handle
  * @retval Status
  */
ald_status_t bsp_w25q128_erase_chip(void)
{
    uint32_t timecnt;
    uint32_t trycounter;

    /* Before Erase Sector,must send write enable */
    if (bsp_w25q128_wr_enable() != OK)
        return ERROR;

    /* Send erase chip command */
    timecnt = 0xFFFF;
    MODIFY_REG(QSPI->FCR, QSPI_FCR_OPCODE_MSK | QSPI_FCR_DUMNUM_MSK | QSPI_FCR_RDEN_MSK | QSPI_FCR_WREN_MSK, \
               ((0xC7UL << 24) | (0 << 7) | (0 << 23) | (0 << 15)));
    SET_BIT(QSPI->FCR, QSPI_FCR_CMDT_MSK);

    while (READ_BIT(QSPI->FCR, QSPI_FCR_CMDS_MSK) && (timecnt--));

    /* Read register1 to check state */
    timecnt = 0xFFFFFFFF;
    MODIFY_REG(QSPI->FCR, QSPI_FCR_OPCODE_MSK | QSPI_FCR_DUMNUM_MSK, ((0x05 << 24) | (0 << 7)));
    MODIFY_REG(QSPI->FCR, QSPI_FCR_RDEN_MSK | QSPI_FCR_RDNUM_MSK, ((0x01 << 23) | (0 << 20)));

    do
    {
        SET_BIT(QSPI->FCR, QSPI_FCR_CMDT_MSK);
        trycounter = 0xFFFF;

        while (READ_BIT(QSPI->FCR, QSPI_FCR_CMDS_MSK) && (trycounter--));

        delay(1000);
    }
    while (((QSPI->FCRLR & 0x01) || (QSPI->FCRLR & 0x02)) && (timecnt--));

    if (!((QSPI->FCRLR & 0x01) || (QSPI->FCRLR & 0x02)))
        return OK;
    else
        return ERROR;
}


/**
  * @brief  Erase block
  * @param  hqspi: QSPI handle
  * @param  blkaddr: Address of block
  * @retval Status
  */
ald_status_t bsp_w25q128_erase_block(uint32_t blkaddr)
{
    uint32_t timecnt;
    uint32_t trycounter;

    /* Before Erase Sector,must send write enable */
    if (bsp_w25q128_wr_enable() != OK)
        return ERROR;

    /* Send erase chip command */
    timecnt = 0xFFFF;
    MODIFY_REG(QSPI->FCR, QSPI_FCR_OPCODE_MSK | QSPI_FCR_DUMNUM_MSK | QSPI_FCR_RDEN_MSK, ((0x20UL << 24) | (0 << 7) | (0 << 23)));
    MODIFY_REG(QSPI->FCR, QSPI_FCR_WREN_MSK | QSPI_FCR_WDNUM_MSK, ((0x01 << 15) | (2 << 12)));
    MODIFY_REG(QSPI->FCWLR, QSPI_FCWLR_CMDDL_MSK, blkaddr);
    SET_BIT(QSPI->FCR, QSPI_FCR_CMDT_MSK);

    while (READ_BIT(QSPI->FCR, QSPI_FCR_CMDS_MSK) && (timecnt--));

    /* Read register1 */
    timecnt = 0xFFFFF;
    MODIFY_REG(QSPI->FCR, QSPI_FCR_OPCODE_MSK | QSPI_FCR_DUMNUM_MSK, ((0x05 << 24) | (0 << 7)));
    MODIFY_REG(QSPI->FCR, QSPI_FCR_RDEN_MSK | QSPI_FCR_RDNUM_MSK, ((0x01 << 23) | (0 << 20)));

    do
    {
        SET_BIT(QSPI->FCR, QSPI_FCR_CMDT_MSK);
        trycounter = 0xFFFF;

        while (READ_BIT(QSPI->FCR, QSPI_FCR_CMDS_MSK) && (trycounter--));

        delay(100);
    }
    while (((QSPI->FCRLR & 0x01) || (QSPI->FCRLR & 0x02)) && (timecnt--));

    if (!((QSPI->FCRLR & 0x01) || (QSPI->FCRLR & 0x02)))
        return OK;
    else
        return ERROR;
}

/**
  * @brief  Enable memory quad mode.
  * @param  hqspi: QSPI handle
  * @retval Status
  */
ald_status_t bsp_w25q128_enable_quad_mode(void)
{
    uint32_t timecnt;
    uint32_t trycounter;
    uint32_t temp;

    /* Read register2 state */
    timecnt = 0xFFFFFF;
    MODIFY_REG(QSPI->FCR, QSPI_FCR_OPCODE_MSK | QSPI_FCR_DUMNUM_MSK, ((0x35 << 24) | (0 << 7)));
    MODIFY_REG(QSPI->FCR, QSPI_FCR_RDEN_MSK | QSPI_FCR_RDNUM_MSK, ((0x01 << 23) | (0 << 20)));
    SET_BIT(QSPI->FCR, QSPI_FCR_CMDT_MSK);

    while (READ_BIT(QSPI->FCR, QSPI_FCR_CMDS_MSK) && (timecnt--));

    /* Write register2 to enable quad */
    timecnt = 0xFFFFFF;
    MODIFY_REG(QSPI->FCR, QSPI_FCR_OPCODE_MSK | QSPI_FCR_DUMNUM_MSK, ((0x31 << 24) | (0 << 7)));
    MODIFY_REG(QSPI->FCR, QSPI_FCR_WREN_MSK | QSPI_FCR_WDNUM_MSK, ((0x01 << 15) | (0 << 12)));
    temp = QSPI->FCRLR;
    temp |= 0x02;
    QSPI->FCWLR = temp;
    SET_BIT(QSPI->FCR, QSPI_FCR_CMDT_MSK);

    while (READ_BIT(QSPI->FCR, QSPI_FCR_CMDS_MSK) && (timecnt--));

    /* Read register1 */
    timecnt = 0xFFFF;
    MODIFY_REG(QSPI->FCR, QSPI_FCR_OPCODE_MSK | QSPI_FCR_DUMNUM_MSK, ((0x05 << 24) | (0 << 7)));
    MODIFY_REG(QSPI->FCR, QSPI_FCR_RDEN_MSK | QSPI_FCR_RDNUM_MSK, ((0x01 << 23) | (0 << 20)));

    do
    {
        SET_BIT(QSPI->FCR, QSPI_FCR_CMDT_MSK);
        trycounter = 0xFFFF;

        while (READ_BIT(QSPI->FCR, QSPI_FCR_CMDS_MSK) && (trycounter--));
    }
    while ((QSPI->FCRLR & 0x02) && (timecnt--));

    if (QSPI->FCRLR & 0x02)
        return ERROR;

    /* Read register2 */
    timecnt = 0xFFFF;
    MODIFY_REG(QSPI->FCR, QSPI_FCR_OPCODE_MSK | QSPI_FCR_DUMNUM_MSK, ((0x35 << 24) | (0 << 7)));
    MODIFY_REG(QSPI->FCR, QSPI_FCR_RDEN_MSK | QSPI_FCR_RDNUM_MSK, ((0x01 << 23) | (0 << 20)));

    do
    {
        SET_BIT(QSPI->FCR, QSPI_FCR_CMDT_MSK);
        trycounter = 0xFFFF;

        while (READ_BIT(QSPI->FCR, QSPI_FCR_CMDS_MSK) && (trycounter--));
    }
    while ((QSPI->FCRLR & 0x02) && (timecnt--));


    if (QSPI->FCRLR & 0x02)
        return OK;
    else
        return ERROR;
}

/**
  * @brief  Disable memory protect mode.
  * @param  hqspi: QSPI handle
  * @retval Status
  */
ald_status_t bsp_w25q128_disable_protect(void)
{
    uint32_t timecnt;
    uint32_t trycounter;

    /* Before Erase Sector,must send write enable */
    if (bsp_w25q128_wr_enable() != OK)
        return ERROR;

    /* Change register to 0 */
    timecnt = 0xFFFFFF;
    MODIFY_REG(QSPI->FCR, QSPI_FCR_OPCODE_MSK | QSPI_FCR_DUMNUM_MSK, ((0x01 << 24) | (0 << 7)));
    MODIFY_REG(QSPI->FCR, QSPI_FCR_WREN_MSK | QSPI_FCR_WDNUM_MSK, ((0x01 << 15) | (0 << 12)));
    QSPI->FCWLR = 0x00;
    SET_BIT(QSPI->FCR, QSPI_FCR_CMDT_MSK);

    while (READ_BIT(QSPI->FCR, QSPI_FCR_CMDS_MSK) && (timecnt--));

    /* Before Erase Sector,must send write enable */
    timecnt = 0xFFFFFF;
    MODIFY_REG(QSPI->FCR, QSPI_FCR_OPCODE_MSK | QSPI_FCR_DUMNUM_MSK, ((0x05 << 24) | (0 << 7)));
    MODIFY_REG(QSPI->FCR, QSPI_FCR_RDEN_MSK | QSPI_FCR_RDNUM_MSK, ((0x01 << 23) | (0 << 20)));

    do
    {
        SET_BIT(QSPI->FCR, QSPI_FCR_CMDT_MSK);
        trycounter = 0xFFFF;

        while (READ_BIT(QSPI->FCR, QSPI_FCR_CMDS_MSK) && (trycounter--));
    }
    while ((QSPI->FCRLR & 0x02) && (timecnt--));

    if (QSPI->FCRLR & 0x02)
        return ERROR;
    else
        return OK;
}

/**
  * @brief  Init for qspi module.
  * @param  None
  * @retval None
  */
void flm_qspi_config(void)
{
    qspi_dac_cfg_t g_dac;
    qspi_handle_t g_qspi = {0};
    qspi_data_capture_cfg_t g_capture_cfg = {0};

    ald_cmu_init();
#if defined TEST_IN_PLL_96
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_96M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 96000000);
#elif defined TEST_IN_PLL_72
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_72M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);
#elif defined TEST_IN_PLL_48
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_48M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
#elif defined TEST_IN_PLL_32
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_32M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 32000000);
#else
#endif
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);
    ald_cmu_qspi_clock_select(CMU_QSPI_CLOCK_SEL_HCLK2);

    /* Init QSPI pins */
    qspi_pin_init();

    g_qspi.perh         = QSPI;
    g_qspi.init.clkdiv  = QSPI_DIV_4;
    g_qspi.init.cpol    = QSPI_CPOL_H;     /*模式3*/
    g_qspi.init.chpa    = QSPI_CPHA_2E;
    g_qspi.init.nssdcode = QSPI_SINGLE_CHIP;
    g_qspi.init.chipsel = QSPI_CS_NSS0;
    g_qspi.init.wrppin  = DISABLE;

    config_flash_param(&g_qspi);

    g_dac.dtrprtcol = DISABLE;
    g_dac.ahbdecoder = DISABLE;
    g_dac.xipimmed = DISABLE;
    g_dac.xipnextrd = DISABLE;
    g_dac.addrremap = DISABLE;
    g_dac.dmaperh = DISABLE;

    g_dac.wrinit.instxfer = QSPI_XFER_SINGLE;   /*Quad写时序，参考Flash的0x32指令*/
    g_dac.wrinit.addxfer = QSPI_XFER_SINGLE;
    g_dac.wrinit.datxfer = QSPI_XFER_QUAD;
    g_dac.wrinit.autowel = ENABLE;     /*写数据时控制器自动发送写使能命令0x06*/
    g_dac.wrinit.wrcde = 0x32;  /*Quad写命令*/
    g_dac.wrinit.dcyles = 0;

    g_dac.rdinit.instxfer = QSPI_XFER_SINGLE;   /*Quad读时序，参考Flash的0xEB指令*/
    g_dac.rdinit.addxfer = QSPI_XFER_QUAD;
    g_dac.rdinit.datxfer = QSPI_XFER_QUAD;
    g_dac.rdinit.ddrbit = DISABLE;
    g_dac.rdinit.modebit = ENABLE;  /*使能模式位*/
    g_dac.rdinit.mbitval = 0xFF;
    g_dac.rdinit.rdcde = 0xEB;  /*Quad读命令*/
    g_dac.rdinit.dcyles = 4;

    qspi_dac_config(&g_qspi, &g_dac);  /*根据以上参数初始化直接访问控制器*/

    /* 如果使用二分频必须按如下配置来配置读数据捕捉寄存器 */
    if (g_qspi.init.clkdiv == QSPI_DIV_2)
    {
        g_capture_cfg.smpledge = QSPI_FALLING_E;
        g_capture_cfg.dlydcl = 3;
        ald_qspi_read_data_capture_config(&g_qspi, &g_capture_cfg);
    }

    if (bsp_w25q128_reset() != OK)
        return;

    if (bsp_w25q128_enable_quad_mode() != OK)
        return;

    return;
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
