/**********************************************************************************
 *
 * @file    ald_qspi.c
 * @brief   QSPI module driver.
 *
 * @date    17 Sep 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          17 Sep 2021     biyq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "flm_qspi.h"


/** @addtogroup ES32FXXX_ALD
  * @{
  */

/** @defgroup QSPI QSPI
  * @brief QSPI module driver
  * @{
  */


/** @defgroup QSPI_Private_Functions   QSPI Private Functions
  * @brief  QSPI Private functions
  * @{
  */

/**
  * @brief  Wait for a flag state until timeout.
  * @param  hperh: QSPI handle
  * @retval Status, see @ref ald_status_t.
  */
static ald_status_t qspi_wait_for_idle(qspi_handle_t *hperh, uint32_t timeout)
{
    /* Wait until flag is in expected state */
    do
    {
        /* Check for the Timeout */
        timeout--;
    }
    while ((READ_BIT(hperh->perh->CR, QSPI_CR_IDLES_MSK) != QSPI_CR_IDLES_MSK) && (0x00 == timeout));

    return OK;
}

/**
  * @}
  */

/** @defgroup QSPI_Public_Functions QSPI Public Functions
  * @brief QSPI Public Functions
  * @{
  */
/** @defgroup QSPI_Public_Functions_Group1 Basic execution functions
  * @brief QSPI basic execution functions
  * @{
  */
/**
  * @brief Initializes the QSPI basic parameters.
  * @param hperh: Pointer to the QSPI qspi_handle_t structure.
  * @retval None.
  */
void ald_qspi_init(qspi_handle_t *hperh)
{
    MODIFY_REG(hperh->perh->CR, QSPI_CR_BAUD_MSK | QSPI_CR_CPOL_MSK | QSPI_CR_CPHA_MSK | \
               QSPI_CR_PSL_MSK  | QSPI_CR_SWPP_MSK, (hperh->init.clkdiv << QSPI_CR_BAUD_POSS) | \
               (hperh->init.cpol << QSPI_CR_CPOL_POS) | \
               (hperh->init.chpa << QSPI_CR_CPHA_POS) | \
               (hperh->init.chipsel << QSPI_CR_PSL_POSS) | \
               (hperh->init.wrppin << QSPI_CR_SWPP_POS));

    return;
}

/**
 * @brief Configure Read Operations.
 * @param hperh: Pointer to the QSPI qspi_handle_t structure.
 * @param rdcfg: Pointer to configuration structure for QSPI read operations.
 * @retval Status, see @ref ald_status_t.
 */
ald_status_t ald_qspi_read_config(qspi_handle_t *hperh, const qspi_read_cfg_t *rdcfg)
{
    ald_status_t status = ERROR;

    status = qspi_wait_for_idle(hperh, 0xFFFF);

    if (status != OK)
        return status;

    MODIFY_REG(hperh->perh->DRIR, QSPI_DRIR_RINST_MSK | QSPI_DRIR_MBEN_MSK | \
               QSPI_DRIR_DCYC_MSK | QSPI_DRIR_ADMODE_MSK | \
               QSPI_DRIR_DMODE_MSK | QSPI_DRIR_IMODE_MSK | \
               QSPI_DRIR_DDRM_MSK, \
               (rdcfg->rdcde | (rdcfg->dcyles << 24) | \
                (rdcfg->addxfer << 12) | (rdcfg->instxfer << 8) | \
                (rdcfg->ddrbit << QSPI_DRIR_DDRM_POS) | (rdcfg->modebit << QSPI_DRIR_MBEN_POS)) | \
               (rdcfg->datxfer << QSPI_DRIR_DMODE_POSS));

    if (rdcfg->modebit)
        MODIFY_REG(hperh->perh->MBR, QSPI_MBR_MODEB_MSK, rdcfg->mbitval);

    return status;
}

/**
 * @brief Configure Write Operations.
 * @param hperh: Pointer to the QSPI qspi_handle_t structure.
 * @param wrcfg: Pointer to configuration structure for QSPI write operations.
 * @retval Status, see @ref ald_status_t.
 */
ald_status_t ald_qspi_write_config(qspi_handle_t *hperh, const qspi_write_cfg_t *wrcfg)
{
    ald_status_t status = ERROR;

    status = qspi_wait_for_idle(hperh, 0xFFFF);

    if (status != OK)
        return status;

    MODIFY_REG(hperh->perh->DWIR, QSPI_DWIR_WINST_MSK | QSPI_DWIR_DCYC_MSK | \
               QSPI_DWIR_ADMODE_MSK | QSPI_DWIR_DMODE_MSK,
               wrcfg->wrcde | \
               (wrcfg->addxfer << 12) | \
               (wrcfg->datxfer << 16) | \
               (wrcfg->dcyles << 24));

    if (wrcfg->autowel)
        CLEAR_BIT(hperh->perh->DWIR, QSPI_DWIR_WELD_MSK);
    else
        SET_BIT(hperh->perh->DWIR, QSPI_DWIR_WELD_MSK);

    return status;
}

/**
 * @brief QSPI Device Delay Configuration.
 * @param hperh: Pointer to the QSPI qspi_handle_t structure.
 * @param dlycfg: Device delay configuration structure.
 * @retval Status, see @ref ald_status_t.
 */
ald_status_t ald_qspi_device_delay_config(qspi_handle_t *hperh, qspi_dly_cfg_t *dlycfg)
{
    ald_status_t status = ERROR;

    status = qspi_wait_for_idle(hperh, 0xFFFF);

    if (status != OK)
        return status;

    MODIFY_REG(hperh->perh->DDLR, QSPI_DDLR_CSSOT_MSK | QSPI_DDLR_CSEOT_MSK | QSPI_DDLR_CSDADS_MSK | QSPI_DDLR_CSDA_MSK, \
               dlycfg->ccsot | (dlycfg->cseot << 8) | (dlycfg->csdads << 16) | (dlycfg->csda << 24));
    return status;
}

/**
 * @brief QSPI Read Data Capture Configuration.
 * @param hperh: Pointer to the QSPI qspi_handle_t structure.
 * @param dtcptcfg: Data capture configuration structure.
 * @retval Status, see @ref ald_status_t.
 */
ald_status_t ald_qspi_read_data_capture_config(qspi_handle_t *hperh, qspi_data_capture_cfg_t *dtcptcfg)
{
    ald_status_t status = ERROR;

    status = qspi_wait_for_idle(hperh, 0xFFFF);

    if (status != OK)
        return status;

    MODIFY_REG(hperh->perh->RDCR, QSPI_RDCR_BYLPC_MSK | QSPI_RDCR_DLYR_MSK | QSPI_RDCR_SMES_MSK | QSPI_RDCR_DLYT_MSK, \
               dtcptcfg->bypsalcc | (dtcptcfg->dlydcl << 1) | (dtcptcfg->smpledge << 5) | (dtcptcfg->dlytd << 16));
    return status;
}

/**
 * @brief QSPI Flash memory Configuration.
 * @param hperh: Pointer to the QSPI qspi_handle_t structure.
 * @param devcfg: flash parameter configuration structure.
 * @retval Status, see @ref ald_status_t.
 */
ald_status_t ald_qspi_device_size_config(qspi_handle_t *hperh, qspi_device_size_t *devcfg)
{
    ald_status_t status = ERROR;

    status = qspi_wait_for_idle(hperh, 0xFFFF);

    if (status != OK)
        return status;

    QSPI_DISABLE(hperh);

    MODIFY_REG(hperh->perh->DSCR, QSPI_DSCR_ADSIZE_MSK | QSPI_DSCR_PASIZE_MSK | QSPI_DSCR_BKSIZE_MSK | QSPI_DSCR_CS0SIZE_MSK |  \
               QSPI_DSCR_CS1SIZE_MSK | QSPI_DSCR_CS2SIZE_MSK | QSPI_DSCR_CS3SIZE_MSK, \
               (devcfg->addr << QSPI_DSCR_ADSIZE_POSS) | \
               (devcfg->page << QSPI_DSCR_PASIZE_POSS) | \
               (devcfg->blk  << QSPI_DSCR_BKSIZE_POSS) | \
               (devcfg->cs0  << QSPI_DSCR_CS0SIZE_POSS) | \
               (devcfg->cs1  << QSPI_DSCR_CS1SIZE_POSS) | \
               (devcfg->cs2  << QSPI_DSCR_CS2SIZE_POSS) | \
               (devcfg->cs3  << QSPI_DSCR_CS3SIZE_POSS));
    QSPI_ENABLE(hperh);

    return OK;
}

/**
  * @brief Initializes the QSPI direct access according to the specified parameters
  *        in the associated handle.
  * @param hperh: Pointer to the QSPI qspi_handle_t structure.
  * @param dcfg : structure that contains the indirect read command configuration information.
  * @retval Status, see @ref ald_status_t.
  */
ald_status_t qspi_dac_config(qspi_handle_t *hperh, qspi_dac_cfg_t *dcfg)
{
    ald_status_t status = ERROR;

    status = qspi_wait_for_idle(hperh, 0xFFFF);

    if (status != OK)
        return status;

    QSPI_DISABLE(hperh);
    ald_qspi_init(hperh);

    if (ald_qspi_write_config(hperh, &dcfg->wrinit) != OK)
        return ERROR;

    if (ald_qspi_read_config(hperh, &dcfg->rdinit) != OK)
        return ERROR;

    if (dcfg->addrremap)
        MODIFY_REG(hperh->perh->RAR, QSPI_RAR_READDR_MSK, dcfg->remapaddr);

    MODIFY_REG(hperh->perh->CR, QSPI_CR_DTRM_MSK | QSPI_CR_ADEN_MSK | QSPI_CR_XIPIM_MSK | \
               QSPI_CR_XIPNX_MSK | QSPI_CR_AREN_MSK | QSPI_CR_DMAEN_MSK, \
               dcfg->dtrprtcol << 24 | dcfg->ahbdecoder << 23);
    QSPI_ENABLE(hperh);

    return OK;
}

/**
  * @}
  */

/**
 * @}
 */
/**
 * @}
 */
/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
