/**********************************************************************************
 *
 * @file    qspi_flash.c
 * @brief   SPI Flash file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    zhuxf          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ----------------------------------------------------------------- */

#include "qspi_flash.h"
#include "w25q128fv.h"
#include "main.h"

/* Private Macros ----------------------------------------------------------- */

/* Private Variables--------------------------------------------------------- */

/* Public Variables --------------------------------------------------------- */

/* Private Constants -------------------------------------------------------- */

/* Private function prototypes ---------------------------------------------- */

/* Private Function---------------------------------------------------------- */

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */


/**
  * @brief  Reset the QSPI memory.
  * @param  hqspi: QSPI handle
  * @retval Status
  */
ald_status_t bsp_w25q128_reset(qspi_handle_t *hqspi)
{
    qspi_stig_cmd_t qspi_stig_cmd = {0};
    uint8_t status;

    /* Initialize the reset enable command */
    qspi_stig_cmd.code     = RESET_ENABLE_CMD;
    qspi_stig_cmd.addr_len = 0;
    qspi_stig_cmd.wr_len   = 0;
    qspi_stig_cmd.mode_bit = DISABLE;
    qspi_stig_cmd.d_sycle  = 0;

    /* Send the command */
    if (ald_qspi_execute_stig_cmd(hqspi, &qspi_stig_cmd) != OK)
        return ERROR;

    /* Send the reset memory command */
    qspi_stig_cmd.code = RESET_MEMORY_CMD;

    if (ald_qspi_execute_stig_cmd(hqspi, &qspi_stig_cmd) != OK)
        return ERROR;

    qspi_stig_cmd.code     = READ_STATUS_REG1_CMD;
    qspi_stig_cmd.addr_len = 0x0;
    qspi_stig_cmd.rd_len   = 0x01;
    qspi_stig_cmd.rd_buf   = &status;

    if (ald_qspi_execute_stig_cmd(hqspi, &qspi_stig_cmd) != OK)
        return ERROR;

    /* Configure automatic polling mode to wait the memory is ready */
    if (!(status & 0x01))
        return OK;

    return ERROR;
}

/**
  * @brief  Enable written.
  * @param  hqspi: QSPI handle
  * @retval Status
  */
ald_status_t bsp_w25q128_wr_enable(qspi_handle_t *hqspi)
{
    qspi_stig_cmd_t qspi_stig_cmd = {0};
    uint8_t status;

    /* Enable write operations */
    qspi_stig_cmd.code     = WRITE_ENABLE_CMD;
    qspi_stig_cmd.addr_len = 0;
    qspi_stig_cmd.wr_len   = 0;
    qspi_stig_cmd.mode_bit = DISABLE;/* QSPI_MODEBIT_DISABLE; */
    qspi_stig_cmd.d_sycle  = 0;
    qspi_stig_cmd.rd_len   = 0;

    /* Send the command */
    if (ald_qspi_execute_stig_cmd(hqspi, &qspi_stig_cmd) != OK)
        while (1);

    qspi_stig_cmd.code     = READ_STATUS_REG1_CMD;
    qspi_stig_cmd.addr_len = 0x0;
    qspi_stig_cmd.rd_len   = 0x01;
    qspi_stig_cmd.rd_buf   = &status;

    do
    {
        if (ald_qspi_execute_stig_cmd(hqspi, &qspi_stig_cmd) != OK)
            return ERROR;
    }
    while (status & 0x01);   /*状态寄存器的bit0为1表示正忙*/

    return OK;
}

/**
  * @brief  Erase block
  * @param  hqspi: QSPI handle
  * @param  blkaddr: Address of block
  * @retval Status
  */
ald_status_t bsp_w25q128_erase_block(qspi_handle_t *hqspi, uint32_t blkaddr)
{
    qspi_stig_cmd_t qspi_stig_cmd = {0};
    uint8_t status;
    uint32_t tickstart = 0U;
    uint32_t timeout   = 1000U;

    /* Before Erase Sector,must send write enable */
    if (bsp_w25q128_wr_enable(hqspi) != OK)
        while (1);

    /* Enable write operations */
    qspi_stig_cmd.code     = SECTOR_ERASE_CMD;
    qspi_stig_cmd.addr_len = 0x3;
    qspi_stig_cmd.wr_len   = 0;
    qspi_stig_cmd.mode_bit = DISABLE;/* QSPI_MODEBIT_DISABLE; */
    qspi_stig_cmd.d_sycle  = 0;
    qspi_stig_cmd.rd_len   = 0;
    qspi_stig_cmd.addr     = blkaddr;
    qspi_stig_cmd.op_addr  = ENABLE;

    /* Send the command */
    while (ald_qspi_execute_stig_cmd(hqspi, &qspi_stig_cmd) != OK)
    {
        if (timeout != ALD_MAX_DELAY)
        {
            if ((timeout == 0) || ((ald_get_tick() - tickstart) > timeout))
                return ERROR;
        }
    }

    qspi_stig_cmd.code     = READ_STATUS_REG1_CMD;
    qspi_stig_cmd.addr_len = 0x0;
    qspi_stig_cmd.rd_len   = 0x01;
    qspi_stig_cmd.rd_buf   = &status;
    qspi_stig_cmd.wr_len   = 0;

    do
    {
        if (ald_qspi_execute_stig_cmd(hqspi, &qspi_stig_cmd) != OK)
            return ERROR;
    }
    while (status & 0x01);   /*状态寄存器的bit0为1表示正忙*/

    return OK;
}

ald_status_t bsp_w25q128_enable_quad_mode(qspi_handle_t *hqspi)
{
    qspi_stig_cmd_t qspi_stig_cmd = {0};
    uint8_t status;

    status = 0x02;

    qspi_stig_cmd.code   = WRITE_STATUS_REG2_CMD;
    qspi_stig_cmd.wr_buf = &status;
    qspi_stig_cmd.wr_len = 1;
    qspi_stig_cmd.rd_len = 0;

    if (ald_qspi_execute_stig_cmd(hqspi, &qspi_stig_cmd) != OK)
        return ERROR;

    qspi_stig_cmd.code = READ_STATUS_REG1_CMD;
    qspi_stig_cmd.rd_buf = &status;
    qspi_stig_cmd.rd_len = 1;
    qspi_stig_cmd.wr_len = 0;

    do
    {
        if (ald_qspi_execute_stig_cmd(hqspi, &qspi_stig_cmd) != OK)
            return ERROR;
    }
    while (status & 0x01);   /*状态寄存器的bit0为1表示正忙*/

    qspi_stig_cmd.code = READ_STATUS_REG2_CMD;
    qspi_stig_cmd.rd_buf = &status;
    qspi_stig_cmd.rd_len = 1;
    qspi_stig_cmd.wr_len = 0;

    if (ald_qspi_execute_stig_cmd(hqspi, &qspi_stig_cmd) != OK)
        return ERROR;

    if ((status & 0x02) == 0)
        return ERROR;

    return OK;
}

ald_status_t bsp_w25q128_disable_protect(qspi_handle_t *hqspi)
{
    qspi_stig_cmd_t qspi_stig_cmd = {0};
    uint8_t status;

    if (bsp_w25q128_wr_enable(hqspi) != OK)
        return ERROR;

    status = 0;
    qspi_stig_cmd.code = WRITE_STATUS_REG1_CMD;
    qspi_stig_cmd.wr_buf = &status;
    qspi_stig_cmd.wr_len = 1;
    qspi_stig_cmd.rd_len = 0;

    if (ald_qspi_execute_stig_cmd(hqspi, &qspi_stig_cmd) != OK)
        return ERROR;

    qspi_stig_cmd.code     = READ_STATUS_REG1_CMD;
    qspi_stig_cmd.addr_len = 0x0;
    qspi_stig_cmd.rd_len   = 0x01;
    qspi_stig_cmd.rd_buf   = &status;
    qspi_stig_cmd.wr_len   = 0;

    do
    {
        if (ald_qspi_execute_stig_cmd(hqspi, &qspi_stig_cmd) != OK)
            return ERROR;
    }
    while (status & 0x01);   /*状态寄存器的bit0为1表示正忙*/

    return OK;
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
