/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    biyq          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ----------------------------------------------------------------- */

#include "main.h"


/* Private Macros ----------------------------------------------------------- */

/* Private Variables--------------------------------------------------------- */

uint32_t g_wdata[64] = {0};
uint32_t g_rdata[64] = {0};

/* Public Variables --------------------------------------------------------- */ 
qspi_dac_cfg_t  g_dac;
qspi_handle_t   g_qspi;
qspi_data_capture_cfg_t g_capture_cfg = {0};

/* Private Constants -------------------------------------------------------- */

/* Private function prototypes ---------------------------------------------- */
void qspi_pin_init(void);
void config_flash_param(void);

/* Private Function---------------------------------------------------------- */

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/**
  * @brief Test main function
  * @param None
  * @retval None
  */
int main(void)
{
    int i;

    ald_cmu_init(); /*初始化滴答时钟*/
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_48M);    /*使能倍频，由晶振三分频倍频至48MHz*/
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);    /*选择倍频时钟为系统时钟*/
    ald_cmu_qspi_clock_select(CMU_QSPI_CLOCK_SEL_HCLK2);    /*选择HCLK2为QSPI时钟*/
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);    /*使能所有外设时钟*/

    led_init(); /*初始化LED管脚为输出*/

    for (i = 0; i < 64; ++i)
    {
        g_wdata[i] = i;
        g_rdata[i] = 0x55;
    }

    qspi_pin_init();

    g_qspi.perh         = QSPI;
    g_qspi.init.clkdiv  = QSPI_DIV_2;
    g_qspi.init.cpol    = QSPI_CPOL_H;     /*模式3*/
    g_qspi.init.chpa    = QSPI_CPHA_2E;
    g_qspi.init.nssdcode = QSPI_SINGLE_CHIP;
    g_qspi.init.chipsel = QSPI_CS_NSS0;
    g_qspi.init.wrppin  = DISABLE;

    config_flash_param();

    g_dac.dtrprtcol = DISABLE;
    g_dac.ahbdecoder = DISABLE;
    g_dac.xipimmed = DISABLE;
    g_dac.xipnextrd = DISABLE;
    g_dac.addrremap = DISABLE;
    g_dac.dmaperh = DISABLE;

    g_dac.wrinit.instxfer = QSPI_XFER_SINGLE;   /*Quad写时序，参考Flash的0x32指令*/
    g_dac.wrinit.addxfer = QSPI_XFER_SINGLE;
    g_dac.wrinit.datxfer = QSPI_XFER_QUAD;
    g_dac.wrinit.autowel = ENABLE;     /*写数据时控制器自动发送写使能命令0x06*/
    g_dac.wrinit.wrcde = 0x32;  /*Quad写命令*/
    g_dac.wrinit.dcyles = 0;

    g_dac.rdinit.instxfer = QSPI_XFER_SINGLE;   /*Quad读时序，参考Flash的0xEB指令*/
    g_dac.rdinit.addxfer = QSPI_XFER_QUAD;
    g_dac.rdinit.datxfer = QSPI_XFER_QUAD;
    g_dac.rdinit.ddrbit = DISABLE;
    g_dac.rdinit.modebit = ENABLE;  /*使能模式位*/
    g_dac.rdinit.mbitval = 0xFF;
    g_dac.rdinit.rdcde = 0xEB;  /*Quad读命令*/
    g_dac.rdinit.dcyles = 4;

    qspi_dac_config(&g_qspi, &g_dac);  /*根据以上参数初始化直接访问控制器*/

    /* 如果使用二分频必须按如下配置来配置读数据捕捉寄存器 */
    if (g_qspi.init.clkdiv == QSPI_DIV_2)
    {
        g_capture_cfg.smpledge = QSPI_FALLING_E;
        g_capture_cfg.dlydcl = 3;
        ald_qspi_read_data_capture_config(&g_qspi, &g_capture_cfg);
    }

    if (bsp_w25q128_reset(&g_qspi) != OK)
        while (1);

    if (bsp_w25q128_disable_protect(&g_qspi) != OK)
        while (1);

    if (bsp_w25q128_enable_quad_mode(&g_qspi) != OK)
        while (1);

    if (bsp_w25q128_erase_block(&g_qspi, 0) != OK)
        while (1);

    for (i = 0; i < 64; ++i)
        ald_qspi_dac_wr(QSPI_MEMORY_ADDRESS + 4 * i, g_wdata[i]);

    for (i = 0; i < 64; ++i)
        g_rdata[i] = ald_qspi_dac_rd(QSPI_MEMORY_ADDRESS + 4 * i);

    while (1)
    {
        led_toggle();
        ald_delay_ms(500);
    }
}

/**
  * @brief  Initializate spi flash pin
  * @retval None.
  */ 
static void config_flash_param(void)
{
    qspi_device_size_t dsize_config;

    dsize_config.cs0 = QSPI_NSS_512M;
    dsize_config.cs1 = QSPI_NSS_512M;
    dsize_config.cs2 = QSPI_NSS_512M;
    dsize_config.cs3 = QSPI_NSS_512M;
    dsize_config.addr = 2;   /*有2+1个地址字节*/
    dsize_config.page = 256;    /*页大小*/
    dsize_config.blk = 16;   /*Block大小64KB*/

    ald_qspi_device_size_config(&g_qspi, &dsize_config);    /*配置Flash参数*/
}

/**
  * @brief  Initializate spi flash pin
  * @retval None.
  */
static void qspi_pin_init(void)
{
    gpio_init_t gpio_config;

    gpio_config.odos  = GPIO_PUSH_PULL;
    gpio_config.pupd  = GPIO_PUSH_UP;
    gpio_config.nodrv = GPIO_OUT_DRIVE_6;
    gpio_config.podrv = GPIO_OUT_DRIVE_6;
    gpio_config.flt   = GPIO_FILTER_DISABLE;
    gpio_config.type  = GPIO_TYPE_TTL;

    gpio_config.func  = GPIO_FUNC_6;
    gpio_config.mode  = GPIO_MODE_OUTPUT;
    ald_gpio_init(GPIOB, GPIO_PIN_12, &gpio_config); /*初始化PB12为片选管脚*/
    ald_gpio_init(GPIOB, GPIO_PIN_13, &gpio_config); /*初始化PB13为时钟输出管脚*/

    gpio_config.func  = GPIO_FUNC_6;
    gpio_config.mode = GPIO_MODE_INPUT;
    ald_gpio_init(GPIOB, GPIO_PIN_14, &gpio_config); /*初始化PB14为IO0管脚*/
    ald_gpio_init(GPIOB, GPIO_PIN_15, &gpio_config); /*初始化PB15为IO1管脚*/
    ald_gpio_init(GPIOD, GPIO_PIN_8, &gpio_config); /*初始化PD8为IO2管脚*/
    ald_gpio_init(GPIOD, GPIO_PIN_9, &gpio_config); /*初始化PD9为IO3管脚*/
    
    return;
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
