/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Only one macro can be opened at a time */
/* #define LP_MODE_STOP1 */
#define LP_MODE_STOP2
/* #define LP_MODE_STANDBY */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initializate pin of exti module.
  * @retval None
  */
void init_wakeup_pin(void)
{
    md_gpio_init_t gpio_init;

    /* Initialize GPIO as input */
    md_gpio_init_struct(&gpio_init);      /* initialize the gpio_init */
    gpio_init.mode  = MD_GPIO_MODE_INPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_FLOATING;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_1;
    md_gpio_init(PMU_EXTI_PORT, PMU_EXTI_PIN, &gpio_init);

    gpio_init.pupd  = MD_GPIO_PUSH_DOWN;
    md_gpio_init(PMU_STANDBY_PORT, PMU_STANDBY_PIN, &gpio_init);

    /* Initialize external interrupt */
    md_giop_interrupt_filter_clock_select(0U);     /* set exit filter clock  0: ULRC(10KHz) */
    md_giop_interrupt_filter_time_set(10U);        /* set filter times: ((10 + 1) * 2)CLOCK */
    md_gpio_interrupt_filter_enable(PMU_EXTI_PIN); /* enable exit filter */

    md_gpio_set_interrupt_port(PMU_EXTI_PORT, PMU_EXTI_PIN);      /* set exit interrupt port */
    md_gpio_enable_riging_edge_trigger(PMU_EXTI_PIN); /* enable riging edge trigger */
    md_gpio_enable_external_interrupt(PMU_EXTI_PIN);  /* enable exti7 */
    md_mcu_irq_config(EXTI7_IRQn, 0, 0, ENABLE);      /* enable NVIC EXTI7_IRQn */
}

/**
  * @brief  Initializate pin of led module.
  * @retval None
  */
void init_led_pin(void)
{
    md_gpio_init_t gpio_init;

    md_gpio_init_struct(&gpio_init);      /* initialize the gpio_init */
    gpio_init.mode = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos = MD_GPIO_PUSH_PULL;
    gpio_init.pupd = MD_GPIO_FLOATING;
    gpio_init.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_init.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_init.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_init.type = MD_GPIO_TYPE_CMOS;
    gpio_init.func = MD_GPIO_FUNC_1;

    md_gpio_init(LED_PORT, LED_PIN, &gpio_init);
    md_gpio_write_pin(LED_PORT, LED_PIN, 1);

    return;
}

/**
  * @brief  Initializate pin of swd module.
  * @retval None
  */
void init_swd_pin(void)
{
    md_gpio_init_t gpio_init;

    md_gpio_init_struct(&gpio_init);      /* initialize the gpio_init */
    gpio_init.mode = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos = MD_GPIO_PUSH_PULL;
    gpio_init.pupd = MD_GPIO_FLOATING;
    gpio_init.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_init.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_init.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_init.type = MD_GPIO_TYPE_CMOS;
    gpio_init.func = MD_GPIO_FUNC_1;

    md_gpio_init(GPIOA, MD_GPIO_PIN_13, &gpio_init);
    md_gpio_write_pin(GPIOA, MD_GPIO_PIN_13, 0);

    md_gpio_init(GPIOA, MD_GPIO_PIN_14, &gpio_init);
    md_gpio_write_pin(GPIOA, MD_GPIO_PIN_14, 0);

    return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Configure system clock */
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HRC_6, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();
    /* 2bits preempt-priority(0-3)
       2bits sub-priority(0-3). */
    NVIC_SetPriorityGrouping(5);

    init_wakeup_pin();
    init_led_pin();

    while (1)
    {
        md_gpio_write_pin(LED_PORT, LED_PIN, 0);
        md_delay_1ms(400);
        md_gpio_write_pin(LED_PORT, LED_PIN, 1);
        md_delay_1ms(400);
#ifdef LP_MODE_STOP1
        md_pmu_stop1_enter();
#endif /* LP_MODE_STOP1*/
#ifdef LP_MODE_STOP2
        md_pmu_stop2_enter();
#endif /* LP_MODE_STOP2*/
#ifdef LP_MODE_STANDBY
        md_pmu_standby_enter(MD_PMU_STANDBY_PORT_SEL_PA6, MD_PMU_STANDBY_LEVEL_HIGH);
#endif /* LP_MODE_STANDBY*/
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
