/**********************************************************************************
 *
 * @file    bat.c
 * @brief   bat file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include "bat.h"

/* Private Variables --------------------------------------------------------- */
/* Public Variables ---------------------------------------------------------- */
/* Private Constants --------------------------------------------------------- */

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples_PMU
  * @{
  */

/* Private Macros ------------------------------------------------------------ */

/* Only one macro can be opened at a time */
/* #define LP_MODE_STOP1 */
#define LP_MODE_STOP2
/* #define LP_MODE_STANDBY */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initializate pin of exti module.
  * @retval None
  */
void init_wakeup_pin(void)
{
    gpio_init_t gpio_init;
    exti_init_t exti_init;

    /* Initialize GPIO as input */
    memset(&gpio_init, 0x0, sizeof(gpio_init)); /* initialize the gpio_init */
    gpio_init.mode  = GPIO_MODE_INPUT;
    gpio_init.pupd  = GPIO_FLOATING;
    gpio_init.flt   = GPIO_FILTER_DISABLE;
    gpio_init.type  = GPIO_TYPE_CMOS;
    gpio_init.func  = GPIO_FUNC_1;
    ald_gpio_init(PMU_EXTI_PORT, PMU_EXTI_PIN, &gpio_init);

    gpio_init.pupd = GPIO_PUSH_DOWN;
    ald_gpio_init(PMU_STANDBY_PORT, PMU_STANDBY_PIN, &gpio_init);

    /* Initialize external interrupt */
    memset(&exti_init, 0x0, sizeof(exti_init)); /* initialize the exti_init */
    exti_init.filter      = ENABLE;
    exti_init.cks         = EXTI_FILTER_CLOCK_10K;
    exti_init.filter_time = 10;
    ald_gpio_exti_init(PMU_EXTI_PORT, PMU_EXTI_PIN, &exti_init);

    /* Clear interrupt flag */
    ald_gpio_exti_clear_flag_status(PMU_EXTI_PIN);
    /* Configure interrupt */
    ald_gpio_exti_interrupt_config(PMU_EXTI_PIN, EXTI_TRIGGER_RISING_EDGE, ENABLE);
    /* Enable external interrupt */
    ald_mcu_irq_config(EXTI7_IRQn, 0, 0, ENABLE);
}

/**
  * @brief  Initializate pin of led module.
  * @retval None
  */
void init_led_pin_bat(void)
{
    gpio_init_t gpio_init;

    memset(&gpio_init, 0x0, sizeof(gpio_init)); /* initialize the gpio_init */
    gpio_init.mode = GPIO_MODE_OUTPUT;
    gpio_init.odos = GPIO_PUSH_PULL;
    gpio_init.pupd = GPIO_PUSH_UP;
    gpio_init.podrv = GPIO_OUT_DRIVE_6;
    gpio_init.nodrv = GPIO_OUT_DRIVE_6;
    gpio_init.flt  = GPIO_FILTER_DISABLE;
    gpio_init.type = GPIO_TYPE_CMOS;
    gpio_init.func = GPIO_FUNC_1;

    ald_gpio_init(LED_PORT, LED_PIN, &gpio_init);
    ald_gpio_write_pin(LED_PORT, LED_PIN, 1);

    return;
}

/**
  * @brief  Initializate pin of swd module.
  * @retval None
  */
void init_swd_pin_bat(void)
{
    gpio_init_t gpio_init;

    memset(&gpio_init, 0x0, sizeof(gpio_init)); /* initialize the gpio_init */
    gpio_init.mode = GPIO_MODE_OUTPUT;
    gpio_init.odos = GPIO_PUSH_PULL;
    gpio_init.pupd = GPIO_PUSH_UP;
    gpio_init.podrv = GPIO_OUT_DRIVE_6;
    gpio_init.nodrv = GPIO_OUT_DRIVE_6;
    gpio_init.flt  = GPIO_FILTER_DISABLE;
    gpio_init.type = GPIO_TYPE_CMOS;
    gpio_init.func = GPIO_FUNC_1;

    ald_gpio_init(GPIOA, GPIO_PIN_13, &gpio_init);
    ald_gpio_write_pin(GPIOA, GPIO_PIN_13, 1);

    ald_gpio_init(GPIOA, GPIO_PIN_14, &gpio_init);
    ald_gpio_write_pin(GPIOA, GPIO_PIN_14, 1);

    return;
}

/**
  * @brief  Initializate lvd.
  * @retval None
  */
void init_lvd_powerup(void)
{
    SYSCFG_UNLOCK();
    ald_pmu_lvd_config(PMU_LVD_VOL_SEL_3_4, PMU_LVD_TRIGGER_LOW_LEVEL, ENABLE);
    SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDCIF_MSK);
    SYSCFG_LOCK();
    NVIC_ClearPendingIRQ(LVD_IRQn);

    ald_mcu_irq_config(LVD_IRQn, 0, 0, ENABLE);
    return;
}

/**
  * @brief  Test battery function
  * @retval Status.
  */
void bat_proc(void)
{
    __disable_irq();

    init_wakeup_pin();
    init_led_pin_bat();
    init_lvd_powerup();

    NVIC_ClearPendingIRQ(EXTI7_IRQn);
    NVIC_ClearPendingIRQ(LVD_IRQn);

    while (1)
    {

        if (ald_gpio_exti_get_flag_status(PMU_EXTI_PIN) == SET)
        {
            ald_gpio_exti_clear_flag_status(PMU_EXTI_PIN);
            ald_gpio_write_pin(LED_PORT, LED_PIN, 0);
            ald_gpio_write_pin(LED_PORT, LED_PIN, 1);
            NVIC_ClearPendingIRQ(EXTI7_IRQn);
        }

        if (READ_BIT(PMU->LVDCR, PMU_LVDCR_LVDIF_MSK))
        {
            SYSCFG_UNLOCK();
            SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDCIF_MSK);
            SYSCFG_LOCK();
            ald_gpio_toggle_pin(LED_PORT, LED_PIN);
            NVIC_SystemReset();
        }

#ifdef LP_MODE_STOP1
        ald_pmu_stop1_enter();
#endif /* LP_MODE_STOP1*/
#ifdef LP_MODE_STOP2
        ald_pmu_stop2_enter();
#endif /* LP_MODE_STOP2*/
#ifdef LP_MODE_STANDBY
        ald_pmu_standby_enter(PMU_STANDBY_PORT_SEL_PA6, PMU_STANDBY_LEVEL_HIGH);
#endif /* LP_MODE_STANDBY*/
    }
}

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
