/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    17 Jan. 2024
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          17 Jan. 2024    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include "main.h"

/* Private Constants --------------------------------------------------------- */

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Public_Function
  * @{
  */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initialize TOGGLE IO.
  * @param  None
  * @retval None
  */
static void init_gpio(void)
{
    md_gpio_init_t gpio_init;

    md_gpio_init_struct(&gpio_init);
    gpio_init.mode = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos = MD_GPIO_PUSH_PULL;
    gpio_init.pupd = MD_GPIO_FLOATING;
    gpio_init.func = MD_GPIO_FUNC_2;
    md_gpio_init(CH4_SRC_PORT, CH4_SRC_PIN, &gpio_init);

    gpio_init.func = MD_GPIO_FUNC_6;
    md_gpio_init(CH4_PIS_PORT, CH4_PIS_PIN, &gpio_init);
}


/**
  * @brief:  Initialize GP16C4T2 to timely overflow funtion
  * @param:  None
  * @retval: None
  */
static void init_timer(void)
{
    md_timer_base_init_t gp16c4t_init;
    md_timer_oc_init_t oc_init;

    md_timer_base_struct_init(&gp16c4t_init);        /* initialize the gp16c4t_init  */
    gp16c4t_init.prescaler = 72 - 1;                 /* clk_count: 72 */
    gp16c4t_init.clk_div = MD_TIMER_CLOCK_DIV1;      /* working clock of dead time and filter */
    gp16c4t_init.mode = MD_TIMER_CNT_MODE_CENTER1;   /* count up */
    gp16c4t_init.period = 5000 - 1;                  /* period is 5000 count clock */
    gp16c4t_init.re_cnt = 1 - 1;                     /* 0 repeat count */
    md_timer_base_set_config(GP16C4T0, &gp16c4t_init);

    md_timer_oc_struct_init(&oc_init);
    oc_init.ocstate = MD_TIMER_OCSTATE_ENABLE;    /* enable CHxO compare output */
    oc_init.oc_mode = MD_TIMER_OC_MODE_PWM2;      /* compare output mode is PWM2 */
    oc_init.oc_polarity = MD_TIMER_OC_POLARITY_HIGH;    /* CHxO compare output polarity is high */
    oc_init.oc_idle = MD_TIMER_OC_IDLE_RESET;           /* CHxO idle is low */
    oc_init.pulse = 2500 - 1;    /* CH4 compare value is 2500 - 1 */
    md_timer_oc_init(GP16C4T0, MD_TIMER_CHANNEL_4, &oc_init);

    md_timer_enable_cc4it_interrupt(GP16C4T0);   /* enable CC4 interrupt request */
    md_timer_enable_uit_interrupt(GP16C4T0);     /* update interrupt request */

    md_timer_set_main_output_enable(GP16C4T0, ENABLE);       /* enable output */
    md_timer_enable_counter_cnten(GP16C4T0);                 /* enable GP16C4T2 */
}

/**
  * @brief  Initialize PIS.
  * @param  None
  * @retval None
  */
void pis_init(void)
{
    md_pis_init_t pis_init;
    memset(&pis_init, 0, sizeof(md_pis_init_t));

    /* Enable PIS CH0 output */
    md_pis_enable_ch0_output(PIS);

    pis_init.p_src     = MD_PIS_GPIO_PIN9;     /* PIS SRC: PB9(GP16C4T0 CH4) */
    pis_init.p_clk     = MD_PIS_CLK_SYS;
    pis_init.p_edge    = MD_PIS_EDGE_NONE;
    pis_init.p_output  = MD_PIS_OUT_LEVEL;
    pis_init.c_trig    = (md_pis_trig_t)0x0;                  /* PIS CH0 output PB9 signal */
    pis_init.c_clk     = MD_PIS_CLK_SYS;
    md_pis_init(&pis_init);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Configure system clock */
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HOSC_3, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();
    /* 2bits preempt-priority(0-3)
       2bits sub-priority(0-3). */
    NVIC_SetPriorityGrouping(5);

    pis_init();

    /* Initialize the gpio */
    init_gpio();

    md_mcu_irq_config(GP16C4T0_IRQn, 0, 0, ENABLE);

    /* Initialize TIMER */
    init_timer();

    while (1);
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
