/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include "main.h"

/* Private Constants --------------------------------------------------------- */

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/** @addtogroup Projects_Examples_MD
  * @{
  */
uint32_t g_adc_result;

/** @addtogroup Public_Function
  * @{
  */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Configure the ADC Pins.
  * @param  None
  * @retval None
  */
void config_adc_pin(void)
{
    md_gpio_init_t gpio_init;
    md_gpio_init_struct(&gpio_init);

    /* Initialize adc gpio pin: PC0 */
    gpio_init.mode  = MD_GPIO_MODE_CLOSE;
    gpio_init.pupd  = MD_GPIO_PUSH_UP;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.podrv = MD_GPIO_OUT_DRIVE_1;
    gpio_init.nodrv = MD_GPIO_OUT_DRIVE_1;
    gpio_init.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_init.type  = MD_GPIO_TYPE_CMOS;
    gpio_init.func  = MD_GPIO_FUNC_0;

    md_gpio_init(ADC_CHANNEL0_GPIO_PORT, ADC_CHANNEL0_PIN, &gpio_init);

    return;
}

/**
  * @brief  Configure the TIM Pins.
  * @param  None
  * @retval None
  */
void config_adc_base(void)
{
    md_adc_init_t adc_init;

    md_adc_struct_init(&adc_init);
    /* Set ADC Base Configuration:  */
    adc_init.align    = MD_ADC_DATAALIGN_RIGHT;     /* Specifies ADC data alignment */
    adc_init.data_bit = MD_ADC_CONV_BIT_12;
    adc_init.div      = MD_ADC_CKDIV_16;            /* ADCCLK divider */
    adc_init.n_ref    = MD_ADC_NEG_REF_VSS;
    adc_init.p_ref    = MD_ADC_POS_REF_VDD;         /* The positive reference voltage */
    md_adc_init(ADC0, &adc_init);

    /* Disable ADC sequencers scan mode*/
    md_adc_disable_scanen(ADC0);
    /* Set ADC continuous conversion mode on ADC group REG */
    md_adc_continuous_conversion_disable_cm(ADC0);

    /* Set ADC group insert sequencer length and scan direction,ADC conversion 1 channel */
    md_adc_set_insert_channel_length_isl(ADC0, MD_ADC_ICH_NR_1);
    /* Set ADC group insert sequence: channel on the selected scan sequence rank */
    md_adc_set_insert_1st_conv_is1(ADC0, MD_ADC_CHANNEL_0);
    md_adc_set_smpt1_cht(ADC0, MD_ADC_CHANNEL_0, MD_ADC_SAMPLETIME_1);

    /* ADC insert trig polarity */
    md_adc_set_insert_conv_extern_polarity(ADC0, MD_ADC_ETS_RISE);

    /* Set priority for ADC_IRQn */
    NVIC_SetPriority(ADC0_IRQn, 0);
    /* Enable ADC_IRQn */
    NVIC_EnableIRQ(ADC0_IRQn);
    /* Enable interruption ADC group insert end of sequence conversions */
    md_adc_inserted_channel_interrupt_enable_icheie(ADC0);
}

/**
  * @brief  Initialize TIMER.
  * @param  None
  * @retval None
  */
void init_timer(void)
{
    md_timer_base_init_t init;
    md_timer_base_struct_init(&init);
    init.prescaler = (uint16_t)0x0003;
    init.period = 0xFFFF;
    md_timer_base_set_config(AD16C4T0, &init);
}

/**
  * @brief  Initialize PIS.
  * @param  None
  * @retval None
  */
void init_pis(void)
{
    md_pis_init_t pis_init;

    /* Initialize PIS */
    memset(&pis_init, 0, sizeof(md_pis_init_t));

    pis_init.p_src     = MD_PIS_TIMER0_UPDATA;
    pis_init.p_clk     = MD_PIS_CLK_PCLK1;
    pis_init.p_edge    = MD_PIS_EDGE_NONE;
    pis_init.p_output  = MD_PIS_OUT_PULSE;
    pis_init.c_trig    = MD_PIS_CH7_ADC0_INSERT;
    pis_init.c_clk     = MD_PIS_CLK_PCLK2;

    md_pis_init(&pis_init);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    /* Configure system clock */
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HOSC_3, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();
    /* 2bits preempt-priority(0-3)
       2bits sub-priority(0-3). */
    NVIC_SetPriorityGrouping(5);

    /* Enable the selected ADC instance */
    md_adc_converter_enable_adcen(ADC0);
    /* Disable the selected ADC instance */
    md_adc_converter_disable_adcen(ADC0);

    /* Initialize PIS */
    init_pis();

    /* Initialize TIMER */
    init_timer();

    /* Pin Configuration */
    config_adc_pin();

    /* ADC Base Feature Configuration */
    config_adc_base();

    /*Enable the selected ADC instance */
    md_adc_converter_enable_adcen(ADC0);

    /* Start TIMER */
    md_timer_enable_counter_cnten(AD16C4T0);

    while (1);
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
