/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    30 Oct 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Oct 2021     biyq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"
#include "stdio.h"

/* Public Variables ---------------------------------------------------------- */
/* Private Macros ------------------------------------------------------------ */
/* Private Variables --------------------------------------------------------- */
static uint32_t s_crc_buf[8] = {1U, 2U, 3U, 4U, 5U, 6U, 7U, 8U};
static uint32_t s_crc_result = 0x0U;

/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */
void gpio_init(void);
void uart_init(void);

/* Private Function ---------------------------------------------------------- */
/**
  * @brief  Initializate system.
  * @retval None
  */
void SystemInit(void)
{

}

/**
  * @brief:  Main program.
  * @param:  None
  * @retval: None
  */
int main()
{
    /* Configure System Clock */
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HOSC_3, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();
    /* 2bits preempt-priority(0-3)
       2bits sub-priority(0-3). */
    NVIC_SetPriorityGrouping(5);

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();

    /* Init GPIO and UART */
    gpio_init();
    uart_init();

    /* Init CRC and output result after 7s */
    init_crc32_module();
    md_delay_1ms(1000);
    s_crc_result = get_crc32_calculation_result(s_crc_buf, (sizeof(s_crc_buf) / 4));
    printf("\r\nCRC32 calculation result for 0x01 ~ 0x08 is: 0x%.8X\n", s_crc_result);

    while (1)
    {
        md_gpio_toggle_pin_output(LED1_PORT, LED1_PIN);
        md_gpio_toggle_pin_output(LED2_PORT, LED2_PIN);
        md_delay_1ms(1000);
    }
}

/**
  * @brief  Init LED pin
  * @retval None
  */
void gpio_init(void)
{
    md_gpio_init_t gpio_config;

    /* Initialize led1 pin */
    gpio_config.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos  = MD_GPIO_PUSH_PULL;
    gpio_config.pupd  = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_config.type  = MD_GPIO_TYPE_CMOS;
    gpio_config.func  = MD_GPIO_FUNC_1;
    md_gpio_init(LED1_PORT, LED1_PIN, &gpio_config);
    md_gpio_init(LED2_PORT, LED2_PIN, &gpio_config);

    md_gpio_write_pin(LED1_PORT, LED1_PIN, 0);
    md_gpio_write_pin(LED2_PORT, LED2_PIN, 1);

    return;
}

/**
  * @brief  Init UART pin
  * @retval None
  */
void uart_pin_init(void)
{
    md_gpio_init_t gpio_config;
    md_gpio_init_struct(&gpio_config);

    /* Initialize tx pin */
    gpio_config.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos  = MD_GPIO_PUSH_PULL;
    gpio_config.pupd  = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_config.type  = MD_GPIO_TYPE_CMOS;
    gpio_config.func  = MD_GPIO_FUNC_3;
    md_gpio_init(GPIOB, MD_GPIO_PIN_10, &gpio_config);

    /* Initialize rx pin */
    gpio_config.mode  = MD_GPIO_MODE_INPUT;
    gpio_config.odos  = MD_GPIO_PUSH_PULL;
    gpio_config.pupd  = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.flt   = MD_GPIO_FILTER_DISABLE;
    gpio_config.type  = MD_GPIO_TYPE_CMOS;
    gpio_config.func  = MD_GPIO_FUNC_3;
    md_gpio_init(GPIOB, MD_GPIO_PIN_11, &gpio_config);

    return;
}

/**
  * @brief:  Initialize the uart.
  * @param:  None
  * @retval: None
  */
void uart_init(void)
{
    md_uart_init_t uart_handle;
    md_uart_init_struct(&uart_handle);

    uart_pin_init();

    /* Initialize UART0 */
    uart_handle.baud        = 115200;
    uart_handle.word_length = MD_UART_WORD_LENGTH_8B;
    uart_handle.stop_bits   = MD_UART_STOP_BITS_1;
    uart_handle.parity      = MD_UART_PARITY_NONE;
    uart_handle.fctl        = MD_UART_FLOW_CTL_DISABLE;
    uart_handle.mode        = MD_UART_MODE;
    md_uart_init(UART0, &uart_handle);

    md_uart_set_tx_threshold(UART0, MD_UART_TXFIFO_EMPTY);

    return;
}

/**
  * @brief:  Rewrite fputc.
  * @param:  None
  * @retval: None
  */
int fputc(int c, FILE *f)
{
    while ((READ_BIT(UART0->STAT, UART_STAT_TFEMPTY_MSK)) != UART_STAT_TFEMPTY_MSK);

    UART0->TXBUF = (uint8_t)c;

    return c;
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
