/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include <string.h>
#include "main.h"
#include "iap_rom.h"
#include "ram_vectors.h"

/** @addtogroup Projects_Examples_MD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initializate pin.
  * @retval None
  */
void init_gpio(void)
{
    md_gpio_init_t gpio_init;

    /* Initialize LED pin */
    md_gpio_init_struct(&gpio_init);      /* initialize the gpio_init */
    gpio_init.mode  = MD_GPIO_MODE_OUTPUT;
    gpio_init.odos  = MD_GPIO_PUSH_PULL;
    gpio_init.pupd  = MD_GPIO_FLOATING;
    gpio_init.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_init.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_init.func  = MD_GPIO_FUNC_1;
    md_gpio_init(LED_PORT, LED_PIN, &gpio_init);

    return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    uint32_t data_buf[32];
    uint32_t iap_flag;

    /* Configure system clock */
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HOSC_6, MD_CMU_PLL1_OUTPUT_72M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);
    /* Initialize SysTick Interrupt */
    md_init_1ms_tick();

    /* Enable ALL peripheral */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();
    SYSCFG_LOCK();
    /* 2bits preempt-priority(0-3)
       2bits sub-priority(0-3). */
    NVIC_SetPriorityGrouping(5);

    /* Initialize pin */
    init_gpio();
    md_gpio_set_pin_high(LED_PORT, LED_PIN);

    __disable_irq();
    SCB->VTOR = (uint32_t)g_vectors_list_ram;
    __enable_irq();

    do
    {
        iap_flag = FALSE;

        /* Simple word write testing */
        if (IAPROM_PAGE_ERASE(0x10000) == RESET) /* Erase page 64 */
            break;

        if (IAPROM_WORD_PROGRAM(0x10000, 0x55AA6996) == RESET) /* Write 0x55AA6996 to 0x10000 */
            break;

        memcpy((void *)data_buf, (void *)0x10000, 4);   /* read 0x10000 */

        if (data_buf[0] != 0x55AA6996)
            break;

        /* Double words write testing */
        if (IAPROM_PAGE_ERASE(0x10000) == RESET) /* Erase page 64 */
            break;

        /* Write 0x12345678 to 0x10000 and write 0x87654321 to 0x10004 */
        if (IAPROM_WORDS_PROGRAM(0x10000, 0x12345678, 0x87654321) == RESET) /* Write 0x12345678 to 0x10000 */
            break;

        memcpy((void *)data_buf, (void *)0x10000, 8);  /* read 0x10000 and 0x10004 */

        if ((data_buf[0] != 0x12345678) || (data_buf[1] != 0x87654321))
            break;

        /* Multiple words write testing */
        memset((void *)data_buf, 0x55, 32 * 4);

        /* Write data_buf to 0x10000 */
        if (IAPROM_MULTI_PROGRAM(0x10000, (uint32_t)data_buf, 32 * 4, AUTO_ERASE_TRUE) == RESET)
            break;

        if (memcmp((void *)0x10000, (void *)data_buf, 32 * 4) != 0)
            break;

        iap_flag = TRUE;
        break;
    }
    while (0);

    __disable_irq();
    SCB->VTOR = 0U;
    __enable_irq();

    while (1)
    {
        if (iap_flag == TRUE)                          /* If iap is succesful */
        {
            md_gpio_toggle_pin_output(LED_PORT, LED_PIN);    /* Toggle the led */
            md_delay_1ms(500);
        }
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
