/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/**
  * @brief  Initializate lcd pin.
  * @retval None
  */
void lcd_pin_init(void)
{
    md_gpio_init_t gpio_init;
    /* LCD      com1 com2 com3 com4 seg1  seg2  seg3  seg4  seg5  seg6  seg7  seg8
     * MCU_LCD      com4 com3 com2 com1 seg16 seg21 seg10 seg15 seg14 seg13 seg12 seg11
     */

    md_gpio_init_struct(&gpio_init);

    gpio_init.pupd = MD_GPIO_FLOATING;
    gpio_init.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_init.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_init.type = MD_GPIO_TYPE_CMOS;
    gpio_init.odos = MD_GPIO_PUSH_PULL;
    gpio_init.mode = MD_GPIO_MODE_CLOSE;
    gpio_init.func = MD_GPIO_FUNC_0;

    /* COM4 */
    md_gpio_init(GPIO_PORT_COM, GPIO_PIN_COM4, &gpio_init);
    /* COM3 */
    md_gpio_init(GPIO_PORT_COM, GPIO_PIN_COM3, &gpio_init);
    /* COM2 */
    md_gpio_init(GPIO_PORT_COM, GPIO_PIN_COM2, &gpio_init);
    /* COM1 */
    md_gpio_init(GPIO_PORT_COM, GPIO_PIN_COM1, &gpio_init);
    /* SEG1 */
    md_gpio_init(GPIO_PORT_SEG, GPIO_PIN_SEG1, &gpio_init);
    /* SEG2 */
    md_gpio_init(GPIO_PORT_SEG, GPIO_PIN_SEG2, &gpio_init);
    /* SEG3 */
    md_gpio_init(GPIO_PORT_SEG, GPIO_PIN_SEG3, &gpio_init);
    /* SEG4 */
    md_gpio_init(GPIO_PORT_SEG, GPIO_PIN_SEG4, &gpio_init);
    /* SEG5*/
    md_gpio_init(GPIO_PORT_SEG, GPIO_PIN_SEG5, &gpio_init);
    /* SEG6 */
    md_gpio_init(GPIO_PORT_SEG, GPIO_PIN_SEG6, &gpio_init);
    /* SEG7 */
    md_gpio_init(GPIO_PORT_SEG, GPIO_PIN_SEG7, &gpio_init);
    /* SEG8 */
    md_gpio_init(GPIO_PORT_SEG, GPIO_PIN_SEG8, &gpio_init);
}

/**
  * @brief  Initializate lcd.
  * @retval None
  */
void lcd_init(void)
{
    md_lcd_init_t init;

    SYSCFG_UNLOCK();
    md_cmu_set_lcd_clock_source(0x5);
    SYSCFG_LOCK();

    /* Initialize lcd */
    md_lcd_struct_init(&init);

    init.lcd_bias   = MD_LCD_BIAS_1_4;
    init.lcd_div    = MD_LCD_DIV_20;
    init.lcd_duty   = MD_LCD_DUTY_1_6;    /* COM0 ~ COM5 */
    init.lcd_prs    = MD_LCD_PRS_128;
    init.lcd_reshd  = MD_LCD_RES_3MOHM;
    init.lcd_resld  = MD_LCD_RES_3MOHM;
    init.lcd_vbufhd = DISABLE;
    init.lcd_vbufld = DISABLE;
    init.lcd_dshd   = 0x0;
    init.lcd_dsld   = 0x0;
    init.lcd_vchps  = MD_LCD_VCHPS_3V2;
    init.lcd_vsel   = MD_LCD_VSEL_VDD;
    init.lcd_wfs    = MD_LCD_WAVE_B;
    init.lcd_vgs    = MD_LCD_VGS_15;
    init.lcd_pon    = MD_LCD_PON_NONE;
    md_lcd_init(LCD, &init);

    /* seg16 seg21 seg10 seg15 seg14 seg13 seg12 seg11 */
//    md_lcd_set_segcr0_value(0x0021FC00);
//    md_lcd_set_segcr1_value(0x00000000);

    /* Enable lcd */
    md_lcd_enable_oe();
    md_lcd_enable_en();
}

/**
 * @brief 1~4 Digital Position COM And SEG MAP.
 */
const uint8_t dig8[16]  =
{
    0xED, 0x44, 0xD9, 0xD5, 0x74, 0xB5, 0xBD, 0xC4,
    0xFD, 0xF5, 0xFC, 0x3D, 0xA9, 0x5D, 0xB9, 0xB8
};

/**
 * @brief  LCD First Volume COM And SEG MAP.
 */
const uint32_t lcd_map[8] =
{
    (1 << 16), (1 << 21), (1 << 10), (1 << 15),
    (1 << 14), (1 << 13), (1 << 12), (1 << 11)
};

/**
  * @brief  LCD fill display buffer.
  * @param  flushbuf:buffer start address.
  * @param  idx: display digital index
  * @param  number: number 0x0 ~ 0x0F
  * @retval None.
  */
void lcd_display(uint32_t *flushbuf, uint8_t idx, uint8_t number)
{
    uint8_t i = 0U;
    uint8_t temp = 0U;
    uint8_t offset = 0x3U;

    if (idx > 4 || idx < 1) return;

    for (i = 0; i < 4; i++)
    {
        temp = (dig8[number] & offset) >> 2 * i;

        switch (temp)
        {
            case 1:
                flushbuf[i] |= lcd_map[idx * 2 - 1];
                break;

            case 2:
                flushbuf[i] |= lcd_map[idx * 2 - 2];
                break;

            case 3:
                flushbuf[i] |= lcd_map[idx * 2 - 1] | lcd_map[idx * 2 - 2];
                break;

            default:
                break;
        }

        offset <<= 2;
    }
}

/**
  * @brief  Reflush the used com_seg buffer.
  * @param  srcbuf:the source data start address.
  * @retval None.
  */
void lcd_reflush(uint32_t *flushbuf)
{
    uint8_t i = 0U;
    uint8_t tmp[4][2] = {{2U, 3U}, {4U, 5U}, {6U, 7U}, {8U, 9U}};

    for (i = 0; i < 4; ++i)
        md_lcd_set_seg_buf_data(tmp[i][0], flushbuf[i]);
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    uint32_t lcd_buf[8] = {0U};

    uint16_t i = 0U;
    uint8_t j = 0U;

    md_cmu_pll_config(MD_CMU_PLL_INPUT_HOSC_3, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);
    md_init_1ms_tick();

    lcd_pin_init();

    lcd_init();

    while (1)
    {
        for (i = 0; i < 10000; ++i)
        {
            if ((i % 10) == 0)
            {
                md_lcd_set_blmod(0x3);
                md_lcd_set_blfrq(0x3);
            }
            else
            {
                md_lcd_set_blmod(0);
            }

            lcd_display(lcd_buf, 1, i / 1000);
            lcd_display(lcd_buf, 2, i % 1000 / 100);
            lcd_display(lcd_buf, 3, i % 100 / 10);
            lcd_display(lcd_buf, 4, i % 10);
            lcd_reflush(lcd_buf);

            for (j = 0; j < 8; ++j)
                lcd_buf[j] = 0;

            md_delay_1ms(1000);
        }
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
