/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/** @addtogroup Projects_Examples_ALD
  * @{
  */

/** @addtogroup Examples
  * @{
  */

/**
  * @brief  Initializate lcd pin.
  * @retval None
  */
void lcd_pin_init(void)
{
    gpio_init_t x;
    /* LCD      com1 com2 com3 com4 seg1  seg2  seg3  seg4  seg5  seg6  seg7  seg8
     * MCU_LCD      com4 com3 com2 com1 seg16 seg21 seg10 seg15 seg14 seg13 seg12 seg11
     */

    memset(&x, 0x0, sizeof(gpio_init_t));

    x.mode = GPIO_MODE_CLOSE;
    x.odos = GPIO_PUSH_PULL;
    x.pupd = GPIO_FLOATING;
    x.podrv = GPIO_OUT_DRIVE_1;
    x.nodrv = GPIO_OUT_DRIVE_1;
    x.flt  = GPIO_FILTER_DISABLE;
    x.type = GPIO_TYPE_CMOS;
    x.func = GPIO_FUNC_0;

    /* MCU_COM4--->LCD_COM1 */
    ald_gpio_init(GPIO_PORT_COM, GPIO_PIN_COM4, &x);

    /* MCU_COM3--->LCD_COM2 */
    ald_gpio_init(GPIO_PORT_COM, GPIO_PIN_COM3, &x);

    /* MCU_COM2--->LCD_COM3 */
    ald_gpio_init(GPIO_PORT_COM, GPIO_PIN_COM2, &x);

    /* MCU_COM1--->LCD_COM4 */
    ald_gpio_init(GPIO_PORT_COM, GPIO_PIN_COM1, &x);

    /* SEG1 */
    ald_gpio_init(GPIO_PORT_SEG, GPIO_PIN_SEG1, &x);

    /* SEG2 */
    ald_gpio_init(GPIO_PORT_SEG, GPIO_PIN_SEG2, &x);

    /* SEG3 */
    ald_gpio_init(GPIO_PORT_SEG, GPIO_PIN_SEG3, &x);

    /* SEG4 */
    ald_gpio_init(GPIO_PORT_SEG, GPIO_PIN_SEG4, &x);

    /* SEG5*/
    ald_gpio_init(GPIO_PORT_SEG, GPIO_PIN_SEG5, &x);

    /* SEG6 */
    ald_gpio_init(GPIO_PORT_SEG, GPIO_PIN_SEG6, &x);

    /* SEG7 */
    ald_gpio_init(GPIO_PORT_SEG, GPIO_PIN_SEG7, &x);

    /* SEG8 */
    ald_gpio_init(GPIO_PORT_SEG, GPIO_PIN_SEG8, &x);
}

/**
  * @brief  Initializate lcd.
  * @retval None
  */
void lcd_init(lcd_handle_t *h_lcd)
{
    /* Initialize lcd */
    h_lcd->perh = LCD;
    h_lcd->init.lcd_bias   = LCD_BIAS_1_4;
    h_lcd->init.lcd_div    = LCD_DIV_20;
    h_lcd->init.lcd_duty   = LCD_DUTY_1_6;
    h_lcd->init.lcd_prs    = LCD_PRS_128;
    h_lcd->init.lcd_reshd  = LCD_RES_3MOHM;
    h_lcd->init.lcd_resld  = LCD_RES_3MOHM;
    h_lcd->init.lcd_vbufhd = LCD_FUNC_DISABLE;
    h_lcd->init.lcd_vbufld = LCD_FUNC_DISABLE;
    h_lcd->init.lcd_dshd   = 0x0;
    h_lcd->init.lcd_dsld   = 0x0;
    h_lcd->init.lcd_vchps  = LCD_VCHPS_3V2;
    h_lcd->init.lcd_vsel   = LCD_VSEL_VDD;
    h_lcd->init.lcd_wfs    = LCD_WAVE_B;
    h_lcd->init.lcd_vgs    = LCD_VGS_15;
    h_lcd->init.clock      = CMU_LCD_SEL_HOSC_1M;

    ald_lcd_init(h_lcd);

    /* seg16 seg21 seg10 seg15 seg14 seg13 seg12 seg11 */
//    ald_lcd_write_seg(h_lcd, SEG_0_TO_31, 0x0021FC00);
//    ald_lcd_write_seg(h_lcd, SEG_32_TO_59, 0x00000000);

    /* Enable lcd */
    ald_lcd_cmd(h_lcd, ENABLE);
}

/**
 * @brief 1~4 Digital Position COM And SEG MAP.
 */
const uint8_t  dig8[16]  =
{
    0xED, 0x44, 0xD9, 0xD5, 0x74, 0xB5, 0xBD, 0xC4,
    0xFD, 0xF5, 0xFC, 0x3D, 0xA9, 0x5D, 0xB9, 0xB8
} ;

/**
 * @brief  LCD First Volume COM And SEG MAP.
 */
const uint32_t lcd_map[8] =
{
    (1 << 16), (1 << 21), (1 << 10), (1 << 15),
    (1 << 14), (1 << 13), (1 << 12), (1 << 11)
} ;

/**
  * @brief  LCD fill display buffer.
  * @param  flushbuf:buffer start address.
  * @param  idx: display digital index
  * @param  number: number 0x0 ~ 0x0F
  * @retval None.
  */
void lcd_display(uint32_t *flushbuf, uint8_t idx, uint8_t number)
{
    uint8_t i = 0U;
    uint8_t temp = 0U;
    uint8_t offset = 0x3U;

    if (idx > 4 || idx < 1)
        return;

    for (i = 0; i < 4; i++)
    {
        temp = (dig8[number] & offset) >> 2 * i;

        switch (temp)
        {
            case 1:
                flushbuf[i] |= lcd_map[idx * 2 - 1];
                break;

            case 2:
                flushbuf[i] |= lcd_map[idx * 2 - 2];
                break;

            case 3:
                flushbuf[i] |= lcd_map[idx * 2 - 1] | lcd_map[idx * 2 - 2];
                break;

            default:
                break;
        }

        offset <<= 2;
    }
}

/**
  * @brief  Reflush the used com_seg buffer.
  * @param  srcbuf:the source data start address.
  * @retval None.
  */
void lcd_reflush(lcd_handle_t *hperh, uint32_t *flushbuf)
{
    uint8_t i = 0U;
    uint8_t tmp[4][2] = {{2U, 3U}, {4U, 5U}, {6U, 7U}, {8U, 9U}};

    for (i = 0; i < 4; ++i)
    {
        ald_lcd_write(hperh, tmp[i][0], flushbuf[i]);
    }
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main()
{
    lcd_handle_t h_lcd;

    uint32_t lcd_buf[8] = {0U};

    uint16_t i = 0U;
    uint8_t j = 0U;

    /* Initialize ALD */
    ald_cmu_init();
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_48M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 48000000);
    ald_cmu_losc_safe_config(ENABLE);
    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

    lcd_pin_init();

    lcd_init(&h_lcd);

    while (1)
    {
        for (i = 0; i < 10000; ++i)
        {
            if ((i % 10) == 0)
                ald_lcd_blink_config(&h_lcd, LCD_BLINK_ALLSEG_ALLCOM, LCD_BLFRQ_64);
            else
                ald_lcd_blink_config(&h_lcd, LCD_BLINK_OFF, LCD_BLFRQ_8);

            lcd_display(lcd_buf, 1, i / 1000);
            lcd_display(lcd_buf, 2, i % 1000 / 100);
            lcd_display(lcd_buf, 3, i % 100 / 10);
            lcd_display(lcd_buf, 4, i % 10);
            lcd_reflush(&h_lcd, lcd_buf);

            for (j = 0; j < 8; ++j)
                lcd_buf[j] = 0;

            ald_delay_ms(1000);
        }
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
