/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *          27 July 2022    Zhuxf           适配ES-PDS-ES32F3696LX开发板
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include <stdio.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

#define SLAVE_ADDR  0xA0    /*Definition from the machine address*/
#define MEM_ADDR    0x00    /*Define the machine from the initial memory address*/

/* Private Variables --------------------------------------------------------- */

static char s_eeprom_txbuf[16] = "eeprom example!";     /*Length must be no more than one page (16 bytes)*/
static char s_eeprom_rxbuf[16];

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */
void i2c_read_write(void)
{
    uint8_t i;
    uint16_t cnt = 4000U;

    md_i2c_set_tx_reg_data(I2C0, MEM_ADDR); /*Sent from the machine to be written to address*/

    for (i = 0; i < strlen(s_eeprom_txbuf) + 1; i++)
    {
        cnt = 4000;

        while (md_i2c_is_active_txf(I2C0) && (--cnt)) /*If send the FIFO is waiting*/
            ;

        md_i2c_set_tx_reg_data(I2C0, s_eeprom_txbuf[i]);  /*To write data to send FIFO*/
    }

    cnt = 4000;

    while (!md_i2c_is_active_tc(I2C0) && (--cnt))  /*The transfer to complete*/
        ;

    md_i2c_set_stop(I2C0);  /*Send the stop bit*/

    md_delay_1ms(10);  /*Inside block waiting for the eeprom write success*/

    md_i2c_set_transmit_fifo_reset(I2C0);   /*Reset send FIFO*/
    md_i2c_set_receive_fifo_reset(I2C0);    /*Reset receive FIFO*/

    md_i2c_enable_master_write(I2C0);   /*Host write data model*/
    md_i2c_set_transmit_length(I2C0, 1);    /*The only transport address of the block 1 eeprom chip*/
    md_i2c_set_slave_addr(I2C0, SLAVE_ADDR);  /*Set from the machine address*/
    md_i2c_set_start(I2C0);     /*+ 7 bus spare time to send a start bit bit from machine address + 1 write a bit*/

    md_i2c_set_tx_reg_data(I2C0, MEM_ADDR);

    cnt = 4000;

    while (!md_i2c_is_active_tc(I2C0) && (--cnt))  /*Waiting for completion of the inside address*/
        ;

    md_i2c_enable_master_read(I2C0);    /*Host read data model*/
    md_i2c_set_transmit_length(I2C0, strlen(s_eeprom_txbuf) + 1); /*The transmission of bytes*/
    md_i2c_set_start(I2C0);     /*restart*/

    for (i = 0; i < strlen(s_eeprom_txbuf) + 1; i++)
    {
        cnt = 4000;

        while (!md_i2c_is_active_rxth(I2C0) && (--cnt))  /*Waiting to receive complete*/
            ;

        s_eeprom_rxbuf[i] = md_i2c_get_rx_reg_data(I2C0);
    }

    cnt = 4000;

    while (!md_i2c_is_active_tc(I2C0) && (--cnt))  /*The transfer to complete*/
        ;

    md_i2c_set_stop(I2C0);  /*Send the stop bit*/
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    md_cmu_pll1_config(MD_CMU_PLL1_INPUT_HOSC_3, MD_CMU_PLL1_OUTPUT_72M);   /*Can make the frequency doubling, by the crystal three times the frequency of vibration to 48 MHZ*/
    md_cmu_clock_config(MD_CMU_CLOCK_PLL1, 72000000);       /*Choose the frequency doubling the clock for the system clock*/

    SYSCFG_UNLOCK();
    md_cmu_enable_perh_all();   /*Can make all the peripheral clock*/
    SYSCFG_LOCK();

    /* 2bits preempt-priority(0-3)
       2bits sub-priority(0-3). */
    NVIC_SetPriorityGrouping(5);

    md_init_1ms_tick();

    init_mcu_uart();    /*To initialize the UART module*/
    init_mcu_i2c();     /*Initialize the I2C module*/

    md_i2c_set_transmit_fifo_reset(I2C0);   /*Reset send FIFO*/

    md_i2c_enable_master_write(I2C0);   /*Host write data model*/
    md_i2c_set_transmit_length(I2C0, strlen(s_eeprom_txbuf) + 1 + 1); /*The transmission of bytes*/
    md_i2c_set_slave_addr(I2C0, SLAVE_ADDR);  /*Set from the machine address*/
    md_i2c_set_start(I2C0);     /*+ 7 bus spare time to send a start bit bit from machine address + 1 write a bit*/

    printf("\r\nThe date written to eeprom address %d is -> %s\r\n", MEM_ADDR, s_eeprom_txbuf);

    i2c_read_write();

    printf("The data read from eeprom address %d is  -> %s\r\n", MEM_ADDR, s_eeprom_rxbuf);

    if (!memcmp(s_eeprom_txbuf, s_eeprom_rxbuf, strlen(s_eeprom_txbuf) + 1))   /*Data is written and read*/
        printf("Read OK!\r\n");
    else
        printf("Read ERROR!\r\n");

    while (1)
        ;
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
