/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main file for DEMO
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *          27 July 2022    Zhuxf           适配ES-PDS-ES32F3696LX开发板
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

uint8_t g_recv_buf[20];
uint8_t g_recv_len = sizeof(g_recv_buf);
uint8_t g_send_buf[20];
uint8_t g_send_len = sizeof(g_send_buf);

uint8_t g_tx_flag = 0U;

i2c_handle_t g_ald_i2c;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/**
  * @brief  Initializate pin of i2c module.
  * @retval None
  */
void init_i2c_pin(void)
{
    gpio_init_t ald_gpio;
    memset(&ald_gpio, 0, sizeof(gpio_init_t));

    /* Initialize scl pin */
    ald_gpio.mode = GPIO_MODE_INPUT;
    ald_gpio.odos = GPIO_OPEN_DRAIN;
    ald_gpio.pupd = GPIO_PUSH_UP;
    ald_gpio.nodrv = GPIO_OUT_DRIVE_1;
    ald_gpio.flt  = GPIO_FILTER_DISABLE;
    ald_gpio.type = GPIO_TYPE_CMOS;
    ald_gpio.func = GPIO_FUNC_5;
    ald_gpio_init(I2C_SCL_PORT, I2C_SCL_PIN, &ald_gpio);

    /* Initialize sda pin */
    ald_gpio.mode = GPIO_MODE_OUTPUT;
    ald_gpio.odos = GPIO_OPEN_DRAIN;
    ald_gpio.pupd = GPIO_PUSH_UP;
    ald_gpio.nodrv = GPIO_OUT_DRIVE_1;
    ald_gpio.flt  = GPIO_FILTER_DISABLE;
    ald_gpio.type = GPIO_TYPE_CMOS;
    ald_gpio.func = GPIO_FUNC_5;
    ald_gpio_init(I2C_SDA_PORT, I2C_SDA_PIN, &ald_gpio);

    return;
}

/**
  * @brief  Send message complete.
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None.
  */
void i2c_send_complete(i2c_handle_t *arg)
{
    g_tx_flag = 0;

    return;
}

/**
  * @brief  Receive a message complete.
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None.
  */
void i2c_recv_complete(i2c_handle_t *arg)
{
    g_tx_flag = 1;

    return;
}

/**
  * @brief  Occurs error.
  * @param  arg: Pointer to uart_handle_t structure.
  * @retval None.
  */
void uart_error(uart_handle_t *arg)
{
    return;
}

/**
  * @brief  Test main function
  * @retval Status.
  */
int main(void)
{
    uint16_t cnt;

    /* Initialize ALD */
    ald_cmu_init();

    /* Configure system clock */
    ald_cmu_pll1_config(CMU_PLL1_INPUT_HOSC_3, CMU_PLL1_OUTPUT_72M);
    ald_cmu_clock_config(CMU_CLOCK_PLL1, 72000000);

    ald_cmu_perh_clock_config(CMU_PERH_ALL, ENABLE);

    /* Enable I2c interrupt */
    ald_mcu_irq_config(DMA_IRQn, 1, 1, ENABLE);

    /* Initialize i2c pin */
    init_i2c_pin();

    /* clear i2c_handle_t structure */
    memset(&g_ald_i2c, 0, sizeof(i2c_handle_t));
    /* Initialize i2c */
    g_ald_i2c.perh = I2C1;
    g_ald_i2c.init.module       = I2C_MODULE_SLAVE;
    g_ald_i2c.init.addr_mode    = I2C_ADDR_7BIT;
    g_ald_i2c.init.clk_speed    = 400000;
    g_ald_i2c.init.dual_addr    = I2C_DUALADDR_ENABLE;
    g_ald_i2c.init.general_call = I2C_GENERALCALL_DISABLE;
    g_ald_i2c.init.no_stretch   = I2C_NOSTRETCH_DISABLE;
    g_ald_i2c.init.own_addr1    = 0x2D << 1;
    g_ald_i2c.slave_tx_cplt_cbk = i2c_send_complete;
    g_ald_i2c.slave_rx_cplt_cbk = i2c_recv_complete;
    ald_i2c_init(&g_ald_i2c);

    MODIFY_REG(I2C1->FCON, I2C_FCON_RXFTH_MSK, (0 << I2C_FCON_RXFTH_POSS));
    MODIFY_REG(I2C1->FCON, I2C_FCON_TXFTH_MSK, (0 << I2C_FCON_TXFTH_POSS));

    while (1)
    {
        if (!g_tx_flag)
            ald_i2c_slave_recv_by_dma(&g_ald_i2c, g_recv_buf, g_recv_len, 1);
        else
            ald_i2c_slave_send_by_dma(&g_ald_i2c, g_recv_buf, g_recv_len, 0);

        cnt = 40000;

        while (I2C_GET_FLAG(&g_ald_i2c, I2C_STAT_BUSY) && (--cnt));
    }
}

/**
  * @}
  */
/**
  * @}
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
