/**********************************************************************************
 *
 * @file    i2c.c
 * @brief   i2c C file
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Includes ------------------------------------------------------------------ */
#include "main.h"
#include "i2c.h"

/* Public Variables ---------------------------------------------------------- */


/* Private Function ---------------------------------------------------------- */
/*Host mode communication pins initialization function*/
static void init_i2c_pin(void)
{
    md_gpio_init_t gpio_config;

    /*The host SCL pins initialization*/
    gpio_config.mode = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos = MD_GPIO_OPEN_DRAIN;    /*Open drain output*/
    gpio_config.pupd = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_config.type = MD_GPIO_TYPE_CMOS;
    gpio_config.func = MD_GPIO_FUNC_5;
    md_gpio_init(I2C1_SCL_PORT, I2C1_SCL_PIN, &gpio_config);

    /*Initialize the host SDA pin*/
    gpio_config.mode = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos = MD_GPIO_OPEN_DRAIN;    /*Open drain output*/
    gpio_config.pupd = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_config.type = MD_GPIO_TYPE_CMOS;
    gpio_config.func = MD_GPIO_FUNC_5;
    md_gpio_init(I2C1_SDA_PORT, I2C1_SDA_PIN, &gpio_config);

    return;
}

/*I2C initialization function*/
void initialize_i2c_master(void)
{
    md_i2c_init_t i2c_config;

    init_i2c_pin();
    /*Will the I2C initialization structure is set to the default values*/
    md_i2c_struct_init(&i2c_config);

    /*User configuration I2C*/
    i2c_config.addr_mode = MD_I2C_ADDR_7BIT;
    i2c_config.clk_speed = 100000;
    i2c_config.dual_addr = MD_I2C_DUALADDR_DISABLE;
    i2c_config.duty = MD_I2C_DUTYCYCLE_16_9;
    i2c_config.general_call = MD_I2C_GENERALCALL_DISABLE;
    i2c_config.no_stretch = MD_I2C_NOSTRETCH_ENABLE;
    i2c_config.own_addr1 = 0x5A;

    /*Initialize the I2C*/
    md_i2c_init(I2C1, &i2c_config);

    /*Make can receive complete automatic reply*/
    md_i2c_enable_ack(I2C1);
    
    return;
}

uint32_t read_i2c_device(eeprom_read_write_handler *i2c_device)
{
    uint32_t i;
    uint32_t xfer_cnt = 0;
    uint16_t cnt = 60000U;

    /* generate start bit */
    md_i2c_generate_start_condition(I2C1);
    while (!md_i2c_is_active_flag_sb(I2C1) && cnt--);
    for (i = 0; i < 60000; i++)
    {
        if (md_i2c_is_active_flag_addr(I2C1))
        {
            md_i2c_is_active_flag_msl(I2C1);
            break;
        }
    }
    /* send slave address */
    md_i2c_transmit_data8(I2C1, (i2c_device->slave_addr) | I2C_REQUEST_WRITE);
    for (i = 0; i < 60000; i++)
    {
        if (md_i2c_is_active_flag_addr(I2C1))
        {
            md_i2c_is_active_flag_msl(I2C1);
            break;
        }
    }
    if (i >= 60000)
        return 0;
    
    /* send mem address */
    if (md_i2c_is_active_flag_txbe(I2C1))
    {
        md_i2c_transmit_data8(I2C1, i2c_device->mem_addr);
    }
    cnt = 60000U;
    while (!md_i2c_is_active_flag_txbe(I2C1) && cnt--);
    /* generate start bit again */
    md_i2c_generate_start_condition(I2C1);
    cnt = 60000U;
    while (!md_i2c_is_active_flag_sb(I2C1) && cnt--);
    for (i = 0; i < 60000; i++)
    {
        if (md_i2c_is_active_flag_addr(I2C1))
        {
            md_i2c_is_active_flag_msl(I2C1);
            break;
        }
    }
    /* send slave address */     
    md_i2c_transmit_data8(I2C1, (i2c_device->slave_addr) | I2C_REQUEST_READ);
    if (i2c_device->r_size < 2)
    {
        md_i2c_disable_ack(I2C1);
        md_i2c_generate_stop_condition_by_master(I2C1);
    }
    else if (i2c_device->r_size == 2)
    {
        md_i2c_disable_ack(I2C1);
        md_i2c_disable_bit_pos(I2C1);
    }
    else
    {
        md_i2c_enable_ack(I2C1);
    }
    /* clear addr flag */
    for (i = 0; i < 60000; i++)
    {
        if (md_i2c_is_active_flag_addr(I2C1))
        {
            md_i2c_is_active_flag_msl(I2C1);
            break;
        }
    }
    if (i >= 60000)
        return 0;
     cnt = 60000U;
    while (!md_i2c_is_active_flag_btf(I2C1) && cnt--);
    if (i2c_device->r_size > 1)
    {
        /* read data */
        for (i = 0; i < i2c_device->r_size; i++)
        {
            /* the last but one */
            if (i == i2c_device->r_size - 2)
            {
                md_i2c_disable_ack(I2C1);
                md_i2c_generate_stop_condition_by_master(I2C1);
            }
            cnt = 60000U;
            while (md_i2c_is_active_flag_rxne(I2C1) && cnt--);
            i2c_device->r_buf[xfer_cnt++] = md_i2c_receive_data8(I2C1);
        }
    }
    else
    {
        cnt = 60000U;
        while (md_i2c_is_active_flag_rxne(I2C1) && cnt--);
        i2c_device->r_buf[xfer_cnt++] = md_i2c_receive_data8(I2C1);
        md_i2c_generate_stop_condition_by_master(I2C1);
    }
    
    return xfer_cnt;
}
 
uint32_t write_i2c_device(eeprom_read_write_handler *i2c_device)
{
    uint32_t i;
    uint32_t xfer_cnt = 0;
    uint16_t cnt = 8000U;

    /* generate start bit */
    md_i2c_generate_start_condition(I2C1);
    while (!md_i2c_is_active_flag_sb(I2C1) && cnt--);
    for (i = 0; i < 60000; i++)
    {
        if (md_i2c_is_active_flag_addr(I2C1))
        {
            md_i2c_is_active_flag_msl(I2C1);
            break;
        }
    }
    /* send slave address */
    md_i2c_transmit_data8(I2C1, (i2c_device->slave_addr) | I2C_REQUEST_WRITE);
    for (i = 0; i < 8000; i++)
    {
        if (md_i2c_is_active_flag_addr(I2C1))
        {
            md_i2c_is_active_flag_msl(I2C1);
            break;
        }
    }
    if (i >= 8000)
        return 0;
    
    /* send mem address */
    md_i2c_transmit_data8(I2C1, i2c_device->mem_addr);
    cnt = 8000U;
    while (!md_i2c_is_active_flag_txbe(I2C1) && cnt--);
        
    /* write data */
    for (i = 0; i < i2c_device->w_size; i++)
    {
        md_i2c_transmit_data8(I2C1, i2c_device->w_buf[xfer_cnt++]);
        cnt = 8000;
        while (!md_i2c_is_active_flag_txbe(I2C1) && cnt--);
    }
    
    md_i2c_generate_stop_condition_by_master(I2C1);
    while (md_i2c_is_active_flag_busy(I2C1) && cnt--);
    
    return xfer_cnt;
}

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */


/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
