/**********************************************************************************
 *
 * @file    main.c
 * @brief   main C file
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"


/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */
static md_dma_descriptor_t s_dma0_ctrl_base[2]__attribute__((aligned(512)));

/* Public Variables ---------------------------------------------------------- */
md_dma_config_t g_dma_tx_cfg;
md_dma_config_t g_dma_rx_cfg;

uint8_t  g_recv_buf[20] = {0};
uint32_t g_recv_len = sizeof(g_recv_buf); 
uint8_t  g_send_buf[20] = {0};
uint32_t g_send_len = sizeof(g_send_buf);

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/*From machine mode communication pins initialization function*/
void init_i2c_pin(void)
{
    md_gpio_init_t a;

    /*From the machine initialization SCL pins*/
    a.mode = MD_GPIO_MODE_OUTPUT;
    a.odos = MD_GPIO_OPEN_DRAIN;    /*Open drain output*/
    a.pupd = MD_GPIO_PUSH_UP;
    a.podrv = MD_GPIO_OUT_DRIVE_6;
    a.nodrv = MD_GPIO_OUT_DRIVE_6;
    a.flt  = MD_GPIO_FILTER_DISABLE;
    a.type = MD_GPIO_TYPE_CMOS;
    a.func = MD_GPIO_FUNC_5;
    md_gpio_init(I2C1_SCL_PORT, I2C1_SCL_PIN, &a);

    /*From the machine initialization SDA pin*/
    a.mode = MD_GPIO_MODE_OUTPUT;
    a.odos = MD_GPIO_OPEN_DRAIN;    /*Open drain output*/
    a.pupd = MD_GPIO_PUSH_UP;
    a.podrv = MD_GPIO_OUT_DRIVE_6;
    a.nodrv = MD_GPIO_OUT_DRIVE_6;
    a.flt  = MD_GPIO_FILTER_DISABLE;
    a.type = MD_GPIO_TYPE_CMOS;
    a.func = MD_GPIO_FUNC_5;
    md_gpio_init(I2C1_SDA_PORT, I2C1_SDA_PIN, &a);

    return;
}

/*I2C initialization function*/
void initialize_i2c_slave(void)
{
    md_i2c_init_t i2c_config;
    /*Will the I2C initialization structure is set to the default values*/
    md_i2c_struct_init(&i2c_config);

    /*User configuration I2C*/
    i2c_config.addr_mode = MD_I2C_ADDR_7BIT;
    i2c_config.clk_speed = 400000;
    i2c_config.dual_addr = MD_I2C_DUALADDR_DISABLE;
    i2c_config.duty = MD_I2C_DUTYCYCLE_16_9;
    i2c_config.general_call = MD_I2C_GENERALCALL_DISABLE;
    i2c_config.no_stretch = MD_I2C_NOSTRETCH_DISABLE;
    i2c_config.own_addr1 = 0x5A;

    /*Initialize the I2C*/
    md_i2c_init(I2C1, &i2c_config);

    /*Make can receive complete automatic reply*/
    md_i2c_enable_ack(I2C1);

    md_i2c_enable_it_evt(I2C1);

    return;
}

/*Send the DMA function*/
void config_dma_tx(void)
{
    memset(&g_dma_tx_cfg, 0x0, sizeof(g_dma_tx_cfg));

    /*To enable the DMA*/
    md_dma_enable();

    /*User configuration DMA*/
    md_dma_config_struct(&g_dma_tx_cfg);
    g_dma_tx_cfg.data_width = MD_DMA_DATA_SIZE_BYTE;
    g_dma_tx_cfg.src_inc    = MD_DMA_DATA_INC_BYTE;       /*The source data address byte increments*/
    g_dma_tx_cfg.dst_inc    = MD_DMA_DATA_INC_NONE;       /*No address incremental target data*/
    g_dma_tx_cfg.R_power    = MD_DMA_R_POWER_1;
    g_dma_tx_cfg.primary    = ENABLE;
    g_dma_tx_cfg.burst      = DISABLE;
    g_dma_tx_cfg.high_prio  = DISABLE;
    g_dma_tx_cfg.interrupt  = ENABLE;
    g_dma_tx_cfg.channel      = 0;
    g_dma_tx_cfg.src          = (void *)(g_send_buf);       /*The source data*/
    g_dma_tx_cfg.dst          = (void *)(&I2C1->DATA);        /*The target data*/
    g_dma_tx_cfg.size         = g_send_len;       /*The length of the data*/
    g_dma_tx_cfg.msel         = MD_DMA_MSEL_I2C1;
    g_dma_tx_cfg.msigsel      = MD_DMA_MSIGSEL_I2C_TXEMPTY;       /*I2C1 send*/

    return;
}

/*The DMA receive function*/
void config_dma_rx(void)
{
    memset(&g_dma_rx_cfg, 0x0, sizeof(g_dma_rx_cfg));

    /*To enable the DMA*/
    md_dma_enable();

    /*User configuration DMA*/
    md_dma_config_struct(&g_dma_rx_cfg);
    g_dma_rx_cfg.data_width = MD_DMA_DATA_SIZE_BYTE;
    g_dma_rx_cfg.src_inc    = MD_DMA_DATA_INC_NONE;       /*Source data is no address increment*/
    g_dma_rx_cfg.dst_inc    = MD_DMA_DATA_INC_BYTE;       /*The target data address byte increments*/
    g_dma_rx_cfg.R_power    = MD_DMA_R_POWER_1;
    g_dma_rx_cfg.primary    = ENABLE;
    g_dma_rx_cfg.burst      = DISABLE;
    g_dma_rx_cfg.high_prio  = DISABLE;
    g_dma_rx_cfg.interrupt  = ENABLE;
    g_dma_rx_cfg.channel      = 1;
    g_dma_rx_cfg.src          = (void *)(&I2C1->DATA);        /*The source data*/
    g_dma_rx_cfg.dst          = (void *)(g_recv_buf);       /*The target data*/
    g_dma_rx_cfg.size         = g_recv_len;       /*The length of the data*/
    g_dma_rx_cfg.msel         = MD_DMA_MSEL_I2C1;
    g_dma_rx_cfg.msigsel      = MD_DMA_MSIGSEL_I2C_RNR;       /*I2C1 receive*/

    return;
}

/*The main function*/
int main(void)
{
    /*Configure the system clock*/
    md_cmu_clock_config(MD_CMU_CLOCK_HRC, 24000000);

    /*Initialize the SysTick interrupt*/
    md_init_1ms_tick();

    /*I2C clock can make*/
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_i2c1();
    SYSCFG_LOCK();

    /*Interrupt priority level*/
    md_mcu_irq_config(I2C1_IRQn, 0, ENABLE);

    /*Pin initialization of communication*/
    init_i2c_pin();

    /*I2C initialization*/
    initialize_i2c_slave();

    md_mcu_irq_config(DMA_IRQn, 2, ENABLE);
    memset(s_dma0_ctrl_base, 0x0, sizeof(s_dma0_ctrl_base));
    md_dma_set_ctrlbase((uint32_t)&s_dma0_ctrl_base);
    config_dma_rx();
    config_dma_tx();

    while (1)
    {
        /*  */
        md_delay_1ms(1000);
    }

}

/**
  * @} Examples_I2C
  */

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
