/**********************************************************************************
 *
 * @file    main.c
 * @brief   Main source file for master transmission.
 *
 * @date    30 Apri 2021
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          30 Apri 2021    yanght          the first version
 *          3 Sep. 2021     Lisq            I2C引脚改为TTL模式
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include <string.h>
#include "main.h"

void SystemInit()
{}
/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */
static md_dma_descriptor_t s_dma0_ctrl_base[2]__attribute__((aligned(512)));

/* Public Variables ---------------------------------------------------------- */
md_dma_config_t g_dma_tx_config, g_dma_rx_config;

uint8_t  g_recv_complete;
uint8_t  g_recv_buf[20];
uint32_t g_recv_len = sizeof(g_recv_buf);
                           
uint8_t  g_send_complete;
uint8_t  g_send_buf[20];
uint32_t g_send_len = sizeof(g_send_buf);

uint8_t  g_read_or_write;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/*Host mode communication pins initialization function*/
void init_i2c_pin(void)
{
    md_gpio_init_t gpio_config;

    /*The host SCL pins initialization*/
    gpio_config.mode = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos = MD_GPIO_OPEN_DRAIN;    /*Open drain output*/
    gpio_config.pupd = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_config.type = MD_GPIO_TYPE_CMOS;
    gpio_config.func = MD_GPIO_FUNC_5;
    md_gpio_init(I2C1_SCL_PORT, I2C1_SCL_PIN, &gpio_config);

    /*Initialize the host SDA pin*/
    gpio_config.mode = MD_GPIO_MODE_OUTPUT;
    gpio_config.odos = MD_GPIO_OPEN_DRAIN;    /*Open drain output*/
    gpio_config.pupd = MD_GPIO_PUSH_UP;
    gpio_config.podrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.nodrv = MD_GPIO_OUT_DRIVE_6;
    gpio_config.flt  = MD_GPIO_FILTER_DISABLE;
    gpio_config.type = MD_GPIO_TYPE_CMOS;
    gpio_config.func = MD_GPIO_FUNC_5;
    md_gpio_init(I2C1_SDA_PORT, I2C1_SDA_PIN, &gpio_config);

    return;
}

/*I2C initialization function*/
void initialize_i2c_master(void)
{
    md_i2c_init_t i2c_config;
    /*Will the I2C initialization structure is set to the default values*/
    md_i2c_struct_init(&i2c_config);

    /*User configuration I2C*/
    i2c_config.addr_mode = MD_I2C_ADDR_7BIT;
    i2c_config.clk_speed = 100000;
    i2c_config.dual_addr = MD_I2C_DUALADDR_DISABLE;
    i2c_config.duty = MD_I2C_DUTYCYCLE_16_9;
    i2c_config.general_call = MD_I2C_GENERALCALL_DISABLE;
    i2c_config.no_stretch = MD_I2C_NOSTRETCH_DISABLE;
    i2c_config.own_addr1 = 0x5A;

    /*Initialize the I2C*/
    md_i2c_init(I2C1, &i2c_config);

    /*Make can receive complete automatic reply*/
    md_i2c_enable_ack(I2C1);

    /*Interrupt priority level*/
    NVIC_SetPriority(I2C1_IRQn, 0);
    NVIC_EnableIRQ(I2C1_IRQn);
    md_i2c_enable_it_evt(I2C1);
    return;
}

/*Send the DMA function*/
void config_dma_tx(void)
{
    memset(&g_dma_tx_config, 0x0, sizeof(g_dma_tx_config));

    /*To enable the DMA*/
    md_dma_enable();

    /*User configuration DMA*/
    md_dma_config_struct(&g_dma_tx_config);
    g_dma_tx_config.data_width = MD_DMA_DATA_SIZE_BYTE;
    g_dma_tx_config.src_inc    = MD_DMA_DATA_INC_BYTE;       /*The source data address byte increments*/
    g_dma_tx_config.dst_inc    = MD_DMA_DATA_INC_NONE;       /*No address incremental target data*/
    g_dma_tx_config.R_power    = MD_DMA_R_POWER_1;
    g_dma_tx_config.primary    = ENABLE;
    g_dma_tx_config.burst      = DISABLE;
    g_dma_tx_config.high_prio  = DISABLE;
    g_dma_tx_config.interrupt  = ENABLE;
    g_dma_tx_config.channel      = 0;
    g_dma_tx_config.src          = (void *)(g_send_buf);        /*The source data*/
    g_dma_tx_config.dst          = (void *)(&I2C1->DATA);       /*The target data*/
    g_dma_tx_config.size         = g_send_len;                  /*The length of the data*/
    g_dma_tx_config.msel         = MD_DMA_MSEL_I2C1;
    g_dma_tx_config.msigsel      = MD_DMA_MSIGSEL_I2C_TXEMPTY;  /*I2C1 send*/

    return;
}

/*The DMA receive function*/
void config_dma_rx(void)
{
    memset(&g_dma_rx_config, 0x0, sizeof(g_dma_rx_config));

    /*To enable the DMA*/
    md_dma_enable();

    /*User configuration DMA*/
    md_dma_config_struct(&g_dma_rx_config);
    g_dma_rx_config.data_width = MD_DMA_DATA_SIZE_BYTE;
    g_dma_rx_config.src_inc    = MD_DMA_DATA_INC_NONE;       /*Source data is no address increment*/
    g_dma_rx_config.dst_inc    = MD_DMA_DATA_INC_BYTE;       /*The target data address byte increments*/
    g_dma_rx_config.R_power    = MD_DMA_R_POWER_1;
    g_dma_rx_config.primary    = ENABLE;
    g_dma_rx_config.burst      = DISABLE;
    g_dma_rx_config.high_prio  = DISABLE;
    g_dma_rx_config.interrupt  = ENABLE;
    g_dma_rx_config.channel      = 1;
    g_dma_rx_config.src          = (void *)(&I2C1->DATA);        /*The source data*/
    g_dma_rx_config.dst          = (void *)(g_recv_buf);       /*The target data*/
    g_dma_rx_config.size         = g_recv_len - 1;     /*The length of the data*/
    g_dma_rx_config.msel         = MD_DMA_MSEL_I2C1;
    g_dma_rx_config.msigsel      = MD_DMA_MSIGSEL_I2C_RNR;       /*I2C1 receive*/

    return;
}

/*The main function*/
int main(void)
{
    uint16_t cnt;
    /* Configure the system clock */
    md_cmu_pll_config(MD_CMU_PLL_INPUT_HRC_6, MD_CMU_PLL_OUTPUT_48M);
    md_cmu_clock_config(MD_CMU_CLOCK_PLL, 48000000);

    /* Initialize the SysTick interrupt */
    md_init_1ms_tick();

    /* Enable I2C clock */
    SYSCFG_UNLOCK();
    md_cmu_enable_perh_i2c1();
    SYSCFG_LOCK();

    /* Pin initialization of communication*/
    init_i2c_pin();

    /* I2C initialization*/
    initialize_i2c_master();

    /* I2C enable */
    MD_I2C_ENABLE(I2C1);

    /* I2C DMA enable */
    md_i2c_enable_dma_req_tx(I2C1);

    /* config DMA */
    memset(s_dma0_ctrl_base, 0x0, sizeof(s_dma0_ctrl_base));
    md_dma_set_ctrlbase((uint32_t)&s_dma0_ctrl_base);
    NVIC_SetPriority(DMA_IRQn, 0);
    NVIC_EnableIRQ(DMA_IRQn);
    config_dma_rx();
    config_dma_tx();

    while (1)
    {
        /*The DMA to receive*/
        g_recv_complete = 0;
        g_read_or_write = I2C_REQUEST_READ;
        /*Generate the start bit, automatically switch to host mode*/
        md_i2c_generate_start_condition(I2C1);
        cnt = 60000;
        while (g_recv_complete != 1 && cnt--);
        cnt = 60000;
        while (!md_i2c_is_active_flag_rxne(I2C1) && cnt--);
        g_recv_buf[g_recv_len - 1] = md_i2c_receive_data8(I2C1);
        md_delay_1ms(3000);
        
        memcpy(g_send_buf, g_recv_buf, sizeof(g_send_buf));
        
        /*The DMA is sent*/
        g_send_complete = 0;
        g_read_or_write = I2C_REQUEST_WRITE;
        /*Generate the start bit, automatically switch to host mode*/
        md_i2c_generate_start_condition(I2C1);
        cnt = 60000;
        while (g_send_complete != 1 && cnt--);
        cnt = 60000;
        while (!md_i2c_is_active_flag_txbe(I2C1) && cnt--);
        md_i2c_generate_stop_condition_by_master(I2C1);
        md_delay_1ms(3000);
    }
}

/**
  * @} Examples_I2C
  */

/**
  * @} Projects_Examples_MD
  */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
